
;; Model for simulate mosquitoes infected with bacterial symbiont Wolbachia

turtles-own
  [ wolb?        ;; if true, the turtle has Wolbachia
    wolb-carrying-count   ;; how long the turtle has Wolbachia
    age          ;; how many weeks old the turtle is
    recover-prob ;; not really used (set to zero)
    x-pos
    y-pos
    ]

globals
[
  %infected            ;; percentage of the population has Wolbachia
  carrying-capacity    ;; the number of turtles that can be in the world at one time
  initialInfected
  avgWolb
  avgNorm
  lifespan-with-Wolbachia
  ndays
  nRelCur
;;  ndays-week
]

to setup
  clear-all
  setup-constants
  setup-turtles
  update-global-variables
  reset-ticks
end

;; Create a variable number of mosquitoes (nMosquitoes) 

to setup-turtles
  set-default-shape turtles "butterfly"
  crt nMosquitoes
    [ 
      set x-pos random-xcor 
      set y-pos random-xcor
      setxy x-pos y-pos
      set age random normal-lifespan
      set wolb-carrying-count 0
      set recover-prob  0
      set size 0.5  ;; easier to see
      get-free ]
    
;; Create a variable number of mosquitoes with wolbachia (nWolbMosqutioes) that are release weekly  
  
  ask n-of initialInfected turtles
    [ set age random lifespan-with-Wolbachia
;;      setxy ((random world-height/4)- world-height/8) ((random world-width/4)- world-width/8)
      set x-pos random-xcor 
      set y-pos random-xcor
      setxy (x-pos / 4.0) (y-pos / 4.0)
;;      setxy x-pos y-pos
      get-wolb ]
end

to get-wolb ;; turtle procedure
  set wolb? true
  set color red
end

to get-free ;; turtle procedure
  set wolb? false
  set wolb-carrying-count 0
  set color blue
end

to setup-constants
  set carrying-capacity 4000
  set nRelCur 1
  set initialInfected nWolbMosquitoes 
;;  set 1days per week 7
  set avgNorm rateOffspring * normal-lifespan
  set lifespan-with-Wolbachia fracLifespanWolbachia * normal-lifespan
  set avgWolb (fracRateOffspWolbachia * rateOffspring) * lifespan-with-Wolbachia
  set ndays 1 
end

to go
  get-older
  move
  reproduce-wolbachia
  release 
  update-global-variables 
  tick
end


to release
  if (ndays mod 7)= 0 and (nRelCur < nReleasesTotal)
  [ 
    crt nWolbMosquitoes [
       set age random lifespan-with-Wolbachia
;;      setxy ((random world-height/4)- world-height/8) ((random world-width/4)- world-width/8)
      set x-pos random-xcor 
      set y-pos random-xcor
;      setxy x-pos y-pos
      setxy (x-pos / 4.0) (y-pos / 4.0)
      get-wolb      
    ]
    set nRelCur (nRelCur + 1)
  ] 
end 

to update-global-variables
  if count turtles > 0
  [
    set %infected (count turtles with [wolb?]) / (count turtles) * 100 
  ]
  set ndays (ndays + 1)
end

;;Turtle counting variables are advanced.
to get-older
  ask turtles
  [
    set age age + 1
    if wolb?
      [ set wolb-carrying-count (wolb-carrying-count + 1) ]
    ;; Turtles die of old age once their age equals the
    ;; lifespan (set at 1500 in this model).
    ifelse wolb?
    [ if age > lifespan-with-Wolbachia
      [ die ]
    ]
    [
     if age > normal-lifespan
     [ die ]
    ]
  ]
end

;;Turtles move about at random.
to move
  ask turtles
  [ rt random 100
    lt random 100
    fd 1 ]
end


to reproduce-wolbachia
;;  ask turtles with [not sick?]
ask turtles
    [ if (count turtles) < carrying-capacity
       [ ifelse wolb? 
         ;; for sick the average offspring is set for a normal lifespan
         ;;          change to hatch a vqariable number b
         [ if (random lifespan-with-Wolbachia) < avgWolb
           [ hatch 1
             [ set age 1
               lt 45 fd 1
               get-wolb ] ] 
         ]
         [ if (random normal-lifespan) < avgNorm and (random 100) > %infected 
           [ hatch 1
             [set age 1
               lt 45 fd 1
               get-free]
             ]           
           
         ]
       ]
     ]    
end


; Copyright 2013 Daniel Villela & Gabriela de Azambuja Garcia.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
262
10
805
574
20
20
13.0
1
10
1
1
1
0
1
1
1
-20
20
-20
20
1
1
1
ticks
30.0

BUTTON
20
14
109
49
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
115
14
240
50
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
6
409
258
573
Populations
days
mosquitoes
0.0
52.0
0.0
5000.0
true
true
"" ""
PENS
"Wolbachia" 1.0 0 -2674135 true "" "plot count turtles with [wolb?]"
"non-Wolbachia" 1.0 0 -13345367 true "" "plot count turtles with [not wolb?]"
"total" 1.0 0 -16777216 true "" "plot count turtles"

SLIDER
16
213
242
246
nMosquitoes
nMosquitoes
10
3000
1852
1
1
NIL
HORIZONTAL

MONITOR
6
360
81
405
NIL
%infected
1
1
11

MONITOR
185
357
259
402
years
ticks / 365
1
1
11

SLIDER
18
138
241
171
normal-lifespan
normal-lifespan
0
40
26
1
1
NIL
HORIZONTAL

SLIDER
16
175
242
208
fracLifespanWolbachia
fracLifespanWolbachia
0
1
0.95
.01
1
NIL
HORIZONTAL

SLIDER
17
99
241
132
rateOffspring
rateOffspring
1
10
8
1
1
NIL
HORIZONTAL

SLIDER
18
60
241
93
fracRateOffspWolbachia
fracRateOffspWolbachia
0
1
0.66
.01
1
NIL
HORIZONTAL

SLIDER
16
250
241
283
nWolbMosquitoes
nWolbMosquitoes
0
1000
67
1
1
NIL
HORIZONTAL

SLIDER
16
304
241
337
nReleasesTotal
nReleasesTotal
12
52
12
1
1
NIL
HORIZONTAL

MONITOR
90
359
178
404
# Releases
nRelCur
17
1
11

@#$#@#$#@
## WHAT IS IT?

This model simulates the releasing of mosquitoes carrying _Wolbachia_ into an area in which mosquitoes are not infected with _Wolbachia_. The model permits us to observe the dynamics of the two populations of mosquitoes, in which a subset has the bacterial symbiont _Wolbachia_. The model shows how the population that has Wolbachia increases (or not) over time and the impact of the number of releases and the number of mosquitoes released.


## HOW IT WORKS

Female mosquitoes from species such as _Aedes aegypti_ when carrying _Wolbachia_ generally 
generate offspring that also have _Wolbachia_ (vertical transmission).  
Eggs from female mosquitoes that do not have _Wolbachia_ fail to produce offspring when fertilized by male mosquitoes that do have _Wolbachia_ due to cytoplasmic incompatibility.

It has been shown that a mosquito such as _Aedes aegypti_, when carrying _Wolbachia_, do not transmit, either partially or totally, pathogens of diseases such as dengue, chikungunya etc.
_Wolbachia_ may also decrease mosquito fitness, by reducing its breeding rate or its survival rate.
For these reasons, efforts to generate _Wolbachia_-carrying mosquitoes are an important path for control of such diseases. 

The model will show how the two subpopulations progress over time and, eventually, one population is supressed, depending on mosquito fitness and the initial frequency of population that carries _Wolbachia_.

The vertical transmission is assumed to be perfect, and cytoplasmic incompatibility is also assumed to happen in the case of crossings between males with _Wolbachia_ and _Wolbachia_-free females.

This model evolves from a previous version considering that a number of releases of mosquitoes infected with _Wolbachia_ occur on a weekly basis.

## HOW TO USE IT

Each "tick" represents a day in the time scale of this model.

The model starts with a population of nMosquitoes, each of which may have _Wolbachia_.  
There is an initial release of nWolbMosquitoes at setup, and there will be weekly releases up to a number determined by nReleasesTotal.
The carrying capacity is set at maximum 4000.

Mosquitoes move randomly in the world.  Red mosquitoes have _Wolbachia_, whereas blue ones do not.

The average lifetime for _Wolbachia_-free mosquitoes is determined by normal-lifespan.
A female mosquito that does not carry Wolbachia produces offspring at rate rateOffspring.

Conversely, the average lifetime for _Wolbachia_-carrying mosquitoes is determined by a fraction of normal-lifespan given by fracLifespanWolbachia.
A female mosquito that carries _Wolbachia_ produces offspring at rate given by a fraction of rateOffspring given by fracRateOffspWolbachia.

After setting parameters, press SETUP, then GO.

## RELATED MODELS

This model has a setup and structure adapted from the original Wolbachia model (Villela 2013). 


## HOW TO CITE

If you mention this model in a publication, we ask that you include these citations for the model itself and for the NetLogo software:

* Villela, D., Garcia, G.A. (2014). NetLogo Wolbachia-Release model.  NetLogo Modeling Commons. Oswaldo Cruz Foundation, Rio de Janeiro, Brazil.

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2014 D. Villela & G. Garcia
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
