;;;;;;;;;;   Declarations   ;;;;;;;;;;

globals [
  dom                                             ; homozygous dominant (AA)
  rec                                             ; homozygous recessive (aa)
  hetero                                          ; hetrozygous (Aa)
  ]

breed [hawks hawk]
breed [mice mouse]

hawks-own [
  vision 
  age 
  hawk-geno
  ]

mice-own [
  camo 
  mouse-geno
  ]

;;;;;;;;;;   Setup and Go Procedures   ;;;;;;;;;;

to setup  
  ca
  
  set dom 1                                       ; so code is easier to read and
  set hetero 2                                    ; values are used to calculate 
  set rec 3                                       ; ifelse logical tests
    
  ask patches [
    set pcolor 63 + random-float 4                ; its a patchwork world!
    ]
  
  create-hawks init-hawks [
    set shape "hawk" 
    set color 36
    set age 0
    set hawk-geno random 3 + 1                    ; random initial geno & pheno freqs 
    set vision random-normal init-mean-vision init-SD-vision
    ]

  create-mice init-mice [
    set shape "mouse top"
    set size .2
    set color 32
    set mouse-geno random 3 + 1                   ; random initial geno & pheno freqs
    set camo random-normal init-mean-camo init-SD-camo
    ] 
  
  distribute                                      ; randomly place hawks and mice
  
  reset-ticks  
end

to go
  if ticks = 50 [stop]
  ask hawks [hunt]
  ask hawks [starve]  
  ask mice [reproduce]
  distribute 
  tick
end

;;;;;;;;;;  Hawk Procedures   ;;;;;;;;;;

to hunt
  if any? mice-here and vision >= min [camo] of mice-here [ ; visible mice here?
    if-else vision > min [camo] of mice-here
      [eat-and-reproduce]                         ; if vision > camo eat and reproduce
      [if random 2 = 0 [eat-and-reproduce]]       ; else vision = camo so flip a coin
    hunt                                          ; recurse to continue eating until all possible mice are eaten
  ]             
end

to eat-and-reproduce
  ask min-one-of mice-here [camo] [die]           ; eat the mouse
  hatch 2 [                                       ; reproduce with 
    set age ticks                                 ; genotype determined by random mating
    set hawk-geno child-geno [hawk-geno] of self [hawk-geno] of one-of other hawks 
    set vision child-vision                       ; phenotype determined by genotype
    ]
  die                                             ; die after reproducing
end

to starve
  if (ticks - age >= 2) [die]                     ; if a hawk hasn't eaten in 2 ticks it dies    
end

;;;;;;;;;;  Mouse Procedures   ;;;;;;;;;;

to reproduce
  if not any? other mice-here [                   ; if there is no competition reproduce with
  hatch 2 [                                       ; genotype determined by random mating
    set mouse-geno child-geno [mouse-geno] of self [mouse-geno] of one-of other mice
    set camo child-camo                           ; phenotype determined by genotype
    ]
    die                                           ; die after reproducing
  ]
end

;;;;;;;;;;  General Procedures   ;;;;;;;;;;
  
to distribute                                     ; randomly distribute hawks and mice
  ask mice [setxy random-xcor random-ycor set heading random 359]
  ask hawks [setxy random-xcor random-ycor set heading random 359] 
end
 
to reset                                          ; load default values
  set mice-coevolve? true
  set camo-dominant? true
  set init-mice 50
  set init-mean-camo 5
  set init-SD-camo 1.5
  set hawks-coevolve? true
  set vision-dominant? true
  set init-hawks 25  
  set init-mean-vision 5
  set init-SD-vision 1.5
end

;;;;;;;;;;   Reporters   ;;;;;;;;;; 

to-report child-vision
  ;returns a new vision score based on current vision score, hawk genotype  
  ;and a random factor which is a f(StdDev) of the vision scores
  
  ; set a random change factor as a f(StdDev)
  let x random-float (init-SD-vision + standard-deviation [vision] of hawks) / 2                    
  ;let x 1 
    
  ifelse hawks-coevolve?                          ; if vision is an advantage
    [ifelse vision-dominant?                      ; and                    
      [ifelse hawk-geno < 3                       ; vision is dominant
        [set vision vision + x]                   ; increase score for AA and Aa
        [set vision vision - x]                   ; decrease score for aa
      ]                                           ; else
      [ifelse hawk-geno < 3                       ; vision is recessive so
        [set vision vision - x]                   ; decrease score for AA and Aa
        [set vision vision + x]                   ; increase score for aa
      ]
    ]                                             ; else vision is not an advantage   
    [set vision random-normal init-mean-vision init-SD-vision] ; do random draw      
  if vision < 0 [set vision 0]                    ; keep scores non-negative
  report vision  
end

to-report child-camo
  ;returns a new camo score based on current camouflage score, mouse genotype 
  ;and a random factor which is a f(StdDev) of the camo scores
  
  ; set a random change factor as a f(StdDev)
  let x random-float (init-SD-camo + standard-deviation [camo] of mice) / 2
  ;let x 1
        
  ifelse mice-coevolve?                           ; if camouflage is an advantage    
    [ifelse camo-dominant?                        ; and
      [ifelse mouse-geno < 3                      ; vision is dominant
        [set camo camo + x]                       ; increase score for AA and Aa 
        [set camo camo - x]                       ; decrease score for aa
      ]                                           ; else
      [ifelse mouse-geno < 3                      ; vision is recessive so
        [set camo camo - x]                       ; decrease score for AA and Aa
        [set camo camo + x]                       ; increase score for aa
      ] 
    ]                                             ; else camouflage is not an advantage                                                
    [set camo random-normal init-mean-camo init-SD-camo]     ; do random draw    
  if camo < 0 [set camo 0]                        ; keep scores non-negative
  report camo    
end

to-report child-geno [geno1 geno2]
  ; reports a child's genotype given both parents' genotypes
  ; parents are a survivor and a random other turtle of the same breed

  let x  0
  let y 0
  let geno 0
  set x geno1 * geno2                             ; produces an unique ifelse criterion
  ifelse x = 1                                    ; AA x AA
         [set geno dom]   
    [ifelse x = 2                                 ; AA x Aa
         [ifelse random 2 = 0
           [set geno dom]                         ; 0.5 prob AA
           [set geno hetero]                      ; 0.5 prob Aa
           ]
    [ifelse x = 3                                 ; AA x aa
         [set geno hetero]
    [ifelse x = 4                                 ; Aa x Aa 
         [set y random-float 1
           ifelse y < 0.25                        ; 0.25 prob AA
             [set geno dom]
             [ifelse y > 0.75                     ; 0.25 prob aa
               [set geno rec]
             [set geno hetero                     ; else must be Aa
             ]]]
    [ifelse x = 6                                 ; Aa x aa
         [ifelse random 2 = 0
           [set geno rec]                         ; 0.5 prob aa
           [set geno hetero]                      ; 0.5 prob Aa
           ]
    [set geno rec                                 ; aa x aa
    ]]]]]  

  report geno
end  
@#$#@#$#@
GRAPHICS-WINDOW
19
79
329
410
-1
-1
30.0
1
10
1
1
1
0
0
0
1
0
9
0
9
1
1
1
ticks
30.0

BUTTON
16
13
90
63
Setup
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
90
13
167
63
Run
go\n
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
340
12
604
132
Camouflage
Score
Freq
0.0
10.0
0.0
20.0
true
false
"set-plot-x-range 0 100" "if count mice != 0 and max [camo] of mice > plot-x-max [set-plot-x-range 0 round max [camo] of mice]"
PENS
"default" 1.0 1 -2674135 true "" "histogram [camo] of mice"

PLOT
604
13
866
133
Vision
Score
Freq
0.0
10.0
0.0
20.0
true
false
"set-plot-x-range 0 100" "if count hawks != 0 and max [vision] of hawks > plot-x-max [set-plot-x-range 0 round max [vision] of hawks]"
PENS
"default" 1.0 1 -13345367 true "" "histogram [vision] of hawks"

PLOT
340
177
603
297
Populations
Generation
Num
0.0
50.0
0.0
10.0
true
true
"" ""
PENS
"Mice" 1.0 0 -2674135 true "" "plot count mice"
"Hawks" 1.0 0 -13345367 true "" "plot count hawks"

PLOT
603
177
867
340
Phenotype Scores
Generation
Score
0.0
50.0
0.0
10.0
true
true
"" ""
PENS
"Camo" 1.0 0 -2674135 true "" "if count mice > 0 [plot mean [camo] of mice]"
"Vision" 1.0 0 -13345367 true "" "if count hawks > 0 [plot mean [vision] of hawks]"

SLIDER
18
528
174
561
init-mean-camo
init-mean-camo
0
10
5
1
1
NIL
HORIZONTAL

SLIDER
174
529
331
562
init-mean-vision
init-mean-vision
0
10
5
1
1
NIL
HORIZONTAL

SLIDER
18
560
174
593
init-SD-camo
init-SD-camo
0.1
3
1.5
.1
1
NIL
HORIZONTAL

SLIDER
174
560
331
593
init-SD-vision
init-SD-vision
0.1
3
1.5
0.1
1
NIL
HORIZONTAL

BUTTON
254
13
330
63
Reset
reset
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
18
497
174
530
init-mice
init-mice
0
200
50
1
1
NIL
HORIZONTAL

SLIDER
173
497
330
530
init-hawks
init-hawks
0
100
25
1
1
NIL
HORIZONTAL

BUTTON
168
13
253
63
5 Gens
repeat 5 [go]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
340
132
425
177
mean camo
mean [camo] of mice
1
1
11

MONITOR
602
132
689
177
mean vision
mean [vision] of hawks
1
1
11

MONITOR
424
132
510
177
SD camo
standard-deviation [camo] of mice
2
1
11

MONITOR
686
132
772
177
SD vision
standard-deviation [vision] of hawks
2
1
11

MONITOR
340
297
474
342
mouse population
count mice
1
1
11

MONITOR
509
132
601
177
range camo
max [camo] of mice - min [camo] of mice
1
1
11

MONITOR
469
296
603
341
hawk population
count hawks
1
1
11

MONITOR
773
132
866
177
range vision
max [vision] of hawks - min [vision] of hawks
1
1
11

SWITCH
174
430
331
463
hawks-coevolve?
hawks-coevolve?
0
1
-1000

SWITCH
18
430
173
463
mice-coevolve?
mice-coevolve?
0
1
-1000

SWITCH
18
465
174
498
camo-dominant?
camo-dominant?
0
1
-1000

SWITCH
173
464
330
497
vision-dominant?
vision-dominant?
0
1
-1000

PLOT
603
340
818
476
Hawk Genotypes
Generation
Freq
0.0
50.0
0.0
1.0
true
true
"" ""
PENS
"AA" 1.0 0 -2674135 true "" "if count hawks != 0 [plot count hawks with [hawk-geno = dom] / count hawks]"
"Aa" 1.0 0 -16777216 true "" "if count hawks != 0 [plot count hawks with [hawk-geno = hetero] / count hawks]"
"aa" 1.0 0 -13345367 true "" "if count hawks != 0 [plot count hawks with [hawk-geno = rec] / count hawks]"

PLOT
340
342
555
477
Mouse Genotypes
Generation
Freq
0.0
50.0
0.0
1.0
true
true
"" ""
PENS
"AA" 1.0 0 -2674135 true "" "if count mice != 0 [plot count mice with [mouse-geno = dom] / count mice]"
"Aa" 1.0 0 -16777216 true "" "if count mice != 0 [plot count mice with [mouse-geno = hetero] / count mice]"
"aa" 1.0 0 -13345367 true "" "if count mice != 0 [plot count mice with [mouse-geno = rec] / count mice]"

PLOT
605
476
818
596
Hawk Alleles
Generation
Freq
0.0
50.0
0.0
1.0
true
true
"" ""
PENS
" A " 1.0 0 -2674135 true "" "if count hawks != 0 [plot (count hawks with [hawk-geno = dom] + 0.5 * count hawks with [hawk-geno = hetero]) / count hawks]"
" a " 1.0 0 -13345367 true "" "if count hawks != 0 [plot (count hawks with [hawk-geno = rec] + 0.5 * count hawks with [hawk-geno = hetero]) / count hawks]"

PLOT
340
476
556
596
Mouse Alleles
Generation
Freq
0.0
50.0
0.0
1.0
true
true
"" ""
PENS
" A " 1.0 0 -2674135 true "" "if count mice != 0 [plot (count mice with [mouse-geno = dom] + 0.5 * count mice with [mouse-geno = hetero]) / count mice]"
" a " 1.0 0 -13345367 true "" "if count mice != 0 [plot (count mice with [mouse-geno = rec] + 0.5 * count mice with [mouse-geno = hetero]) / count mice]"

MONITOR
555
342
605
387
   AA
count mice with [mouse-geno = dom] / count mice
2
1
11

MONITOR
555
386
605
431
   Aa
count mice with [mouse-geno = hetero] / count mice
2
1
11

MONITOR
555
430
605
475
    aa
count mice with [mouse-geno = rec] / count mice
2
1
11

MONITOR
818
340
868
385
   AA
count hawks with [hawk-geno = dom] / count hawks
2
1
11

MONITOR
818
386
868
431
   Aa
count hawks with [hawk-geno = hetero] / count hawks
2
1
11

MONITOR
818
432
868
477
    aa
count hawks with [hawk-geno = rec] / count hawks
2
1
11

MONITOR
555
508
605
553
    A 
(count mice with [mouse-geno = dom] + 0.5 * count mice with [mouse-geno = hetero]) / count mice
2
1
11

MONITOR
555
550
605
595
    a
(count mice with [mouse-geno = rec] + 0.5 * count mice with [mouse-geno = hetero]) / count mice
2
1
11

MONITOR
817
505
867
550
   A
(count hawks with [hawk-geno = dom] + 0.5 * count hawks with [hawk-geno = hetero]) / count hawks
2
1
11

MONITOR
817
550
867
595
   a
(count hawks with [hawk-geno = rec] + 0.5 * count hawks with [hawk-geno = hetero]) / count hawks
2
1
11

@#$#@#$#@
## WHAT IS IT?

This simulation models coevolution of a predator-prey system.  It is based on the board game **SIMULATING NATURAL SELECTION** by Robert P. Gendron from Indiana University of Pennsylvania.  The model incorporates the following elements of a natural system: Variability, heritability, competition, predation, and differential reproduction.  Users can set the intial values for adaptation scores, variability of adaptation scores and population size.  Users can also set whether a trait is inherited as a dominant or recessive allele.  The simulation produces behaviors that demonstrate directional selection, the coevolution of traits, genetic drift, carrying capacity and predator-prey population cycles.

## HOW IT WORKS

The two populations in the model are hawks and mice.  Hawks have a vision score and mice have a camouflage score.  Populations are intialized with user specified number, mean score and score variation.  Individual mice and hawks are assigned a random genotype and phenotype on setup.  Each population is then randomly distributed in the world. 

Each tick in the model represents a discrete set of events that occur in the following order:
  
  1. **PREDATION:** 
If a hawk and mouse are on the same patch and the hawk's vision score is greater than the mouse's camouflage score then the hawk eats the mouse, reproduces and dies (See 2 below). If there is a tie in the scores a coin flip determines if the mouse is eaten or survives.  If there are many mice on a patch, the mouse with the minimum camouflage score is eaten.
  2. **HAWK REPRODUCTION:** 
Each hawk that successfully eats produces two offspring as follows: 
**> If hawks-coevolve? is ON:** 
Hawk vision **is** affected by mouse camouflage.  Each hawk reproduces with another random hawk.  Each child's genotype is determined by their parents' genotypes based on simple Mendelian probabilites. Hawk vision scores (phenotypes) are determined from their genotypes as described below:
**If vision-dominant? is ON:** Homozygous dominant and hetrozygous children are assigned a vision score equal to the parent score _**plus**_ a random value which is calculated as a function of the initial variation in the vision scores. Homozygous recessive children are assigned a vision score equal to the parent score _**minus**_ the random value calculated as described above.
**If vision-dominant? is OFF:** Homozygous recessive children are assigned a vision score equal to the parent score _**plus**_ a random value which is calculated as a function of the intial variation in the vision scores. Homozygous dominant and hetrozygous children  children are assigned a vision score equal to the parent score _**minus**_ the random value calculated as described above.
**> If hawks-coevolve? is OFF:** 
Hawk vision **is not** affected by mouse camouflage.  Each hawk reproduces with another random hawk.  Each child's genotype is determined by their parents' genotypes based on simple Mendelian probabilites.  Both children are assigned vision scores (phenotypes) drawn randomly from the normal distribution defined by the init-mean-vision and init-SD-vision sliders.
**>Both Conditions:** 
Each new hawk is assigned an age equal to the tick count when it is hatched.  
The parent hawk dies immediately after reproducing. 
If there are still mice on the patch where the new hawks are hatched, the newly hatched hawks get a chance to hunt and reproduce until all mice on a given patch with camouflage scores lower than hawk vision scores are eaten.
  3. **HAWK STARVATION:**  
If a hawk does not eat within 2 ticks of being hatched it dies.
  4. **MOUSE REPRODUCTION:** 
Mice that survive and are alone on a patch (i.e., no competition for resources from other mice) reproduce two offpring as follows:
**> If mice-coevolve? is ON:** 
Mouse camouflage **is** affected by hawk vision.  Each mouse reproduces with another random mouse.  Each child's genotype is determined by their parents' genotypes based on simple Mendelian probabilites. Mouse caouflage scores (phenotypes) are determined from their genotypes as described below: 
**If camo-dominant? is ON:** Homozygous dominant and hetrozygous children are assigned a camouflage score equal to the parent score _**plus**_ a random value which is calculated as a function of the initial variation in the camouflage scores. Homozygous recessive children are assigned a camouflage score equal to the parent score _**minus**_ the random value calculated as described above.
**If camo-dominant? is OFF:** Homozygous recessive children are assigned a camouflage score equal to the parent score _**plus**_ a random value which is calculated as a function of the intial variation in the camouflage scores. Homozygous dominant and hetrozygous children  children are assigned a camouflage score equal to the parent score _**minus**_ the random value calculated as described above.
**> If mice-coevolve? is OFF:** 
Mouse camouflage **is not** affected by hawk vision.  Each mouse reproduces with another random mouse.  Each child's genotype is determined by their parents' genotypes based on simple Mendelian probabilites.  Both children are assigned camouflage scores (phenotypes) drawn randomly from the normal distribution defined by the init-mean-camo and init-SD-camo sliders.
**>Both Conditions:** 
The parent mouse dies immediately after reproducing. 
  5. **RANDOM DISTRIBUTION:**  
All surviving hawks and mice are randomly redistributed in the world and the the above steps are repeated.

## HOW TO USE IT

  1. The **mice-coevolve?** and **hawks-coevolve?** switches determine if the mouse and hawk populations evlove in response to environmental pressures (On) or randomly (Off)
  2. The **vision-dominant?** and **camo-dominant?** switches determine if improvements in each trait are dominant (On) or recessive (Off) 
  2. The **init-?**, **init-mean-?** and **init-SD-?** sliders set the initial population size (mice or hawks), the intial mean value of the adpatations (camouflage or vision) and the variation within the population for each adaptation (camouflage or vision), respectively.
  3. The **Reset** button returns all settings to their default values.
  4. Use the **Setup** button to initalize the simulation to the values you set using the sliders and switches described above.  Press the **Run** button to run the simulation continuously or use the **5 Gens** button to advance the simulation 5 generations at a time.   

## THINGS TO TRY

Use the model to experiment.  Start by making changes to only one setting at a time. Run each experiment several times as the results may differ. Make note of these different outcomes and the frequency with which they occur.  For each experiment be sure to explain what you think might be causing the results you observe. 

Some suggested experiments are listed below 

  1. Run the model using the default settings.  Use the **5 Gens** button to advance the simulation 5 generations at a time.  Try to predict what will happen in the next 5 generations.  Press the button again to see if your predictions were accurate.   
  2. Set the init-hawks slider to zero.
  3. Set camo, vision or both adapt? switches to off.
  4. Change the mean and variation (SD) values of the mouse camouflage value.
  5. Change the mean and variation (SD) values of the hawk vision value.
  6. Increase or decrease the intial sizes of the mouse, hawk or both populations.
  7. Try to get both populations to survive most of the time when mice-coevolve? is off.
  8. Change the mode of inheritance combinations for vision and camouflage (i.e., both dominant, both recessive, one dominant and the other recessive)

## THINGS TO NOTICE

1. What happens to the vision and camouflage scores under different settings?
2. How do these scores change relative to one another?
3. What happens to the vision and camouflage variation under different settings?
4. How do the populations change relative to one another?
5. Does either population go extinct? reach a carrying capacity?
6. Are there any notable trends in the allele frequencies?
7. Is change allele frequency affected by population size?
8. Do any alleles become fixed / extinct?
9. Does recessiveness or dominance affect the rate at which alleles spread through the population and / or become fixed?

## EXTENSIONS

Play the board game (see link below) and explore the code to see how it works.
Use the difference between patch color and mouse color to determine camouflage scores.

## RELATED MODELS

In the NetLogo Library:

  1. Bug Hunt Camouflage
  2. Bug Hunt Coevolution
  3. Bug Hunt Drift
  4. Red Queen
  5. Wolf Sheep Predation
  6. Wolf Sheep Stride Inheritance

## CREDITS AND REFERENCES

Dr Robert Gendron's Faculty Page at IUP: 
http://www.iup.edu/page.aspx?id=85437

A copy of the original board game directions:
http://evolution.berkeley.edu/evolibrary/search/lessonsummary.php?source=Gendron,%20Robert&thisaudience=13-16&resource_id=47

## COPYRIGHT AND LICENSE

Copyright 2013 Michael Zito.

![CC BY-NC-SA 3.0](http://i.creativecommons.org/l/by-nc-sa/3.0/88x31.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

hawk
true
0
Polygon -7500403 true true 151 170 136 170 123 229 143 244 156 244 179 229 166 170
Polygon -16777216 true false 152 154 137 154 125 213 140 229 159 229 179 214 167 154
Polygon -7500403 true true 151 140 136 140 126 202 139 214 159 214 176 200 166 140
Polygon -16777216 true false 151 125 134 124 128 188 140 198 161 197 174 188 166 125
Polygon -7500403 true true 152 86 227 72 286 97 272 101 294 117 276 118 287 131 270 131 278 141 264 138 267 145 228 150 153 147
Polygon -7500403 true true 160 74 159 61 149 54 130 53 139 62 133 81 127 113 129 149 134 177 150 206 168 179 172 147 169 111
Circle -16777216 true false 144 55 7
Polygon -16777216 true false 129 53 135 58 139 54
Polygon -7500403 true true 148 86 73 72 14 97 28 101 6 117 24 118 13 131 30 131 22 141 36 138 33 145 72 150 147 147

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

mouse side
false
0
Polygon -7500403 true true 38 162 24 165 19 174 22 192 47 213 90 225 135 230 161 240 178 262 150 246 117 238 73 232 36 220 11 196 7 171 15 153 37 146 46 145
Polygon -7500403 true true 289 142 271 165 237 164 217 185 235 192 254 192 259 199 245 200 248 203 226 199 200 194 155 195 122 185 84 187 91 195 82 192 83 201 72 190 67 199 62 185 46 183 36 165 40 134 57 115 74 106 60 109 90 97 112 94 92 93 130 86 154 88 134 81 183 90 197 94 183 86 212 95 211 88 224 83 235 88 248 97 246 90 257 107 255 97 270 120
Polygon -16777216 true false 234 100 220 96 210 100 214 111 228 116 239 115
Circle -16777216 true false 246 117 20
Line -7500403 true 270 153 282 174
Line -7500403 true 272 153 255 173
Line -7500403 true 269 156 268 177

mouse top
true
0
Polygon -7500403 true true 144 238 153 255 168 260 196 257 214 241 237 234 248 243 237 260 199 278 154 282 133 276 109 270 90 273 83 283 98 279 120 282 156 293 200 287 235 273 256 254 261 238 252 226 232 221 211 228 194 238 183 246 168 246 163 232
Polygon -7500403 true true 120 78 116 62 127 35 139 16 150 4 160 16 173 33 183 60 180 80
Polygon -7500403 true true 119 75 179 75 195 105 190 166 193 215 165 240 135 240 106 213 110 165 105 105
Polygon -7500403 true true 167 69 184 68 193 64 199 65 202 74 194 82 185 79 171 80
Polygon -7500403 true true 133 69 116 68 107 64 101 65 98 74 106 82 115 79 129 80
Polygon -16777216 true false 163 28 171 32 173 40 169 45 166 47
Polygon -16777216 true false 137 28 129 32 127 40 131 45 134 47
Polygon -16777216 true false 150 6 143 14 156 14
Line -7500403 true 161 17 195 10
Line -7500403 true 160 22 187 20
Line -7500403 true 160 22 201 31
Line -7500403 true 140 22 99 31
Line -7500403 true 140 22 113 20
Line -7500403 true 139 17 105 10

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

rabbit
false
0
Polygon -7500403 true true 61 150 76 180 91 195 103 214 91 240 76 255 61 270 76 270 106 255 132 209 151 210 181 210 211 240 196 255 181 255 166 247 151 255 166 270 211 270 241 255 240 210 270 225 285 165 256 135 226 105 166 90 91 105
Polygon -7500403 true true 75 164 94 104 70 82 45 89 19 104 4 149 19 164 37 162 59 153
Polygon -7500403 true true 64 98 96 87 138 26 130 15 97 36 54 86
Polygon -7500403 true true 49 89 57 47 78 4 89 20 70 88
Circle -16777216 true false 37 103 16
Line -16777216 false 44 150 104 150
Line -16777216 false 39 158 84 175
Line -16777216 false 29 159 57 195
Polygon -5825686 true false 0 150 15 165 15 150
Polygon -5825686 true false 76 90 97 47 130 32
Line -16777216 false 180 210 165 180
Line -16777216 false 165 180 180 165
Line -16777216 false 180 165 225 165
Line -16777216 false 180 210 210 240

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
