globals [
  rich-population
  middle-population
  poor-population
  region-boundaries
  hospital-beds-occupied
  population-density
  dead-total-age
  dead-total-gdp
  number-vaccinated-once
  number-vaccinated-twice
  n-cases
  n-confirmed
  day
  hour
  vaccine-interval
  test-interval
  leave-time
  return-time
  one-dose-success
]

breed[ rich-houses rich-house ]
breed[ middle-houses middle-house ]
breed[ poor-houses poor-house ]
breed[ people person]
breed[ houses house ]

patches-own [
  region
]

turtles-own[
  homebase
  commute
  susceptible?
  infected?
  immune?
  stationary?
  hospitalized?
  vaccinated?
  one-dose?
  two-dose?
  sick-time
  age
  mobility
  infected-mortality
  chance-of-recovery
  income
  antivaxxer?
  household-size
  tested?
  tested-positive?
  quarantined?
  masked?
  dose-counter
  infection-rate
  area
]

to go
  if (count people with [infected?] = 0) [stop]
  move
  random-testing
  quarantine-positive-tested
  infect-susceptibles
  vaccinate
  illness
  end-quarantine
  clock
  tick
end

to setup
  clear-all
  setup-globals
  setup-regions 3
  setup-people
  setup-houses
  setup-segregate
  setup-infected
  reset-ticks
end

to setup-globals
  set population-density initial-population
  set number-vaccinated-once 0
  set number-vaccinated-twice 0
  set dead-total-age 0
  set dead-total-gdp 0
  set n-cases 0
  set n-confirmed 0
  set hospital-beds-occupied 0
  (ifelse distribution-interval = "Every Hour" [set vaccine-interval 24]
      distribution-interval = "Once a Day" [set vaccine-interval 1]
      distribution-interval = "Once a Week" [set vaccine-interval 1 / 7])

  (ifelse testing-interval = "Every Hour" [set test-interval 24]
    testing-interval = "Once a Day" [set test-interval 1]
    testing-interval = "Once a Week" [set test-interval 1 / 7])

  (ifelse lockdown-end = "1:00 AM" [set leave-time 1]
  lockdown-end = "2:00 AM" [set leave-time 2]
  lockdown-end = "3:00 AM" [set leave-time 3]
  lockdown-end = "4:00 AM" [set leave-time 4]
  lockdown-end = "5:00 AM" [set leave-time 5]
  lockdown-end = "6:00 AM" [set leave-time 6]
  lockdown-end = "7:00 AM" [set leave-time 7]
  lockdown-end = "8:00 AM" [set leave-time 8]
  lockdown-end = "9:00 AM" [set leave-time 9]
  lockdown-end = "10:00 AM" [set leave-time 10]
  lockdown-end = "11:00 AM" [set leave-time 11]
  lockdown-end = "12:00 PM" [set leave-time 12])

  (ifelse lockdown-start = "1:00 PM" [set return-time 1]
    lockdown-start = "2:00 PM" [set return-time 2]
    lockdown-start = "3:00 PM" [set return-time 3]
    lockdown-start = "4:00 PM" [set return-time 4]
    lockdown-start = "5:00 PM" [set return-time 5]
    lockdown-start = "6:00 PM" [set return-time 6]
    lockdown-start = "7:00 PM" [set return-time 7]
    lockdown-start = "8:00 PM" [set return-time 8]
    lockdown-start = "9:00 PM" [set return-time 9]
    lockdown-start = "10:00 PM" [set return-time 10]
    lockdown-start = "11:00 PM" [set return-time 11]
    lockdown-start = "12:00 AM" [set return-time 12])
end


to setup-regions [ num-regions ] ;num-regions = 3
  if income-separation? [
  foreach region-divisions num-regions draw-region-division set region-boundaries calculate-region-boundaries num-regions
  let region-numbers (range 1 (num-regions + 1))
  (foreach region-boundaries region-numbers [ [boundaries region-number] ->
    ask patches with [ pxcor >= first boundaries and pxcor <= last boundaries ] [
      set region region-number
    ]
  ])
  ]

  ask patches with [ region != 0 ] [
   set pcolor 2 + region * 10
    set plabel-color pcolor + 1

  ]
end

to-report region-divisions [ num-regions ] ;num-regions = 3
  ; This procedure reports a list of pxcor that should be outside every region.
  ; Patches with these pxcor will act as "dividers" between regions.
  report n-values (num-regions + 1) [ n ->
    [ pxcor ] of patch (min-pxcor + (n * ((max-pxcor - min-pxcor) / num-regions))) 0
  ]
end

to draw-region-division [ x ]
  ask patches with [ pxcor = x ] [
    set pcolor grey
  ]
  create-turtles 1 [
    ; use a temporary turtle to draw a line in the middle of our division
    setxy x max-pycor
    set heading 0
    set color grey - 3
    pen-down
    forward world-height
    set xcor xcor + 1 / patch-size
    right 180
    set color grey + 3
    forward world-height
    die]
end

to-report calculate-region-boundaries [ num-regions ] ;num-regions = 3
  ; The region definitions are built from the region divisions:
  let divisions region-divisions num-regions
  ; Each region definition lists the min-pxcor and max-pxcor of the region.
  ; To get those, we use `map` on two "shifted" copies of the division list,
  ; which allow us to scan through all pairs of dividers
  ; and built our list of definitions from those pairs:
  report (map [ [d1 d2] -> list (d1 + 1) (d2 - 1) ] (but-last divisions) (but-first divisions))
end

to setup-people
  create-people initial-population [
    set color blue
    set shape "person"
    set size 2
    set commute commute-distance
    set heading random 360
    set susceptible? true
    set infected? false
    set immune? false
    set stationary? false
    set hospitalized? false
    set vaccinated? false
    set tested? false
    set tested-positive? false
    set quarantined? false
    set one-dose? false
    set two-dose? false
    set sick-time 0
    set dose-counter 0
    set antivaxxer? random 100 <= anti-vaccination-proportion
    set area 0

    let coinA random 100
      let coinB random 10
        ifelse (coinA < percent-age-ten )[
          set age coinB
          set infected-mortality 0.0001][
            ifelse (coinA < percent-age-twenty )[
              set age (coinB + 10)
              set infected-mortality 0.05][
                ifelse (coinA < percent-age-thirty )[
                  set age (coinB + 20)
                  set infected-mortality 0.105][
                    ifelse (coinA < percent-age-forty )[
                      set age (coinB + 30)
                      set infected-mortality 0.188][
                        ifelse (coinA < percent-age-fifty )[
                          set age (coinB + 40)
                          set infected-mortality 0.295][
                            ifelse (coinA < percent-age-sixty )[
                              set age (coinB + 50)
                              set infected-mortality 0.8][
                                ifelse (coinA < percent-age-seventy )[
                                  set age (coinB + 60)
                                  set infected-mortality 2.7][
                                    ifelse (coinA < percent-age-eighty )[
                                       set age (coinB + 70)
                                       set infected-mortality 7.98]
                                       [set age (coinB + 80)
                                        set infected-mortality 15.9]
                            ]
                           ]
                          ]
                         ]
                        ]
                       ]
                      ]

    set income random-normal average-household-income income-spread
    if (income < 0)[set income 0]

    set infection-rate  (transmission-rate + (population-density * 0.00004 ) - (income * 0.0000001))
    if (infection-rate > 1) [set infection-rate 1]

    set infected-mortality (infected-mortality + (-0.00004 * (income - 50577)))
    if (infected-mortality < 0) [set infected-mortality 0]

    set chance-of-recovery (100 - infected-mortality)
    if (chance-of-recovery < 0)[set chance-of-recovery 0]

    ifelse masks?
    [
      ifelse (random-float 100 < mask-effort)
      [set masked? true]
      [set masked? false]
    ]
    [
      set masked? false
    ]

    ifelse age < 15 [
      set mobility (age / 50)][
      ifelse age > 65[
        set mobility ((age - 50) / 50)]
      [set mobility ((100 - age) / 50)] ;15-65 year olds
    ]
  ]
end

to setup-infected

  let rich-count count people with [income > middle-class-income-end]
  let middle-count count people with [income > middle-class-income-start and income < middle-class-income-end]
  let poor-count count people with [income < middle-class-income-start]

  ifelse rich-count > initial-infected-rich
  [
    ask n-of initial-infected-rich people with [income > middle-class-income-end]
    [
      set color red
      set infected? true
    ]
  ]
  [
    ask n-of rich-count people with [income > middle-class-income-end]
    [
      set color red
      set infected? true
    ]
  ]

  ifelse middle-count > initial-infected-middle
  [
    ask n-of initial-infected-middle people with [income > middle-class-income-start and income < middle-class-income-end]
    [
      set color red
      set infected? true
    ]
  ]
  [
    ask n-of middle-count people with [income > middle-class-income-start and income < middle-class-income-end]
    [
      set color red
      set infected? true
    ]
  ]

  ifelse poor-count > initial-infected-poor
  [
    ask n-of initial-infected-poor people with [income < middle-class-income-start]
    [
      set color red
      set infected? true
    ]
  ]
  [
    ask n-of poor-count people with [income < middle-class-income-start]
    [
      set color red
      set infected? true
    ]
  ]


end

to setup-houses

    (ifelse housing? and income-separation? [

    set rich-population count people with [income > middle-class-income-end]
    set middle-population count people with [income > middle-class-income-start and income < middle-class-income-end]
    set poor-population count people with [income < middle-class-income-start]

    let r 0
    ifelse (rich-population < rich-household-size)
    [set r 1]
    [set r rich-population / rich-household-size]

    create-rich-houses r [
    let rich-patch patches with [ region = 3 ]
    move-to one-of rich-patch
    set color white set shape "house" set size 2]

    let m 0
    ifelse (middle-population < middle-class-household-size)
    [set m 1]
    [set m middle-population / middle-class-household-size]

    create-middle-houses m [
    let middle-patch patches with [ region = 2 ]
    move-to one-of middle-patch
    set color white set shape "house" set size 2
    ]

    let p 0
    ifelse (poor-population < poor-household-size)
    [set p 1]
    [set p poor-population / poor-household-size]

    create-poor-houses p [
      let poor-patch patches with [ region = 1 ]
      move-to one-of poor-patch
      set color white set shape "house" set size 2
    ]
  ]
    housing? and not income-separation? [
      create-houses initial-population / 4 [
      setxy random-xcor random-ycor
        set color white set shape "house" set size 2
      ]
    ]
    )

end

to setup-segregate
  ask people [
  (ifelse housing? and income-separation? [
      if (income > middle-class-income-end and count rich-houses > 0 )[set homebase one-of rich-houses move-to homebase]
      if (income < middle-class-income-end and income > middle-class-income-start and count middle-houses > 0) [set homebase one-of middle-houses move-to homebase]
      if (income < middle-class-income-start and count poor-houses > 0) [set homebase one-of poor-houses move-to homebase]
    ]
      housing? and not income-separation? [
        set homebase one-of houses move-to homebase
      ]
      not housing? and not income-separation? [
      setxy random-xcor random-ycor
      ]
    )
  ]
end

to move
  (ifelse housing? and lockdown?
    [
      ifelse ((hour mod 24) >= leave-time and (hour mod 24) <= (return-time + 12))
      [
        ask people with [not hospitalized? and not quarantined?]
        [
          forward mobility                                         ; move ahead
          if distance homebase > commute [ face homebase ]            ; if too far from home, head back
          if (any? patches in-radius 1 with [pxcor = max-pxcor or pxcor = min-pxcor or pycor = min-pycor or pycor = max-pycor]) [face homebase]
          set heading heading + (random-float 5 - random-float 5)
        ]
      ]
      [ask people with [not hospitalized? and not quarantined?]
        [
          ifelse patch-here = homebase [set heading random-float 360 forward 0] [face homebase forward mobility]
        ]
      ]                       ; once it is not commute time, ask people to return to homes and stay there until next day
    ]
    housing? and not lockdown?
    [
      ask people with [not hospitalized? and not quarantined?]
      [
        forward mobility                                         ; move ahead
        if distance homebase > commute [ face homebase ]            ; if too far from home, head back
        if (any? patches in-radius 1 with [pxcor = max-pxcor or pxcor = min-pxcor or pycor = min-pycor or pycor = max-pycor]) [face homebase]
        set heading heading + (random-float 5 - random-float 5)
      ]
    ]
    not housing? and lockdown?
    [
      ifelse ((hour mod 24) >= leave-time and (hour mod 24) <= (return-time + 12))
      [
        ask people with [not hospitalized? and not quarantined?]
        [forward mobility]
      ]
      [ask people with [not hospitalized? and not quarantined?]
        [forward 0]
      ]
    ]
    not housing? and not lockdown?
    [
      ask people with [not hospitalized? and not quarantined?]
      [forward mobility]
    ]
  )
end

to random-testing
  if testing?
  [
    let target turtle-set people with [not tested-positive?]
    ask n-of (count target * population-tested-percent) target
    [
      if (ticks mod (round(ticks-day / test-interval)) = 0 and ticks > 0)
      [
        set tested? true

        if infected?
        [
          ifelse (random-float 1 < false-negative-rate)
          [set tested-positive? false]
          [
            set tested-positive? true
            set n-confirmed n-confirmed + 1
          ]
        ]

        if not infected?
        [
          ifelse (random-float 1 < false-positive-rate)
          [
            set tested-positive? true
            set n-confirmed n-confirmed + 1
          ]
          [set tested-positive? false]
        ]
      ]
    ]
  ]
end

to quarantine-positive-tested
  if testing? and housing?
  [
    ask people with [tested? and tested-positive? and not hospitalized?]
    [
      if (random 100 < quarantine-effort)
      [set quarantined? true
        ifelse patch-here = homebase [forward 0] [face homebase forward mobility]
      ]
    ]
  ]
end

to vaccinate
  if vaccination?
  [
    let vaccinate-priority-people turtle-set people

    (ifelse vaccine-priority = "No Priority"
      [set vaccinate-priority-people turtle-set people with [not vaccinated? and not infected? and not immune? and not hospitalized? and not antivaxxer?]]

      vaccine-priority = "Ages 0-20"
      [set vaccinate-priority-people turtle-set people with [not vaccinated? and not infected? and not immune? and not hospitalized? and not antivaxxer? and age < 20]]

      vaccine-priority = "Ages 20-50"
      [set vaccinate-priority-people turtle-set people with [not vaccinated? and not infected? and not immune? and not hospitalized? and not antivaxxer? and age > 20 and age < 50]]

      vaccine-priority = "Ages 50 and up"
      [set vaccinate-priority-people turtle-set people with [not vaccinated? and not infected? and not immune? and not hospitalized? and not antivaxxer? and age >= 50]]
    )

    let vaccine-ready-people turtle-set people with [not vaccinated? and not infected? and not immune? and not hospitalized? and not antivaxxer?]
    let n count vaccine-ready-people
    let x count vaccinate-priority-people

    if (x = 0)
    [
      set vaccinate-priority-people turtle-set vaccine-ready-people
      set x n
    ]

    (ifelse dose-amount = 1
      [
        if (ticks mod (round(ticks-day / vaccine-interval)) = 0 and ticks > 0)
        [
          ;let n count vaccinate-people
          ifelse ((vaccinated-proportion * .01 * n) <= x)
          [
            ask (n-of (vaccinated-proportion * .01 * n) vaccinate-priority-people)
            [
              set number-vaccinated-once number-vaccinated-once + 1
              set vaccinated? true
              set one-dose? true
              set color green

              if (random-float 100 < one-dose-success)
              [
                set susceptible? false
                set immune? true
              ]
            ]
          ]
          [
            ask (vaccinate-priority-people)
            [
              set number-vaccinated-once number-vaccinated-once + 1
              set vaccinated? true
              set one-dose? true
              set color green

              if (random-float 100 < one-dose-success)
              [
                set susceptible? false
                set immune? true
              ]
            ]
          ]
        ]

        ask people with [one-dose?]
        [
          set dose-counter dose-counter + 1

          ifelse (dose-counter / ticks-day) > 14
          [
            set one-dose-success one-dose-success-after-14-days
          ]
          [
            set one-dose-success one-dose-success-first-14-days
          ]
        ]
      ]

      dose-amount = 2
      [
        if (ticks mod (round(ticks-day / vaccine-interval)) = 0 and ticks > 0)[
          ;let n count vaccinate-people
          ifelse ((vaccinated-proportion * .01 * n) <= x)
          [
            ask (n-of (vaccinated-proportion * .01 * n) vaccinate-priority-people)
            [
              set number-vaccinated-once number-vaccinated-once + 1
              set vaccinated? true
              set one-dose? true
              set color green

              if (random-float 100 < one-dose-success)
              [
                set susceptible? false
                set immune? true
              ]
            ]
          ]
          [
            ask (vaccinate-priority-people)
            [
              set number-vaccinated-once number-vaccinated-once + 1
              set vaccinated? true
              set one-dose? true
              set color green

              if (random-float 100 < one-dose-success)
              [
                set susceptible? false
                set immune? true
              ]
            ]
          ]
        ]

        let m people with [one-dose? and not two-dose?]
        let two count m
        ask n-of two m
        [
          set dose-counter dose-counter + 1

          ifelse (dose-counter / ticks-day) > 14
          [
            set one-dose-success one-dose-success-after-14-days
          ]
          [
            set one-dose-success one-dose-success-first-14-days
          ]
        ]

        ask (n-of (vaccine-completion * .01 * two) m)
        [
          if (dose-counter / ticks-day) > dose-wait-time
          [
            if (random-float 100 < two-dose-success)
            [
              set number-vaccinated-twice number-vaccinated-twice + 1
              set immune? true
              set vaccinated? true
              set susceptible? false
              set color green
              set two-dose? true
            ]
          ]
        ]
      ]
    )
  ]
end

to infect-susceptibles
  ;S -> I
  ask people with [susceptible?]
  [
    let infected-neighbors (count other people with [infected?] in-radius (0.12 - (0.06 * social-distancing-effort / 100)) )

    let mask-effect 1

    if (masked?)
    [
      set mask-effect (1 - (mask-effectiveness * 0.01))
    ]

    if (random-float 1 <  (1 - (1 - (infection-rate * mask-effect)) ^ infected-neighbors) and not immune? and not hospitalized? and not quarantined?)
    [
      set susceptible? false
      set infected? true
      set color red
      set n-cases n-cases + 1
    ]
  ]
end

to illness
  ask people with [infected? or (tested? and tested-positive?)]
  [
    set sick-time sick-time + 1
  ]

  ask people with [infected?]
  [

    ;I -> H
    if (not tested? and not hospitalized? and int(sick-time / ticks-day) > 5) or
       (tested? and tested-positive? and not hospitalized? and not vaccinated?)
    [
      if (hospital-beds-occupied < healthcare-capacity and age >= 50 and income > middle-class-income-start )
      [
        set hospitalized? true
        set color yellow
        set hospital-beds-occupied hospital-beds-occupied + 1
        set chance-of-recovery chance-of-recovery * 2

        if (chance-of-recovery > 100)
        [set chance-of-recovery 100]
      ]
    ]

    ;H -> Die during sickness
    if (int(sick-time / ticks-day) < duration and hospitalized?)
    [
      if (random-float 100 < ((infected-mortality / 2) / (duration * ticks-day)))
      [
        set dead-total-age dead-total-age + age
        set dead-total-gdp dead-total-gdp + income
        die
        set hospital-beds-occupied hospital-beds-occupied - 1
      ]
    ]

    ;I -> Die during sickness
    if (int(sick-time / ticks-day) < duration and not hospitalized?)
    [
      if (random-float 100 < ((infected-mortality) / (duration * ticks-day)))
      [
        set dead-total-age dead-total-age + age
        set dead-total-gdp dead-total-gdp + income
        die
        set hospital-beds-occupied hospital-beds-occupied - 1
      ]
    ]

    ;Sickness duration complete
    if (int(sick-time / ticks-day) >= duration)
    [
      ;I -> R
      ifelse (random-float 100 < chance-of-recovery)
      [
        if hospitalized?
        [
          set hospitalized? false
          set hospital-beds-occupied hospital-beds-occupied - 1
        ]

        set infected? false
        set immune? true
        set sick-time 0
        set susceptible? false
        set color gray
      ]

      ;I -> Die after sickness duration is complete
      [
        if hospitalized?
        [
          set hospitalized? false
          set hospital-beds-occupied hospital-beds-occupied - 1
          set dead-total-age dead-total-age + age
          set dead-total-gdp dead-total-gdp + income
          die
        ]
        set dead-total-age dead-total-age + age
        set dead-total-gdp dead-total-gdp + income
        die
      ]
    ]
  ]
end

to end-quarantine
  ask people with [quarantined?]
  [
    if (int(sick-time / ticks-day) >= duration)
    [set quarantined? false]
  ]

end

to clock                                    ; ticks-day refers to the total number of ticks in one day
  set day int (ticks / ticks-day)           ; track of number of days elapsed since beginning
  set hour int ((ticks / ticks-day) * 24)   ; track of number of hours elapsed since beginning
end

to-report prop-infected
  report ((count people with [infected?]) / initial-population)
end

to-report prop-uninfected
  report 1 - ((count people with [infected?]) / initial-population)
end

to-report num-dead
  report (initial-population - count people)
end

to-report prop-dead
  report (initial-population - count people) / initial-population
end

to-report population
  report count people
end

to-report immune
  report count people with [immune?]
end

to-report infected
  report count people with [infected?]
end

to-report hospitalized-proportion
  report count people with [hospitalized?]
end

to-report cumulative-infected
  report (n-cases + initial-infected-rich + initial-infected-middle + initial-infected-poor) / initial-population
end

to-report prop-hospital-beds-occupied
  report hospital-beds-occupied / healthcare-capacity
end

to-report dead-avg-age
  report (dead-total-age / (initial-population - count people))
end

to-report dead-avg-gdp
  report (dead-total-gdp / (initial-population - count people))
end

to-report total-cases
  report n-cases + initial-infected-rich + initial-infected-middle + initial-infected-poor
end

to-report confirmed-cases
  report n-confirmed
end

to-report total-vaccinated
  report count people with [vaccinated?]
end

to-report recieved-two-doses
  report count people with [two-dose?]
end

to-report recieved-one-dose
  report count people with [one-dose? and not two-dose?]
end
@#$#@#$#@
GRAPHICS-WINDOW
211
10
543
343
-1
-1
6.353
1
10
1
1
1
0
0
0
1
-25
25
-25
25
0
0
1
ticks
30.0

BUTTON
47
22
114
56
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
120
22
184
56
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
30
238
201
271
initial-population
initial-population
10
3000
300.0
1
1
NIL
HORIZONTAL

SLIDER
503
579
676
612
initial-infected-rich
initial-infected-rich
0
20
5.0
1
1
NIL
HORIZONTAL

SLIDER
29
134
202
167
transmission-rate
transmission-rate
0
1
0.7
.01
1
NIL
HORIZONTAL

PLOT
546
10
913
211
SIRV Plot
Time
Proportion
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Infected" 1.0 0 -2674135 true "" "plot (count people with [infected?]) / (count people)"
"Susceptible" 1.0 0 -13345367 true "" "plot (count people with [susceptible?]) / (count people)"
"Immune" 1.0 0 -7500403 true "" "plot (count people with [immune?]) / (count people)"
"Vaccinated" 1.0 0 -13840069 true "" "plot (count people with [vaccinated?]) / (count people)"
"Dead" 1.0 0 -955883 true "" "plot 1 - (population / initial-population)"
"Tested" 1.0 0 -6459832 true "" "plot (count people with [tested?]) / (count people)"

MONITOR
1040
65
1222
110
Current Proportion of Infected
prop-infected
2
1
11

MONITOR
933
118
1036
163
Number of Dead
num-dead
17
1
11

MONITOR
943
216
1085
261
Current Population Size
population
17
1
11

SLIDER
29
278
202
311
healthcare-capacity
healthcare-capacity
0
20
10.0
1
1
NIL
HORIZONTAL

SLIDER
28
315
209
348
average-household-income
average-household-income
0
150000
50577.0
1
1
NIL
HORIZONTAL

SLIDER
35
417
207
450
percent-age-ten
percent-age-ten
0
100
12.0
1
1
NIL
HORIZONTAL

SLIDER
35
455
212
488
percent-age-twenty
percent-age-twenty
0
100
25.0
1
1
NIL
HORIZONTAL

SLIDER
34
495
206
528
percent-age-thirty
percent-age-thirty
0
100
39.0
1
1
NIL
HORIZONTAL

SLIDER
34
536
206
569
percent-age-forty
percent-age-forty
0
100
52.0
1
1
NIL
HORIZONTAL

SLIDER
218
417
390
450
percent-age-fifty
percent-age-fifty
0
100
64.0
1
1
NIL
HORIZONTAL

SLIDER
217
456
389
489
percent-age-sixty
percent-age-sixty
0
100
77.0
1
1
NIL
HORIZONTAL

SLIDER
218
496
390
529
percent-age-seventy
percent-age-seventy
0
100
88.0
1
1
NIL
HORIZONTAL

SLIDER
218
536
392
569
percent-age-eighty
percent-age-eighty
0
100
95.0
1
1
NIL
HORIZONTAL

TEXTBOX
32
217
182
235
Demographics Factors
11
0.0
0

TEXTBOX
35
394
185
412
Cumulative Age Distribution
11
0.0
1

SLIDER
30
173
202
206
duration
duration
0
100
14.0
1
1
NIL
HORIZONTAL

MONITOR
941
265
1143
310
Number of Occupied Hospital Beds
hospital-beds-occupied
17
1
11

MONITOR
932
65
1029
110
Current Cases
infected
17
1
11

MONITOR
1149
265
1307
310
Cumulative Immune Cases
immune
17
1
11

MONITOR
1047
118
1178
163
Proportion of Dead
prop-dead
2
1
11

MONITOR
934
166
1034
211
Dead Avg Age
dead-avg-age
2
1
11

PLOT
741
218
915
353
Number of Occupied Hospital Beds
time
beds
0.0
10.0
0.0
20.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot hospitalized-proportion"

TEXTBOX
33
111
183
129
Virus Qualities
11
0.0
1

TEXTBOX
413
341
563
359
Intervention Policies
11
0.0
1

SLIDER
411
505
588
538
social-distancing-effort
social-distancing-effort
0
100
75.0
1
1
NIL
HORIZONTAL

SLIDER
30
355
202
388
income-spread
income-spread
0
150000
40000.0
1
1
NIL
HORIZONTAL

MONITOR
1041
165
1161
210
Dead Avg Income
dead-avg-gdp
2
1
11

MONITOR
933
14
1011
59
Total Cases
total-cases
17
1
11

SLIDER
749
367
976
400
one-dose-success-first-14-days
one-dose-success-first-14-days
0
100
50.0
1
1
NIL
HORIZONTAL

SWITCH
600
360
724
393
vaccination?
vaccination?
0
1
-1000

SLIDER
750
491
924
524
vaccinated-proportion
vaccinated-proportion
0
100
1.0
1
1
NIL
HORIZONTAL

MONITOR
1019
13
1220
58
Cumulative Proportion of Infected
cumulative-infected
2
1
11

CHOOSER
600
395
738
440
dose-amount
dose-amount
1 2
1

SLIDER
750
530
925
563
vaccine-completion
vaccine-completion
0
100
20.0
1
1
NIL
HORIZONTAL

SLIDER
749
453
921
486
two-dose-success
two-dose-success
0
100
94.0
1
1
NIL
HORIZONTAL

SLIDER
46
66
138
99
ticks-day
ticks-day
0
2400
48.0
1
1
NIL
HORIZONTAL

MONITOR
142
64
199
109
Clock:
(word day \"d, \" (hour mod 24) \"h\")
17
1
11

SLIDER
600
445
740
478
dose-wait-time
dose-wait-time
0
30
28.0
1
1
NIL
HORIZONTAL

SLIDER
750
569
950
602
anti-vaccination-proportion
anti-vaccination-proportion
0
100
50.0
1
1
NIL
HORIZONTAL

SWITCH
222
346
326
379
housing?
housing?
0
1
-1000

CHOOSER
600
485
738
530
vaccine-priority
vaccine-priority
"No Priority" "Ages 0-20" "Ages 20-50" "Ages 50 and up"
3

MONITOR
939
315
1053
360
Total Vaccinated
total-vaccinated
17
1
11

CHOOSER
601
534
739
579
distribution-interval
distribution-interval
"Every Hour" "Once a Day" "Once a Week"
1

CHOOSER
410
453
548
498
lockdown-end
lockdown-end
"1:00 AM" "2:00 AM" "3:00 AM" "4:00 AM" "5:00 AM" "6:00 AM" "7:00 AM" "8:00 AM" "9:00 AM" "10:00 AM" "11:00 AM" "12:00 PM"
6

CHOOSER
411
401
549
446
lockdown-start
lockdown-start
"1:00 PM" "2:00 PM" "3:00 PM" "4:00 PM" "5:00 PM" "6:00 PM" "7:00 PM" "8:00 PM" "9:00 PM" "10:00 PM" "11:00 PM" "12:00 AM"
7

SWITCH
412
361
533
394
lockdown?
lockdown?
1
1
-1000

SLIDER
944
442
1144
475
population-tested-percent
population-tested-percent
0
1
0.03
.01
1
NIL
HORIZONTAL

SLIDER
946
518
1118
551
false-positive-rate
false-positive-rate
0
1
0.03
.01
1
NIL
HORIZONTAL

SLIDER
945
483
1117
516
false-negative-rate
false-negative-rate
0
1
0.03
.01
1
NIL
HORIZONTAL

SLIDER
410
543
582
576
quarantine-effort
quarantine-effort
0
100
75.0
1
1
NIL
HORIZONTAL

CHOOSER
945
553
1097
598
testing-interval
testing-interval
"Every Hour" "Once a Day" "Once a Week"
1

SWITCH
983
406
1088
439
testing?
testing?
0
1
-1000

SWITCH
1156
442
1259
475
masks?
masks?
0
1
-1000

SLIDER
1152
479
1289
512
mask-effort
mask-effort
0
100
75.0
1
1
NIL
HORIZONTAL

TEXTBOX
1078
352
1228
370
NIL
11
0.0
1

MONITOR
1059
359
1268
404
Number of People Vaccinated Twice
recieved-two-doses
17
1
11

MONITOR
1057
313
1263
358
Number of People Vaccinated Once
recieved-one-dose
17
1
11

PLOT
551
217
733
353
Vaccine Progress
time
Number Vaccinated
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"One Dose" 1.0 0 -8990512 true "" "plot (count people with [one-dose? and not two-dose?]) / (count people)"
"Two Doses" 1.0 0 -5825686 true "" "plot (count people with [two-dose?]) / (count people)"

SLIDER
748
405
979
438
one-dose-success-after-14-days
one-dose-success-after-14-days
0
100
92.0
1
1
NIL
HORIZONTAL

SLIDER
1150
520
1300
553
mask-effectiveness
mask-effectiveness
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
32
612
284
645
middle-class-income-end
middle-class-income-end
0
200000
120000.0
1
1
NIL
HORIZONTAL

SLIDER
32
575
266
608
middle-class-income-start
middle-class-income-start
0
100000
40000.0
1
1
NIL
HORIZONTAL

SLIDER
292
576
464
609
rich-household-size
rich-household-size
0
5
2.0
1
1
NIL
HORIZONTAL

SLIDER
292
610
494
643
middle-class-household-size
middle-class-household-size
0
5
3.0
1
1
NIL
HORIZONTAL

SLIDER
292
645
464
678
poor-household-size
poor-household-size
0
5
4.0
1
1
NIL
HORIZONTAL

SWITCH
223
381
386
414
income-separation?
income-separation?
0
1
-1000

SLIDER
504
615
676
648
initial-infected-middle
initial-infected-middle
0
20
5.0
1
1
NIL
HORIZONTAL

SLIDER
504
649
676
682
initial-infected-poor
initial-infected-poor
0
20
5.0
1
1
NIL
HORIZONTAL

SLIDER
685
650
857
683
commute-distance
commute-distance
0
20
10.0
1
1
NIL
HORIZONTAL

MONITOR
1095
215
1201
260
Confirmed Cases
n-confirmed
17
1
11

@#$#@#$#@
## WHAT IS IT?

This COVID-19 Model is an Agent-Based-Model looking at how different demographic factors and policy interventions impact COVID-19 transmission and health outcomes.

## HOW IT WORKS

An initial population of agents (blue-colored humanoids) are randomly placed in the model space with an initial population of infected individuals (red). As time moves forward, agents move randomly through the model space according to specified parameters such as number of stationary individuals and mobility. Infected individuals transmit the virus to susceptible individuals by coming within a certain distance of each other. Whether the susceptible individual becomes infected is determined by a random probability, the likelihood of which increases as transmission rate increases. The model can be run with and without immune individuals; when ran with immunity, infected individuals will become immune (color changes to gray) according to a random probability, the liklihood of which increases with increasing recovery rate. Mortality rate can be adjusted. The ability of hospitals to cope with the proportion of infected individuals can be adjusted as well. Once the proportion of infected individuals is greater than _health care capacity_ mortality increases an order of magnitude, as predicted by other current models. 

## HOW TO USE IT

After adjusting the parameters, described below, simply click the _setup_ button and click _go_. The model will continue to run until there are either no more infected individuals or no more susceptible individuals.


## THINGS TO NOTICE



## THINGS TO TRY



## EXTENDING THE MODEL

If you have any suggestions for things to add or change in the model feel free to contact me at jonathan.huang@richmond.edu.

## RELATED MODELS

This model was inspired from these models:

SIR Model with random movement
by Paul Smaldino
http://smaldino.com/wp/

COVID-19 epidemics with Non-Pharmaceutical Interventions and zonal restraints 
by Sergio Rojas-Galeano and Lindsay Alvarez
http://modelingcommons.org/browse/one_model/6374#model_tabs_browse_info

Many Regions Example
by Nicolas Payette
http://www.netlogoweb.org/launch#http://ccl.northwestern.edu/netlogo/models/models/Code%20Examples/Many%20Regions%20Example.nlogo


## CREDITS AND REFERENCES

CREATIVE COMMONS LICENSE
This code is distributed by Nich Martin under a Creative Commons License:
Attribution-ShareAlike 4.0 International (CC 4.0)
https://creativecommons.org/licenses/by/4.0/
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="average_community" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="above_average_community" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="70577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="15"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="below_average_community" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="30577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="30000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="not_testing" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="testing_part1" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="testing_part2" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="testing_part3" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="testing_part4" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="testing_part5" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="testing_part6" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="testing_part7" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="high-anti-vax_low_vax_completion_part1" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="high-anti-vax_low_vax_completion_part2" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="high-anti-vax_low_vax_completion_part3" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="high-anti-vax_low_vax_completion_part4" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="high-anti-vax_low_vax_completion_part5" repetitions="500" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-cases</metric>
    <metric>cumulative-infected</metric>
    <metric>num-dead</metric>
    <metric>prop-dead</metric>
    <metric>dead-avg-age</metric>
    <metric>dead-avg-gdp</metric>
    <metric>immune</metric>
    <metric>total-vaccinated</metric>
    <metric>n-confirmed</metric>
    <enumeratedValueSet variable="testing-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rich-household-size">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-start">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-forty">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-tested-percent">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-fifty">
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-population">
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-negative-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="duration">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-sixty">
      <value value="77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-middle">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transmission-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-amount">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dose-wait-time">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-priority">
      <value value="&quot;Ages 50 and up&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quarantine-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-seventy">
      <value value="88"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-end">
      <value value="&quot;7:00 AM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-first-14-days">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="poor-household-size">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-poor">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-household-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="false-positive-rate">
      <value value="0.03"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="anti-vaccination-proportion">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown-start">
      <value value="&quot;8:00 PM&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="commute-distance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="two-dose-success">
      <value value="94"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="one-dose-success-after-14-days">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ticks-day">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-effort">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccine-completion">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-household-income">
      <value value="50577"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-thirty">
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected-rich">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-ten">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-twenty">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percent-age-eighty">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccinated-proportion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-separation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="distribution-interval">
      <value value="&quot;Once a Day&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-spread">
      <value value="25000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-class-income-end">
      <value value="120000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="healthcare-capacity">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
