globals [
  max-percent                                    ;; most numerous color turtles in %
  max-color                                      ;; color of most numerous
  pop-removed-ticks                              ;; ticks since pop removal
  alive-colors                                   ;; # of colors alive
  wall-list                                      ;; list of wall coordinates
]

turtles-own [age]                                ;; turtles' age

to setup
  clear-commands
  populate-world                                 ;; create turtles
  ask turtles [ set shape "circle" ]             ;; set turtles to circle shape
  find-top-species                               ;; find percentage of most numerous species
  set alive-colors [5 15 25 35 45 55 65 85 95 125]
  count-alive-colors                             ;; count alive colors after setting list to all by default
  remove-all-walls
  set mutation-rate 0.0
  reset-ticks
  setup-plots
  output-print( word "[New simulation]" )
end

to clear-commands
  clear-patches
  clear-turtles
  clear-all-plots
  clear-ticks
end

to populate-world
  create-turtles population
  [
    set color 5 + (random colors) * 10
    if color = turquoise                         ;; turquoise (75) is too similar to another color
      [ set color magenta ]                      ;; so use magenta (125) instead
    setxy random-xcor random-ycor
  ]
end

to go
  if stop-run? [                                 ;; if fixation = true stop
    output-print (word "Fixation:" ticks " gen, mut rate:" mutation-rate ", pop:" population ", colors:" colors )
    stop
  ]
  ask turtles [                                  ;; move turtles
    rt random 50
    lt random 50
    ifelse [ pcolor ] of patch-ahead 1 = black   ;; if wall ahead change direction
      [ fd 0.2 ]                                 ;; Move forward 0.2 (not 1 to slow down)
    [ bk 0.5 rt random 180 ] ;
  ]
  if ticks - pop-removed-ticks > 10 [            ;; if 10 ticks from removal clear death patches color
    ask patches with  [pcolor = violet ] [
      set pcolor black
    ]
  ]
  birth
  grow-old
  ifelse death-cause = "random-death"
  [ random-death ]
  [ death-by-age ]
  find-top-species
  count-alive-colors
  tick
end

;; Get the string (name) of a color
to-report translate-color [num]
  if num = 5 [report "grey"]
  if num = 15 [report "red"]
  if num = 25 [report "orange"]
  if num = 35 [report "brown"]
  if num = 45 [report "yellow"]
  if num = 55 [report "green"]
  if num = 65 [report "lime"]
  if num = 75 [report "turquoise"]
  if num = 85 [report "cyan"]
  if num = 95 [report "sky"]
  if num = 105 [report "blue"]
  if num = 115 [report "violet"]
  if num = 125 [report "magenta"]
  ifelse num = 135
    [ report "pink" ]
    [ report "unknown color" ]
end


to birth
  ask turtles [
    fd 0.3 hatch (random max-offspring + 1) [    ;; turtles hatch between 0 and max-offspring babies and move forward 0.3
      if(random 1000) < (mutation-rate * 1000)   ;; random mutation given rate
      [ set color one-of alive-colors ]
      set age 0
    ]
  ]
end

;; if the overall population is greater than the original number,
;; randomly kill turtles to keep population roughly constant
to random-death
  let total-turtles count turtles
  ask turtles [
    if random total-turtles > population
      [ die ]
  ]
end

to grow-old
  ask turtles [
    set age age + 1                              ;; turtles age increases by 1 every tick
  ]
end

to death-by-age                                  ;; kill (randomly) older turtles first
  let list-turtles sort-on [age] turtles
  while [count turtles > population] [
    ask last list-turtles [
      die]
    set list-turtles remove-item (length list-turtles - 1) list-turtles

  ]
end

to remove-pop
  let num-turtles count turtles                  ;; count current number of turtles
  let xcoor random world-width - max-pxcor       ;; select random x coordinate
  let ycoor random world-height - max-pycor      ;; select random y coordinate
  ask patches with [ pxcor = xcoor and pycor = ycoor ] [
    let radius (10 + random 10)
    ask patches in-radius radius [
      if pcolor = black [
        set pcolor violet                        ;; set death area to violet
      ]
    ]
    ask turtles in-radius radius [die]           ;; make turtles in area die
  ]
  let killed-turtles (num-turtles - count turtles) / num-turtles * 100
  output-print (word "Removed " round (killed-turtles) "% of turtles at generation " ticks ".")
  set pop-removed-ticks ticks
end

;; find the percentage of the most populous species
to find-top-species
  let winning-amount 0
  foreach base-colors [ c ->
    let how-many count turtles with [color = c]
    if how-many > winning-amount
      [ set winning-amount how-many
        set max-color translate-color c
      ]
  ]
  set max-percent (100 * winning-amount / count turtles)
end

to count-alive-colors
  foreach alive-colors [ c ->
    if count turtles with [color = c] = 0 [      ;; if # of turtles of that color = 0 remove color from list
      set alive-colors remove c alive-colors
    ]
  ]
end

to-report stop-run?
  let stop-run (list)
  let i 0
  let colors-in-area (list)
  while [i < length wall-list - 1]
    ;; get turtles between each set of walls (area)
    [ ask turtles with [ycor > item i wall-list and ycor < item (i + 1) wall-list ]
      [
        ;; get color of all turtles in the area
        set colors-in-area insert-item 0 colors-in-area color
      ]
      if length colors-in-area > 1 [
        ;; if there are more than 1 color in area add false to list
        ifelse variance colors-in-area = 0.0
          [ set stop-run insert-item 0 stop-run true ]
          [ set stop-run insert-item 0 stop-run false ]
      ]
      ;; reset counters
      set colors-in-area (list)
      set i (i + 1)
    ]
  ;; if list has one or more 'false' values keep running simulation
  ifelse member? false stop-run
  [ report false ]
  ;; else stop simulation = true
  [ report true ]
end

;; ---------------------------------------------------------------------------------
;; Below this point are procedure definitions that have to do with "walls," which
;; the user may create in order to separate groups of turtles from one another.
;; The use of walls is optional, and can be seen as a more advanced topic.
;; ---------------------------------------------------------------------------------

to place-wall
  if mouse-down? [
    ;; Note that when we place a wall, we must also place walls
    ;; at the world boundaries, so turtles can't change rooms
    ;; by wrapping around the edge of the world.
    ask patches with [abs pycor = max-pycor or
                      pycor = round mouse-ycor] [
      ;set pcolor white
      set pcolor white
      ;; There might be some turtles standing where the
      ;; new walls is, so we need to move them into a room.
      ask turtles-here [ move-off-wall ]
    ]
    set wall-list insert-item 1 wall-list round mouse-ycor
    set wall-list remove-duplicates wall-list
    set wall-list sort wall-list
    display
  ]
end

to remove-wall
  if mouse-down? [
    ask patches with [pycor = round mouse-ycor]
      [ set pcolor black ]
    ;; if only 1 wall in wall-list remove-all (+ boundaries)
    if length wall-list = 3 [
      remove-all-walls
    ]
    ;; remove wall from list
    set wall-list remove round mouse-ycor wall-list
    display
  ]
end

to remove-all-walls
  clear-patches
  set wall-list (list min-pycor max-pycor)
end

to move-off-wall  ;; turtle procedure
  while [pcolor != black] [
    move-to one-of neighbors
  ]
end


; Copyright 1997 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
201
10
621
431
-1
-1
9.16
1
10
1
1
1
0
1
1
1
-22
22
-22
22
1
1
1
ticks
30.0

BUTTON
105
22
183
55
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
5
22
89
55
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
6
64
184
97
colors
colors
2
10
4.0
1
1
NIL
HORIZONTAL

SLIDER
7
102
184
135
population
population
20
1000
510.0
10
1
NIL
HORIZONTAL

PLOT
624
10
949
280
Turtle Populations
Time
Population
0.0
50.0
0.0
50.0
true
false
"set-plot-y-range 0 count turtles" ""
PENS
"color5" 1.0 0 -7500403 true "" "plot count turtles with [color = 5]"
"color15" 1.0 0 -2674135 true "" "plot count turtles with [color = 15]"
"color25" 1.0 0 -955883 true "" "plot count turtles with [color = 25]"
"color35" 1.0 0 -6459832 true "" "plot count turtles with [color = 35]"
"color45" 1.0 0 -1184463 true "" "plot count turtles with [color = 45]"
"color55" 1.0 0 -10899396 true "" "plot count turtles with [color = 55]"
"color65" 1.0 0 -13840069 true "" "plot count turtles with [color = 65]"
"color125" 1.0 0 -5825686 true "" "plot count turtles with [color = 125]"
"color85" 1.0 0 -11221820 true "" "plot count turtles with [color = 85]"
"color95" 1.0 0 -13791810 true "" "plot count turtles with [color = 95]"

OUTPUT
202
433
622
588
12

BUTTON
7
171
95
204
NIL
place-wall
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
100
171
192
204
NIL
remove-wall
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6
210
192
243
NIL
remove-all-walls
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
7
278
192
311
NIL
remove-pop
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
654
29
756
74
Most numerous
word max-color \": \" round max-percent \"%\"
2
1
11

BUTTON
9
546
191
579
NIL
clear-output
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
625
283
951
588
Turtle Population per Color
Color
Population
0.0
2.0
0.0
10.0
false
true
"" "clear-plot\nset-plot-x-range 0 colors\nset-plot-y-range 0 population\nset-histogram-num-bars colors"
PENS
"color5" 1.0 1 -7500403 true "" "foreach (range 0 (count turtles with [color = 5]) 0.05) [a -> plotxy 0 a ]"
"color15" 1.0 1 -2674135 true "" "foreach (range 0 (count turtles with [color = 15]) 0.05) [a -> plotxy 1 a ]"
"color25" 1.0 1 -955883 true "" "foreach (range 0 (count turtles with [color = 25]) 0.05) [a -> plotxy 2 a ]"
"color35" 1.0 1 -6459832 true "" "foreach (range 0 (count turtles with [color = 35]) 0.05) [a -> plotxy 3 a ]"
"color45" 1.0 1 -1184463 true "" "foreach (range 0 (count turtles with [color = 45]) 0.05) [a -> plotxy 4 a ]"
"color55" 1.0 1 -10899396 true "" "foreach (range 0 (count turtles with [color = 55]) 0.05) [a -> plotxy 5 a ]"
"color65" 1.0 1 -13840069 true "" "foreach (range 0 (count turtles with [color = 65]) 0.05) [a -> plotxy 6 a ]"
"color125" 1.0 1 -5825686 true "" "foreach (range 0 (count turtles with [color = 125]) 0.05) [a -> plotxy 7 a ]"
"color85" 1.0 1 -11221820 true "" "foreach (range 0 (count turtles with [color = 85]) 0.05) [a -> plotxy 8 a ]"
"color95" 1.0 1 -13791810 true "" "foreach (range 0 (count turtles with [color = 95]) 0.05) [a -> plotxy 9 a ]"

TEXTBOX
11
328
184
349
Mutation rate [0%,5%]
12
0.0
1

TEXTBOX
16
147
166
165
Walls
12
0.0
1

TEXTBOX
7
257
157
275
Bottle-neck effect
12
0.0
1

SLIDER
8
346
192
379
mutation-rate
mutation-rate
0.0
0.05
0.0
0.001
1
NIL
HORIZONTAL

CHOOSER
8
404
190
449
death-cause
death-cause
"random-death" "death-by-age"
1

SLIDER
9
465
189
498
max-offspring
max-offspring
1
5
2.0
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model is an example of genetic drift. It shows that competing organisms, turtles of different colors, each with equal likelihood reproduction, will ultimately converge on one breed without any particular selection pressure.

## HOW IT WORKS

The model starts with a random distribution of colored turtles. They move by wiggling randomly across the world. Each turn, a turtle produces between 0 and 5 offspring, value that can be modified by the user.  If the total number of turtles is greater than the original number, then turtles are randomly killed until the original number is restored. This can be achieved by randomly kill considering all turtles or by killing the older turtles first. When only one breed of turtles is alive it is said that the breed as come to fixation and the simulation ends.

## HOW TO USE IT

* The "setup" button initializes the model.
* The "go" button runs the model.
* Use the "colors" slider to select the number of competing colors (breeds).
* The "number" slider sets the number of turtles.
* Use the "walls" buttons to divide the world.
* Use the "remove-pop" button to randomly kill turtles and simulate bottle-neck effect.
* The "mutation-rate" buttons adds a turtle birth mutation rate.
* The "death-cause" chooser determines death cause when population exceeds "population" slider value.
* The "max-offspring" slider sets the maximum number of offspring for each turtles on each generation.
* Use the "clear-output" to clear the message log screen.

## THINGS TO TRY

This models adds several features compared to other models. It is recommended to change things one at a time and notice how that changes the outcome of the simulation. Here is a list of possible things to do:

Experiment with adding walls. When walls are added, groups of individuals can be geographically isolated. Groups that are geographically isolated with walls will often end up with a different dominant color than the larger population.

Experiment with the two different options of death cause. Run a few simulations for each type. Did you notice any difference?

Looking at the graphs, use the "remove-pop" button to randomly remove turtles from the world. How does that affect each alive breed?

Change the population value and keep the number of colors constant. How does that affect the number of generations that it takes for a breed to go to fixation? Then keep the population constant and change the number of colors. What is the effect this time? How many generations does it take before a breed goes extinct?

When introducing mutations fixation can take many generations. Try selecting continuous update instead of by tick. Furthermore, try unselecting the "view update" option and focus on the plots.

Experiment with the bottle-neck effect. For a more extreme scenario click the "remove-pop" button two or three times at once.

## THINGS TO NOTICE

Notice that often colors can get to quite a high population but still fail to win the race. Also notice that when a breed dies it can not come back, not even when mutation rate is greater than zero.

## EXTENDING THE MODEL

Add-ons to the model by Carlo Maley were kept (e.g. organisms moving more slowly).

Juan Garcia Mesa added a mutation rate option and the ability for fixation in each area when the world is divided by walls.

The grim reaper in the procedure death does a random harvesting of the population to keep it roughly constant. This might be somewhat like a natural environment with a limited food supply. Juan Garcia Mesa added an option where older turtles die first to simulate aging. Can you think of other ways to write this procedure? Are the results affected?

## RELATED MODELS

* GenDrift P global
* GenDrift P local
* GenDrift T interact
* GenDrift T reproduce
* Genetic Drift by Carlo Maley

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Wilensky, U. (1997).  NetLogo GenDrift T reproduce model.  http://ccl.northwestern.edu/netlogo/models/GenDriftTreproduce.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 1997 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

This model was created as part of the project: CONNECTED MATHEMATICS: MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL MODELS (OBPML).  The project gratefully acknowledges the support of the National Science Foundation (Applications of Advanced Technologies Program) -- grant numbers RED #9552950 and REC #9632612.

This model was converted to NetLogo as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and/or INTEGRATED SIMULATION AND MODELING ENVIRONMENT. The project gratefully acknowledges the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227. Converted from StarLogoT to NetLogo, 2001.

<!-- 1997 2001 -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
