;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; GNU GENERAL PUBLIC LICENSE ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; LeLeLe
;; LeLeLe is a model designed to analyse
;; the effect of conditional dissociation
;; in the evolutionary emergence of cooperation.
;; Copyright (C) 2013 Segismundo S. Izquierdo & Luis R. Izquierdo
;; 
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;
;; Contact information:
;; Luis R. Izquierdo 
;;   University of Burgos, Spain. 
;;   e-mail: luis@izquierdo.name


;;;;;;;;;;;;;;;;;
;;; Variables ;;;
;;;;;;;;;;;;;;;;;


globals [
  
  ;; Variables for the agent-based model
 strategy-mode
 strategy-mode-%
  
  pool-of-singles ;; players without mate at the beginning of the period
  n-of-singles-at-the-beginning
  n-of-singles-selected-to-be-paired
  paired-players
  


  num-strategies                                  ;; 162
  strategy-frequencies
  strategy-names  

;  strategy-numbers

;  strategy-frequencies-pool-of-singles
;  strategy-frequencies-pool-of-singles-norm
;  strategy-avg-payoffs
;  strategy-mod-payoffs



  num-plots ;; 12
  plot-names
  plot-frequencies
  plot-numbers
  plot-avg-payoffs
  

  
  total-payoff
  avg-payoff
  paired
  
  
  Z  ;; temp variable

  
  
  numSS numSL numLL       ;; these variables store how many times each of the leave outcomes has been observed in one match
  %-SS %-SL %-LL
  num-outcomesS

  numCC numCD numDD       ;; these variables store how many times each of the outcomes has been observed in one match
  %-CC %-CD %-DD
  num-outcomes  
  
  
  ;; to be aded later
  %-A    ;; percentage of strategy
  %p-A   ;; percentage of strategy in the pool
  
  
  
  
]

breed [agents agent] 

agents-own [ 
  strategy-number  ;; strategies are numbered from 0,1,2,3,4,6,7...160,161
                   ;; floor(# /81)       is action for new    0=C, 1= D
                   ;; floor(# /27) mod 3 is action for CC     0=C, 1= D, 2= out
                   ;; floor(# / 9) mod 3 is action for CD     0=C, 1= D, 2= out
                   ;; floor(# / 3) mod 3 is action for DC     0=C, 1= D, 2= out
                   ;; floor(#    ) mod 3 is action for DD     0=C, 1= D, 2= out
  past          ;; LAST PD GAME 0=New, 1=CC, 2=CD, 3=DC, 4=DD
  action        ;; the action is either 0 (C) or 1 (D)
  break-if-C    ;; 0 if the player does not split up after the other player plays C. 1 otherwise.
  break-if-D    ;; 0 if the player does not split up after the other player plays D. 1 otherwise.
  
  name          ;; plot name on graph, A, B, C ...
  mate
  
  new-partnership?
  
  payoff

  plot-number       ;; strategies are numbered from 0 to num-plots-1
  
]

;; The following breeds are the variables for the mean dynamics


breed [singles single]

singles-own [
  value
  cooperate?    
  break-if-C?   
  break-if-D?   
  plot-number      ;; strategies are numbered from 0 to num-plots-1
  strategy-number  ;; strategies are numbered from 0 to num-strategies-1
]

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setup Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

;; the following procedure is called when the model is first loaded
to startup
  clear-all
  setup-variables
  reset-ticks
  setup-initial-conditions
  no-display
end

to setup-variables
  set num-strategies 162  
  set num-plots 12
  set plot-names ["TfTat" "OfTat" "C-Prt" "AllC" "OthC" "TfTit" "OfTit" "WSLS" "FGOF" "D-Prt" "AllD" "OthD"]
  set plot-numbers n-values num-plots [?]
end


to setup-initial-conditions
  create-agents num-players [
    set past 0
    set mate nobody
    set payoff 0    
    set hidden? true
  ]
  set pool-of-singles (agents with [mate = nobody])
  set n-of-singles-at-the-beginning (count pool-of-singles)
 
  ask agents [
    ifelse ((main-strategy < 0) or (main-strategy > 161)) [set strategy-number (floor(random-float 162.0))] 
                                                          [set strategy-number (floor(main-strategy))]
    update-strategy-variables 
             ]
  ask singles [
         let my-st-number strategy-number
         set value (count agents with [strategy-number = my-st-number]) / num-players
       ]
  
  ;; here we plot the initial conditions, before any matching occurs
  gather-data  
  
  update-graphs   
  
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Run-time procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go  
  
  make-couples

  play
    
  gather-data  

  tick  
  
  update-graphs   

  split-up 
 
  ;; new generation
  
  kill-and-breed
  
  if (count agents != num-players) [adjust-num-players]      
  
end


to make-couples
  ask agents [set new-partnership? false]
  set pool-of-singles (agents with [mate = nobody])
  set n-of-singles-at-the-beginning (count pool-of-singles)
  let players-to-be-paired (pool-of-singles with [random-float 1.0 < prob-rematch])
  set n-of-singles-selected-to-be-paired (count players-to-be-paired)
   
  if (n-of-singles-selected-to-be-paired mod 2 = 1) [
    set n-of-singles-selected-to-be-paired (n-of-singles-selected-to-be-paired - 1)
    set players-to-be-paired (n-of n-of-singles-selected-to-be-paired players-to-be-paired)
  ]  
  
  ask players-to-be-paired [set new-partnership? true]
  ask players-to-be-paired [
    if (mate = nobody) [  
      set mate one-of (players-to-be-paired with [mate = nobody and self != myself])
      ask mate [set mate myself]
    ]
  ] 
end

to play
  
  set total-payoff 0
  set paired 0
  set avg-payoff 0

  ask agents [
    set past ifelse-value (mate = nobody)
      [0]
      [past-for action ([action] of mate)] 
      
    set payoff ifelse-value (mate = nobody)
      [out-payoff]
      [payoff-for action ([action] of mate)]
      
    set total-payoff (total-payoff + payoff)
    set paired (paired + 1)
  ]
    set avg-payoff (total-payoff / paired )
end

to-report past-for [my-action her-action]
  ;; my-action is 0 if C, 1 if D
  report 
    ifelse-value (my-action = 0)
        [ifelse-value (her-action = 0)
          [1][2]]
        [ifelse-value (her-action = 0)
          [3][4]]
end

to-report payoff-for [my-action her-action]
  ;; my-action is 0 if C, 1 if D
  report 
    (1 - my-action) * (1 - her-action) * CC-payoff +
    (1 - my-action) * her-action * CD-payoff +
    my-action * (1 - her-action) * DC-payoff +
    my-action * her-action * DD-payoff
end

to split-up
  set numLL 0
  set numSL 0
  set numSS 0
  ask agents[
    if (mate != nobody) [ 
      ifelse ((([action] of mate = 0) and (break-if-C = 1))or(([action] of mate = 1) and (break-if-D = 1)))
        [ifelse (((action = 0) and ([break-if-C] of mate = 1))or((action = 1) and ([break-if-D] of mate = 1)))
          [  set past 0   set numLL (numLL + 0.5) ]
          [  set past 0   set numSL (numSL + 0.5) ] ]
        [ifelse (((action = 0) and ([break-if-C] of mate = 1))or((action = 1) and ([break-if-D] of mate = 1)))
          [  set past 0   set numSL (numSL + 0.5) ]
          [               set numSS (numSS + 0.5) ] ]
          ;; if programmed to split up after your mate's selected action
        ]
    ]
  set num-outcomesS (numLL + numSL + numSS)
  set %-LL ifelse-value (num-outcomesS = 0) [0] [(numLL / num-outcomesS)]
  set %-SL ifelse-value (num-outcomesS = 0) [0] [(numSL / num-outcomesS)]
  set %-SS ifelse-value (num-outcomesS = 0) [0] [(numSS / num-outcomesS)]
  
end  

to kill-and-breed
  ask agents [  
    ifelse (random-float 1.0 < ifelse-value (die-together) [ ((expected-life -  (sqrt (expected-life) * sqrt (expected-life - 1 ))) / expected-life ) ] [( 1.0 / expected-life )])
     [ if (mate != nobody)  [ ifelse (die-together)  [ask mate [die]] [ask mate [set past 0]     ask mate [set mate nobody]  ask mate[update-strategy-variables] ]]
       die ]
     [ if (past = 0) [set mate nobody]
       update-strategy-variables ] ]
 
end        





to adjust-num-players 
  let adjustment (num-players - (count agents))
  if adjustment != 0 [  
  ifelse adjustment > 0 
    [
      
    ;;;;;  ^ 1 is original
     let list-fitness n-values 162 [ifelse-value (count agents with [strategy-number = ?] > 0) [(((sum [payoff] of agents with [strategy-number = ?]) / (count agents with [strategy-number = ?])) ^ 5) * (count agents with [strategy-number = ?])][0]]
     
     

     if (sum list-fitness = 0) [set list-fitness n-values 162 [1]]        ;; Applies when all players have zero fitness 
     
     let cum-fitness [0]                                                  ;; cum-fitness last value is 0 and is 9 items long
     foreach list-fitness [set cum-fitness fput (? + first cum-fitness) cum-fitness]      
      
     create-agents adjustment [
        set past 0   
        set mate nobody

        if-else (random-float 1.0 < prob-mutation)
          [set strategy-number (floor(random 162.0))]
          [set strategy-number 161 
           let tmp random-float first cum-fitness    ;; select the new strategy with probability proportional to fitness 
           foreach butfirst cum-fitness [ if ( tmp < ?) [set strategy-number (strategy-number - 1)] ]
          ]
        update-strategy-variables
     ]
    ]
    [
      ask n-of (0 - adjustment) agents [
        if (mate != nobody)  [ask mate [set past 0]   ask mate [set mate nobody]   ]
        die
      ]
    ]
  ]
  
end
        
to update-strategy-variables
  ;; update strategy variables from strategy-number
  
  if-else (past = 0)            [set action            (floor ( strategy-number / 81)        ) ] [
   if-else (past = 1)           [set action     floor ((floor ( strategy-number / 27)) mod  3) ] [
    if-else (past = 2)          [set action     floor ((floor ( strategy-number /  9)) mod  3) ] [
     if-else (past = 3)         [set action     floor ((floor ( strategy-number /  3)) mod  3) ]
                                [set action     floor ((floor ( strategy-number     )) mod  3) ] ]]] 
  if-else (action = 0)          [set break-if-C ifelse-value (floor((floor ( strategy-number / 27)) mod  3) = 2) [1] [0] 
                                 set break-if-D ifelse-value (floor((floor ( strategy-number /  9)) mod  3) = 2) [1] [0]  ] 
                                [set break-if-C ifelse-value (floor((floor ( strategy-number /  3)) mod  3) = 2) [1] [0] 
                                 set break-if-D ifelse-value (floor((floor ( strategy-number     )) mod  3) = 2) [1] [0]  ] 
end

;;;;;;;;;;;;;;;;;;;;;;;;
;;;    Statistics    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

to gather-data
  
  ;; Agent-based model
  set paired-players (agents with [mate != nobody])
  set numCC (count paired-players with [action = 0 and [action] of mate = 0]) / 2
  set numDD (count paired-players with [action = 1 and [action] of mate = 1]) / 2
  set numCD (count paired-players with [action = 0 and [action] of mate = 1]) 
  set num-outcomes (numCC + numDD + numCD)
  set %-CC ifelse-value (num-outcomes = 0) [0] [(numCC / num-outcomes)]
  set %-CD ifelse-value (num-outcomes = 0) [0] [(numCD / num-outcomes)]
  set %-DD ifelse-value (num-outcomes = 0) [0] [(numDD / num-outcomes)]
  let n-players (count agents)


 
  
  
  
  set strategy-mode first (modes [strategy-number] of agents)
  set strategy-mode-% (count agents with [strategy-number = strategy-mode] / count agents * 100)
  
  
  set strategy-frequencies n-values 162 [count agents with [strategy-number = ?]]
  set strategy-names n-values 162 [?]
  


    
  
    set Z                                            (count           agents with [(floor(strategy-number /  9) mod 27) =  1])                           ;; TfTat
    set plot-avg-payoffs fput (ifelse-value (Z > 0) [(sum [payoff] of agents with [(floor(strategy-number /  9) mod 27) =  1]) /  Z] [0]) []
    set plot-frequencies fput Z []
  
    set Z                                            (count           agents with [(floor(strategy-number /  9) mod 27) =  2])                             ;; OfTat
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [(floor(strategy-number /  9) mod 27) =  2]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies
  
    set Z                                            (count           agents with [(floor(strategy-number / 27) mod 27) =  1])                             ;; C-prt
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [(floor(strategy-number / 27) mod 27) =  1]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies

    set Z                                            (count           agents with [((floor(strategy-number /  9) mod 27) =  0) or ((floor(strategy-number /  9) mod 27) =  6) or ((floor(strategy-number /  9) mod 27) =  8)])                           ;; AllC
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((floor(strategy-number /  9) mod 27) =  0) or ((floor(strategy-number /  9) mod 27) =  6) or ((floor(strategy-number /  9) mod 27) =  8)]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies

    set Z                                            (count           agents with [(floor(strategy-number /  9) mod 27) =  7])                             ;; OthC
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [(floor(strategy-number /  9) mod 27) =  7]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies


    set Z                                            (count           agents with [((strategy-number > 80) and ((floor(strategy-number /  1) mod 9) =  1))])                             ;; TfTit
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((strategy-number > 80) and ((floor(strategy-number /  1) mod 9) =  1))]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies

    set Z                                            (count           agents with [((strategy-number > 80) and ((floor(strategy-number /  1) mod 9) =  2))])                             ;; OfTit
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((strategy-number > 80) and ((floor(strategy-number /  1) mod 9) =  2))]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies

    set Z                                            (count           agents with [((strategy-number = 81) or (strategy-number = 84) or (strategy-number = 90) or (strategy-number = 93))])                             ;; WSLS
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((strategy-number = 81) or (strategy-number = 84) or (strategy-number = 90) or (strategy-number = 93))]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies

    set Z                                            (count           agents with [((strategy-number = 87) or (strategy-number = 96) or (strategy-number = 99) or (strategy-number = 102) or (strategy-number = 105))])                             ;; FGOF
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((strategy-number = 87) or (strategy-number = 96) or (strategy-number = 99) or (strategy-number = 102) or (strategy-number = 105))]) / Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies
     
    set Z                                            (count           agents with [((strategy-number > 107) and (strategy-number < 135) and ((floor(strategy-number /  1) mod 3) =  0))])   ;; D-prt
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((strategy-number > 107) and (strategy-number < 135) and ((floor(strategy-number /  1) mod 3) =  0))]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies
     
    set Z                                            (count           agents with [((strategy-number > 80) and(((floor(strategy-number /  1) mod 9) =  4) or ((floor(strategy-number /  1) mod 9) =  5) or ((floor(strategy-number /  1) mod 9) =  7) or ((floor(strategy-number /  1) mod 9) =  8)))])   ;; AllD
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((strategy-number > 80) and(((floor(strategy-number /  1) mod 9) =  4) or ((floor(strategy-number /  1) mod 9) =  5) or ((floor(strategy-number /  1) mod 9) =  7) or ((floor(strategy-number /  1) mod 9) =  8)))]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies     
  

    set Z                                            (count           agents with [((strategy-number > 134) and ((floor(strategy-number /  1) mod 3) =  0))])   ;; AllD
    set plot-avg-payoffs lput (ifelse-value (Z > 0) [(sum [payoff] of agents with [((strategy-number > 134) and ((floor(strategy-number /  1) mod 3) =  0))]) /  Z] [0]) plot-avg-payoffs
    set plot-frequencies lput Z plot-frequencies     


end



;;;;;;;;;;;;;;;;;;;;;;;;
;;;      Plots       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

to update-graphs   
  ;; all graphs refer to the situation before the new breed comes in.
  let current-num-agents (count agents)
   
  set-current-plot "Players C/D"
    set-current-plot-pen "no play"  plotxy ticks current-num-agents    
    set-current-plot-pen "in DD"    plotxy ticks (2 * num-outcomes)
    set-current-plot-pen "in CD/DC" plotxy ticks (2 * (num-outcomes - numDD))
    set-current-plot-pen "in CC"    plotxy ticks (2 * numCC)
    set-plot-y-range 0 current-num-agents
    
  set-current-plot "Players S/L"
    set-current-plot-pen "no play"  plotxy ticks current-num-agents    
    set-current-plot-pen "in LL"    plotxy ticks (2 * num-outcomesS)
    set-current-plot-pen "in SL/LS" plotxy ticks (2 * (num-outcomesS - numLL))
    set-current-plot-pen "in SS"    plotxy ticks (2 * numSS)
    set-plot-y-range 0 current-num-agents
  
  set-current-plot "Players' pairs"
    set-current-plot-pen "singles before matching"  
    plot n-of-singles-at-the-beginning
    set-current-plot-pen "singles after matching"
    plot (n-of-singles-at-the-beginning - n-of-singles-selected-to-be-paired)
    set-current-plot-pen "in pairs after matching"
    plot count paired-players
    set-plot-y-range 0 current-num-agents
    
  set-current-plot "Average Payoff"
    set-current-plot-pen "payoff"  
    plot avg-payoff
    set-plot-y-range DD-payoff CC-payoff

  set-current-plot "Strategy Distribution"
    let total (sum plot-frequencies)
    let bar 1
    foreach (n-values num-plots [?]) [
      set-current-plot-pen item ? plot-names 
      set-plot-pen-mode 1
      plotxy ticks bar 
      set bar (bar - ((item ? plot-frequencies) / total))
    ]    
    set-plot-y-range 0 1 
    
  set-current-plot "Strategy Distribution 2"
    let total2 (sum plot-frequencies)

    foreach (n-values num-plots [?]) [
      set-current-plot-pen item ? plot-names 
      set-plot-pen-mode 0
      plotxy ticks ((item ? plot-frequencies) / total2)   ]    
    set-plot-y-range 0 1   
    
  set-current-plot "Strategy Payoffs"


    
    foreach (n-values num-plots [?]) [
      set-current-plot-pen item ? plot-names 
      set-plot-pen-mode 0
      if (ticks > 0) [plotxy ticks (item ? plot-avg-payoffs) ]
    ]    
    set-plot-y-range CD-payoff  DC-payoff 
 
  
end
@#$#@#$#@
GRAPHICS-WINDOW
1050
293
1295
481
1
1
52.33333333333334
1
3
1
1
1
0
0
0
1
-1
1
-1
1
1
1
1
ticks
100.0

SLIDER
138
83
255
116
num-players
num-players
100
100000
10000
100
1
NIL
HORIZONTAL

SLIDER
11
10
131
43
CC-payoff
CC-payoff
0
10
3
1
1
NIL
HORIZONTAL

SLIDER
138
11
255
44
CD-payoff
CD-payoff
0
10
0
1
1
NIL
HORIZONTAL

SLIDER
11
46
131
79
DC-payoff
DC-payoff
0
10
4
1
1
NIL
HORIZONTAL

SLIDER
138
47
255
80
DD-payoff
DD-payoff
0
10
1
1
1
NIL
HORIZONTAL

BUTTON
11
229
88
262
Setup
startup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
176
230
257
263
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
657
13
1024
289
Players C/D
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"no play" 1.0 1 -10899396 true "" ""
"in DD" 1.0 1 -2674135 true "" ""
"in CD/DC" 1.0 1 -4539718 true "" ""
"in CC" 1.0 1 -13345367 true "" ""

MONITOR
657
293
764
338
% CC
%-CC * 100
6
1
11

MONITOR
901
293
1022
338
% DD
%-DD * 100
4
1
11

MONITOR
766
293
898
338
% CD
%-CD * 100
4
1
11

PLOT
662
652
1027
901
Players' pairs
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"singles before matching" 1.0 0 -16777216 true "" ""
"in pairs after matching" 1.0 0 -4699768 true "" ""
"singles after matching" 1.0 0 -10899396 true "" ""

SLIDER
11
119
140
152
expected-life
expected-life
1
100
50
1
1
NIL
HORIZONTAL

BUTTON
92
229
173
262
Go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
12
156
140
189
prob-mutation
prob-mutation
0
1
0
0.0001
1
NIL
HORIZONTAL

MONITOR
143
172
257
225
NIL
ticks
17
1
13

PLOT
271
16
650
648
Strategy Distribution
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"TfTat" 1.0 1 -12345184 true "" ""
"OfTat" 1.0 1 -11033397 true "" ""
"C-prt" 1.0 1 -14070903 true "" ""
"AllC" 1.0 1 -15390905 true "" ""
"OthC" 1.0 1 -612749 true "" ""
"TfTit" 1.0 1 -955883 true "" ""
"OfTit" 1.0 1 -2674135 true "" ""
"WSLS" 1.0 1 -8053223 true "" ""
"FGOF" 1.0 0 -10873583 true "" ""
"D-prt" 1.0 0 -16777216 true "" ""
"AllD" 1.0 0 -7500403 true "" ""
"OthD" 1.0 0 -6459832 true "" ""

SLIDER
12
192
138
225
prob-rematch
prob-rematch
0
1
1
0.01
1
NIL
HORIZONTAL

SLIDER
11
83
131
116
out-payoff
out-payoff
0
10
0
1
1
NIL
HORIZONTAL

PLOT
657
342
1022
597
Players S/L
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"no play" 1.0 1 -10899396 true "" ""
"in LL" 1.0 1 -2674135 true "" ""
"in SL/LS" 1.0 1 -4539718 true "" ""
"in SS" 1.0 1 -13345367 true "" ""

PLOT
1030
652
1393
901
Average Payoff
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"payoff" 1.0 0 -16777216 true "" ""

PLOT
1027
10
1389
646
Strategy Payoffs
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"TfTat" 1.0 1 -11221820 true "" ""
"OfTat" 1.0 1 -11033397 true "" ""
"C-prt" 1.0 1 -14070903 true "" ""
"AllC" 1.0 1 -15390905 true "" ""
"OthC" 1.0 1 -612749 true "" ""
"TfTit" 1.0 1 -955883 true "" ""
"OfTit" 1.0 1 -2674135 true "" ""
"WSLS" 1.0 1 -8053223 true "" ""
"FGOF" 1.0 0 -10873583 true "" ""
"D-prt" 1.0 0 -16777216 true "" ""
"AllD" 1.0 0 -7500403 true "" ""
"OthD" 1.0 0 -6459832 true "" ""

MONITOR
903
600
1019
645
% LL
%-LL * 100
4
1
11

MONITOR
784
600
901
645
% SL
%-SL * 100
4
1
11

MONITOR
657
600
779
645
% SS
%-SS * 100
4
1
11

MONITOR
1069
808
1143
853
avg payoff
total-payoff / num-players
4
1
11

SWITCH
145
118
258
151
die-together
die-together
1
1
-1000

MONITOR
44
324
156
369
strategy-mode-%
strategy-mode-%
17
1
11

MONITOR
160
324
255
369
strategy-mode
strategy-mode
17
1
11

MONITOR
45
384
102
429
DD
strategy-mode mod 3
17
1
11

MONITOR
65
384
122
429
DC
(floor (strategy-mode / 3)) mod 3
17
1
11

MONITOR
84
384
141
429
CD
(floor (strategy-mode / 9)) mod 3
17
1
11

MONITOR
106
384
163
429
CC
(floor (strategy-mode / 27)) mod 3
17
1
11

MONITOR
135
384
192
429
New
(floor (strategy-mode / 81)) mod 3
17
1
11

MONITOR
54
990
1603
1035
NIL
plot-avg-payoffs
2
1
11

MONITOR
25
913
4037
958
NIL
strategy-frequencies
17
1
11

MONITOR
39
953
609
998
NIL
plot-frequencies
17
1
11

INPUTBOX
13
445
168
505
main-strategy
-1
1
0
Number

PLOT
275
652
654
898
Strategy Distribution 2
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"TfTat" 1.0 0 -11221820 true "" ""
"OfTat" 1.0 0 -7500403 true "" ""
"C-prt" 1.0 0 -2674135 true "" ""
"AllC" 1.0 0 -955883 true "" ""
"OthC" 1.0 0 -6459832 true "" ""
"TfTit" 1.0 0 -1184463 true "" ""
"OfTit" 1.0 0 -10899396 true "" ""
"WSLS" 1.0 0 -13840069 true "" ""
"FGOF" 1.0 0 -14835848 true "" ""
"D-prt" 1.0 0 -13791810 true "" ""
"AllD" 1.0 0 -13345367 true "" ""
"OthD" 1.0 0 -8630108 true "" ""

@#$#@#$#@
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
