globals [
  this-cluster
  infinity                             ;; a very large number.
  tick-counter
  endogenous?
  payoff

  my-palette
]


turtles-own [
  strategy

  cluster

  pos-x
  pos-y
  node-clustering-coefficient


]

links-own
[
  rewired?                    ;; keeps track of whether the link has been rewired or not
]



to go
end


to construct-agent
  ;; turtles-own agent constructor

  set shape "circle"
  ;set label who
end


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setup Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;


to setup
  clear-all
  set endogenous? true
  set my-palette [ [255 127 39] [255 0 0] [255 255 0] [0 255 0] [0 64 128] [0 255 255] [128 128 255] [128 0 128] [255 0 128] [128 128 0] [0 0 0] [133 122 122]
  [255 127 39] [255 0 0] [255 255 0] [0 255 0] [0 64 128] [0 255 255] [128 128 255] [128 0 128] [255 0 128] [128 128 0] [0 0 0] [133 122 122] ]


  ask patches [ set pcolor white ]

  if topology = "Scale-Free" [ create-hubspoke ]
  if topology = "Random Permute" [ create-ringlat permute-all
      layout-circle (sort turtles) max-pxcor - 1
  ]
  if topology = "Square Lattice" [ create-square-lattice ]
  if topology = "Ring Lattice" [ create-ringlat ]
  if topology = "ER Graph" [ setup-erdos-renyi ]
  if topology = "Small World" [ rewire-all ]
  if topology = "Star" [ create-star ]
  if topology = "Full" [ crt N [ construct-agent ] ask turtles [ create-links-with other turtles ] layout-circle turtles 10 ]






  let num-solutions n-solutions
  if num-solutions < 1 [ set num-solutions 9999999999 ]

  ifelse endogenous?
  [
    if num-solutions > N  [ set num-solutions N ]
    ask turtles [ set strategy 0 set label strategy ]
    let i 0
    ask n-of num-solutions turtles [
      set i i + 1
      set strategy i
    ]
  ]
  [

    ask turtles [
      set strategy (random num-solutions) + 1
    ]
  ]

  let all-strategies [strategy] of turtles

  let need-zero? min all-strategies = 0

  ask turtles [
    set strategy ( position strategy sort remove-duplicates all-strategies )
    set label strategy
  ]

  set payoff n-values count-unique [ random-payoff ]
  ifelse need-zero? [ set payoff replace-item 0 payoff -99999999 ]
  [
    ask turtles [ set strategy strategy + 1 set label strategy ]
    set payoff fput -99999999 payoff
  ]

  reset-ticks

end

to run-step

  ask one-of turtles with [ strategy = 0 or strategy != update-strategy ] [
    set strategy update-strategy
    set label strategy
    set-color-ng strategy
  ]

  tick
end

to-report update-strategy

  ;; NEED TO DEFINE THIS LIST
  ;; because the dynamic definition is a randomized order!

  let worldview [strategy] of link-neighbors

  let option-payoff ( map
    [ [?1] -> (occurrences ?1 worldview) * (item ?1 payoff) ]
    (remove-duplicates worldview)
  )


  report item ( position (max option-payoff) option-payoff ) (remove-duplicates worldview)
end


;; creates a new lattice
to create-ringlat
  crt N [ set color black construct-agent ]
  wire-ringlat
end

to wire-ringlat
  layout-circle (sort turtles) max-pxcor - 1

  layout-circle (sort turtles with [ who mod 2 = 0] ) max-pxcor - 4
  ;; iterate over the turtles
  let ni 0
  while [ni < count turtles]
  [
    ;; make edges with the next two neighbors
    ;; this makes a lattice with average degree of 4
    let z 1
    while [z <= floor (k / 2)]
    [
      ask turtle ni [ create-link-with turtle ((ni + z) mod count turtles) [ set rewired? false ] ]
      set z z + 1
    ]
    set ni ni + 1
  ]
end















;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SCALE-FREE Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; used for creating a new node
to make-node [old-node]
end

;; This code is the heart of the "preferential attachment" mechanism, and acts like
;; a lottery where each node gets a ticket for every connection it already has.
;; While the basic idea is the same as in the Lottery Example (in the Code Examples
;; section of the Models Library), things are made simpler here by the fact that we
;; can just use the links as if they were the "tickets": we first pick a randaom link,
;; and than we pick one of the two ends of that link.
to-report find-partner
  report [one-of both-ends] of one-of links
end

;;;;;;;;;;;;;;;;;
;;; SF Layout ;;;
;;;;;;;;;;;;;;;;;

;; resize-nodes, change back and forth from size based on degree to a size of 1
to resize-nodes
  ifelse all? turtles [size <= 1]
  [
    ;; a node is a circle with diameter determined by
    ;; the SIZE variable; using SQRT makes the circle's
    ;; area proportional to its degree
    ask turtles [ set size sqrt count link-neighbors ]
  ]
  [
    ask turtles [ set size 1 ]
  ]
end



to layout
  ;; the number 3 here is arbitrary; more repetitions slows down thea
  ;; model, but too few gives poor layouts
  repeat 3 [
    ;; the more turtles we have to fit into the same amount of space,
    ;; the smaller the inputs to layout-spring we'll need to use
    let factor sqrt count turtles
    ;; numbers here are arbitrarily chosen for pleasing appearance
    layout-spring turtles links (1 / factor) (7 / factor) (( 2 * k) / factor)
    display  ;; for smooth animation
  ]
  ;; don't bump the edges of the world
  let x-offset max [xcor] of turtles + min [xcor] of turtles
  let y-offset max [ycor] of turtles + min [ycor] of turtles
  ;; big jumps look funny, so only adjust a little each time
  set x-offset limit-magnitude x-offset 0.1
  set y-offset limit-magnitude y-offset 0.1
  ask turtles [ setxy (xcor - x-offset / 2) (ycor - y-offset / 2) ]
end

to-report limit-magnitude [number limit]
  if number > limit [ report limit ]
  if number < (- limit) [ report (- limit) ]
  report number
end


;  Scale-Free Code Adapted From NetLogo library item
;  that is Copyright 2005 Uri Wilensky.
;  see info tab for full copyright info.



to setup-erdos-renyi

  ;; Make a circle of turtles
  create-turtles N [ construct-agent ]
  layout-circle turtles (max-pxcor - 1)

  let success? false
  let count-tries 0
  while [ not success? ] [
    ask links [ die ]
    ;; Now give each pair of turtles an equal chance
    ;; of creating a link
    ask turtles [
      ;; we use "self > myself" here so that each pair of turtles
      ;; is only considered once
      create-links-with turtles with [self > myself and
        random-float 1.0 < p ]
    ]

    set success? ( item 0 identify-clusters = 1 )
    set count-tries count-tries + 1
    if ( count-tries > 1000 ) [ set success? true print "couldn't make connected network!  try different parameters!" ]

  ]
end


;; clusters code adapted from "Dissemination of Culture" by ???
;; see info tab for full copyright info

to-report identify-clusters
  let max-cluster 0
  let num-clusters 0

  let seed one-of turtles
  ask turtles [ set cluster nobody ]
  while [seed != nobody] [
    ask seed [
      set cluster self
      set num-clusters num-clusters + 1
      set this-cluster 1
      grow-cluster
    ]
    if this-cluster > max-cluster [ set max-cluster this-cluster]
    set seed one-of turtles with [cluster = nobody]
  ]
  report list num-clusters max-cluster
end

to grow-cluster

    ask link-neighbors with [cluster = nobody] [
      if cluster = nobody [ set this-cluster this-cluster + 1 ]
      set cluster [cluster] of myself
      grow-cluster
    ]
end



;;  small-worlds code adapted by material
;;  Copyright 2005 Uri Wilensky.
;;  see info tab for full copyright info.

to rewire-all

  create-ringlat

  ;; set up a variable to see if the network is connected
  let success? false

  ;; if we end up with a disconnected network, we keep trying, because the APL distance
  ;; isn't meaningful for a disconnected network.
  let count-tries 0
  while [not success?] [
    ;; kill the old lattice, reset neighbors, and create new lattice
    ask links [ die ]
    wire-ringlat
;   set number-rewired 0

    ask links [

      ;; whether to rewire it or not?
      if (random-float 1) < p
      [
        ;; "a" remains the same
        let node1 end1
        ;; if "a" is not connected to everybody
        if [ count link-neighbors ] of end1 < (count turtles - 1)
        [
          ;; find a node distinct from node1 and not already a neighbor of node1
          let node2 one-of turtles with [ (self != node1) and (not link-neighbor? node1) ]
          ;; wire the new edge
          ask node1 [ create-link-with node2 [ set color cyan  set rewired? true ] ]

;          set number-rewired number-rewired + 1  ;; counter for number of rewirings
          set rewired? true
        ]
     ]
      ;; remove the old edge
      if (rewired?)
      [
        die
      ]
    ]

    set success? ( item 0 identify-clusters = 1 )
    set count-tries count-tries + 1
    if ( count-tries > 1000 ) [ set success? true print "couldn't make connected network!  try different parameters!" ]
  ]

end



to rewire-one

  ;; make sure num-turtles is setup correctly else run setup first
  if count turtles != N [
    setup
  ]


  let potential-edges links with [ not rewired? ]
  ifelse any? potential-edges [
    ask one-of potential-edges [
      ;; "a" remains the same
      let node1 end1
      ;; if "a" is not connected to everybody
      if [ count link-neighbors ] of end1 < (count turtles - 1)
      [
        ;; find a node distinct from node1 and not already a neighbor of node1
        let node2 one-of turtles with [ (self != node1) and (not link-neighbor? node1) ]
        ;; wire the new edge
        ask node1 [ create-link-with node2 [ set color cyan  set rewired? true ] ]


        ;; remove the old edge
        die
      ]
    ]
  ]
  [ user-message "all edges have already been rewired once" ]
end

to-report percent-saturated
  report ((count turtles with [color = red ] ) / (count turtles))
end

to create-square-lattice
  let lat-width (closest-factors N)
  let lat-height (N / lat-width)
  let width max-pxcor - min-pxcor
  let height max-pycor - min-pycor
  let left-edge min-pxcor + 0.5
  let top-edge min-pycor + 0.5
  let slot-width width / lat-width
  let slot-height height / lat-height


  ;; this sets the order of neighbor addition to easily create a square lattice
  ;; with any degree (up to the length of the map)
  let square-neighbor-map [ [1 0] [0 1] [-1 0] [0 -1] [1 1] [-1 1] [-1 -1] [1 -1] ]
  if (k > length square-neighbor-map) [ print "Square lattice degree too large!  Check code." stop ]

  print lat-width

  let xi 0
  while [ xi < lat-width ] [
    let yi 0
    while [ yi < lat-height ] [
      crt 1 [
        set pos-x xi
        set pos-y yi
        setxy ( (xi * slot-width) + left-edge) ( (yi * slot-height) + top-edge)
        construct-agent
      ]
      set yi yi + 1
    ]
    set xi xi + 1
  ]
  ask turtles [
    let ni 0
    while [ni < k] [
      let target-pos-x pos-x + item 0 item ni square-neighbor-map
      let target-pos-y pos-y + item 1 item ni square-neighbor-map
      if (target-pos-x > lat-width) [ set target-pos-x target-pos-x - lat-width ]
      if (target-pos-y > lat-height) [ set target-pos-y target-pos-y - lat-height ]
      if (target-pos-x < 0 ) [ set target-pos-x target-pos-x + lat-width ]
      if (target-pos-y < 0 ) [ set target-pos-y target-pos-y + lat-height ]
      create-links-with turtles with [ pos-x = target-pos-x and pos-y = target-pos-y ]
      set ni ni + 1
    ]
  ]
end

to-report closest-factors [ num ]
  let greatestEdge 1
  let i 1
  while [ i < num ] [
    if(num mod i = 0) [
      if abs (greatestEdge - (num / greatestEdge) ) > abs ( i - (num / i) ) [
        set greatestEdge i
      ]
    ]
    set i i + 1
  ]
  report greatestEdge
end


to create-hubspoke

  set-default-shape turtles "circle"
  ;; make the initial network of two turtles and an edge
  crt 1 [ construct-agent ]        ;; first node, unattached
  let mi 1
  while [ mi < m ][
    crt 1 [
      let counter 1
      repeat mi [
        create-link-with turtle (mi - counter)
        set counter counter + 1
      ]
      construct-agent
    ]
    set mi mi + 1
  ]      ;; second node, attached to first node

  repeat (N - m) [ add-link ]

  ask turtles with [who <= 5] [
    while [ count link-neighbors < m ] [
      let old-node find-partner
      while [ [who] of old-node = who ] [ set old-node find-partner ]
      create-link-with old-node
      ;; position the new node near its partner
      move-to old-node
      fd 8
    ]

  ]


end

to add-link
  ;; new edge is green, old edges are gray
  ask links [ set color gray ]
  make-node find-partner         ;; find partner & use it as attachment
                                 ;; point for new node

  crt 1
  [
    repeat m [
      let old-node find-partner
      while [ ([who] of old-node = who) OR (length filter [ [?1] -> ?1 = [who] of old-node ] [who] of link-neighbors != 0) ]
      [ set old-node find-partner ]
      create-link-with old-node
      ;; position the new node near its partner
      move-to old-node
      fd 8
    ]
    construct-agent
  ]

end


to permute

  let success false

  while [ success = false ]
  [
     let graphset sort link-set links

     let permutewith random count link-set links
     let permuter random count link-set links

     let abort false

     if [end1] of item permuter graphset = [end1] of item permutewith graphset [ set abort true ]
     if [end2] of item permuter graphset = [end1] of item permutewith graphset [ set abort true ]
     if [end2] of item permutewith graphset = [end1] of item permuter graphset [ set abort true ]
     ask [end1] of item permuter graphset
     [
       if link-neighbor? [end2] of item permutewith graphset [ set abort true ]
     ]

     ask [end1] of item permutewith graphset
     [
       if link-neighbor? [end2] of item permuter graphset [ set abort true ]
     ]

     if abort = false [
          ask [end1] of item permuter graphset    [ create-link-with [end2] of item permutewith graphset ]

          ask [end1] of item permutewith graphset [ create-link-with [end2] of item permuter graphset ]


          ask item permuter graphset [ die ]

          ask item permutewith graphset [ die ]
          set success true
     ]

  ]
end

to permute-all
  let z 0
  while [z < ((count turtles) * k * 3)]
  [
    permute
    set z z + 1
  ]
end

to labels-from-list [ label-list color-list ]
  let i 0
  while [i < length label-list] [
    ask turtle i [
      set label item i label-list
      set label-color item i color-list
    ]

    set i i + 1
  ]
end


to-report clustering-coefficient
  let global-clustering-coefficient 0
  ifelse all? turtles [count link-neighbors <= 1]
  [
    ;; it is undefined
    ;; what should this be?
    set global-clustering-coefficient -1
  ]
  [
    let total 0
    ask turtles with [ count link-neighbors <= 1]
      [ set node-clustering-coefficient "undefined" ]
    ask turtles with [ count link-neighbors > 1]
    [
      let hood link-neighbors
      set node-clustering-coefficient (2 * count links with [ in-neighborhood? hood ] /
                                         ((count hood) * (count hood - 1)) )
      ;; find the sum for the value at turtles
      set total total + node-clustering-coefficient
    ]
    ;; take the average
    set global-clustering-coefficient total / count turtles with [count link-neighbors > 1]
  ]
  report global-clustering-coefficient
end

to-report in-neighborhood? [ hood ]
  report ( member? end1 hood and member? end2 hood )
end


to-report graph-density
  set N count turtles
  report ( (2 * count links) / ( N * (N - 1) ) )
end


to-report random-lognormal [ mymean mystd ]
      let varx ( mystd ^ 2)
      let ess sqrt( ln(1 + (varx / ( mymean ^ 2))) )
      let mu ln(mymean) - ((1 / 2) * (ess ^ 2))
      report exp ( (mu + (ess * random-normal 0 1)) )
end

to-report random-payoff
  report random-lognormal 100 50
end


to-report occurrences [x the-list]
  report reduce
  [ [?1 ?2] -> ifelse-value (?2 = x) [?1 + 1] [?1] ] (fput 0 the-list)
end

to set-color-ng [ value ]
  ifelse  N > 24 [ set color value ]
  [
    let this-color value - 1
    set color item this-color my-palette
    if this-color < 12 [ set shape "circle" ]
    if this-color >= 12 [ set shape "x" ]
  ]
end

to create-star
  crt N [ construct-agent ]
  ask turtles with [ who < k ] [ create-links-with turtles with [ who >= k ] ]
  ask turtles with [ who >= k ] [ create-links-with turtles with [ who < k ] ]

  layout-circle turtles 10

end

to-report count-unique
report  length remove-duplicates [strategy] of turtles
end
@#$#@#$#@
GRAPHICS-WINDOW
9
10
386
388
-1
-1
11.2
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

CHOOSER
466
80
625
125
topology
topology
"Random Permute" "Scale-Free" "Full" "Ring Lattice" "Square Lattice" "Small World" "ER Graph" "Star"
1

INPUTBOX
468
131
518
191
k
8.0
1
0
Number

BUTTON
431
204
495
237
NIL
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
523
131
573
191
p
0.0
1
0
Number

INPUTBOX
576
132
626
192
m
4.0
1
0
Number

INPUTBOX
467
15
517
75
N
50.0
1
0
Number

PLOT
833
262
1033
412
Degree Distribution
NIL
NIL
0.0
20.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "histogram [count link-neighbors] of turtles"

MONITOR
473
370
530
415
ticks/N
tick-counter
17
1
11

BUTTON
507
204
586
237
Go Once
run-step
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
595
205
658
238
Go
run-step\nif count turtles with [ strategy != update-strategy or strategy = 0] = 0 \n[ stop ]
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
698
419
1035
613
plot 1
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"# Unique Strats" 1.0 0 -16777216 true "" "plot length remove-duplicates [strategy] of turtles"
"# Turts Unhappy" 1.0 0 -2674135 true "" "plot count turtles with [ strategy != update-strategy or strategy = 0]"
"# Turts Suboptimal" 1.0 0 -7500403 true "" "plot N - count turtles with [ item strategy payoff = max payoff]"
"pen-3" 1.0 0 -955883 true "" "plot mean [ item strategy payoff ] of turtles"

MONITOR
946
494
1011
539
not best?
N - count turtles with [ item strategy payoff = max payoff]
17
1
11

INPUTBOX
690
16
764
76
n-solutions
10.0
1
0
Number

@#$#@#$#@
## WHAT IS IT?

Coming Soon.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>run-step</go>
    <exitCondition>count turtles with [ strategy != update-strategy or strategy = 0] = 0</exitCondition>
    <metric>N - count turtles with [ item strategy payoff = max payoff]</metric>
    <metric>length remove-duplicates [strategy] of turtles</metric>
    <metric>mean [ item strategy payoff ] of turtles</metric>
    <enumeratedValueSet variable="p">
      <value value="0"/>
      <value value="0.001"/>
      <value value="0.01"/>
      <value value="0.05"/>
      <value value="0.1"/>
      <value value="0.15"/>
      <value value="0.2"/>
      <value value="0.25"/>
      <value value="0.3"/>
      <value value="0.5"/>
      <value value="0.7"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="m">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topology">
      <value value="&quot;Small World&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="star" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>run-step</go>
    <exitCondition>count turtles with [ strategy != update-strategy or strategy = 0] = 0</exitCondition>
    <metric>N - count turtles with [ item strategy payoff = max payoff]</metric>
    <metric>length remove-duplicates [strategy] of turtles</metric>
    <enumeratedValueSet variable="p">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="m">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topology">
      <value value="&quot;Star&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="scaling test" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>run-step</go>
    <exitCondition>count turtles with [ strategy != update-strategy or strategy = 0] = 0</exitCondition>
    <metric>N - count turtles with [ item strategy payoff = max payoff]</metric>
    <metric>length remove-duplicates [strategy] of turtles</metric>
    <metric>mean [ item strategy payoff ] of turtles</metric>
    <enumeratedValueSet variable="p">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="1000"/>
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="m">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topology">
      <value value="&quot;Scale-Free&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="exogeneous?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="theory-test" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>run-step</go>
    <exitCondition>count turtles with [ strategy != update-strategy or strategy = 0] = 0</exitCondition>
    <metric>N - count turtles with [ item strategy payoff = max payoff]</metric>
    <metric>length remove-duplicates [strategy] of turtles</metric>
    <metric>mean [ item strategy payoff ] of turtles</metric>
    <enumeratedValueSet variable="p">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-solutions">
      <value value="0"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="endogenous?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="m">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topology">
      <value value="&quot;Scale-Free&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="scale-test-revised" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>run-step</go>
    <exitCondition>count turtles with [ strategy != update-strategy or strategy = 0] = 0</exitCondition>
    <metric>count turtles with [ item strategy payoff = max payoff]</metric>
    <metric>length remove-duplicates [strategy] of turtles</metric>
    <metric>mean [ item strategy payoff ] of turtles</metric>
    <enumeratedValueSet variable="p">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-solutions">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="endogenous?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="k">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="m">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topology">
      <value value="&quot;Scale-Free&quot;"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
