;; ----------------------------------------------------------------
;; Income distribution, inequality and the evolution of credit risk
;; Created by Georgios Papadopoulos as a final project for the Introduction to Agent-Based Modeling MOOC, 2016 by Bill Rand @ Complexity Explorer
;; Contact: gpapad.gr@gmail.com
;; ----------------------------------------------------------------

globals [ annual-savers-rate% monthly-savers-rate minimum-expenditure
   total-deposit-amount total-outstanding-borrowing reserve-ratio 
   total-non-performing-loans gini p10 p20 p30 p40 p50 p60 p70 p80 p90 ]

breed [ households household ]

;; Define households' properties
households-own [
    budget
    has-loan?
    has-deposits?
    deposit-amount
    amount-withdrawn
    amount-needed
    available-this-month
    expenditure-this-month
    monthly-loan-payment
    monthly-loan-rate
    capital-repayment
    borrower-interest-payment
    outstanding-borrowing
    defaulter?
  ]

to setup
    clear-all
    ask patches  [ set pcolor white ]
    
    ;; Create household agents
    create-households num-households [
    set shape "person"
    set color blue
    setxy random-pxcor random-pycor
    set has-loan? false
    set has-deposits? false
    set deposit-amount 0
    set amount-withdrawn 0
    set amount-needed 0
    set available-this-month 0
    set expenditure-this-month 0
    set monthly-loan-payment 0
    set monthly-loan-rate 0
    set capital-repayment 0
    set borrower-interest-payment 0
    set outstanding-borrowing 0
    set defaulter? false
    
    ;; Move turtles to free space
    if any? other turtles-here [ 
     move-to one-of patches with [ not any? turtles-here ] 
     ]
  ]
    
    ;; Assign budgets according to selected distribution (uniform, normal, exponential) following Hamill & Gilbert (2016)
    if income-distribution = "uniform" [
      ask households [ set budget 350 + random ( 1000 + 350 )  ]
    ]
    
    if income-distribution = "normal" [
      ask households [ set budget random-normal 1000 200  ]
    ]
    
    if income-distribution = "exponential" [
      ask households [ set budget 350 + random-exponential ( 1000 - 300 )  ]
    ]
    if income-distribution = "gamma" [
      let average 1000
      let var 80000
      let alpha average * average / var
      let lambda 1 / ( var / average )
      ask households [ set budget random-gamma alpha lambda  ]
    ]
  
    ;; Set minimum expenditure as 80% of minimum budget
    set minimum-expenditure 0.8 * min [ budget ] of households
    
    ;; Set reserve ratio
    ;; (1 - reserve-ratio) * deposits is the available amount for being lent
    set reserve-ratio 0.1 ;; directly in %
    
    ;; Initialize total deposits, loans and non-performing loans
    set total-deposit-amount 0
    set total-outstanding-borrowing 0
    set total-non-performing-loans 0
    
    ;; (income) Inequality measure estimation
    update-gini
    percentile-calc
    
  reset-ticks
end

to go
  
  ;; The simulation will stop at 3 years (3 * 12 months) after the initiation of the stress scenario
  ;; This timeframe is typical for stress testing exercises
  if ticks >= ( shock-t + 36 ) and shock-t != 0 [ stop ]
  
  set total-deposit-amount sum [ deposit-amount ] of households with [ has-deposits? ]
  set total-outstanding-borrowing sum [ outstanding-borrowing ] of households with [ has-loan? ]
  set total-non-performing-loans sum [ outstanding-borrowing ] of households with [ defaulter? ]
  
  ;; Implement scenario
  ;; ------------------
  
  ;; Mild shock
  if scenario = "mild shock" and ( ticks >= shock-t ) [
    
    ;; The orange-ish color denotes the state of the economic "environment"
    ;; Households will take more conservative actions once in adverse (mild or severe) conditions
    ask patches  [ set pcolor 27 ]
    
    ask households [
      
      ;; Budget decreases by 0.2% each month or by approximately 2.4% per year
      set budget ( 0.998 * budget )
      
      ]
    ;; Unemployment (budget = 0) rises by 2.4% per year (number based on 500 households due to rounding)
    let num-unemployed round ( 0.0015 * num-households )
    ask n-of num-unemployed households with [ budget > 0 ] [ 
      set budget 0
    ]
  ] 
  
  ;; Severe shock
  ;; This scenario reflects the 2010 sovereign debt crisis experienced in Greece (based on Eurostat data)
  if scenario = "severe shock" and ( ticks >= shock-t ) [
    
    ;; The red-ish color denotes the state of the economic "environment"
    ;; Households will take more conservative actions once in adverse (mild or severe) conditions
    ask patches  [ set pcolor 13 ]
    
    ask households [
      
      ;; Budget decreases by 0.5% each month or by approximately 5.8% per year (number based on 500 households due to rounding)
      set budget ( 0.995 * budget )   
    ]
    ;; Unemployment (budget = 0) rises by about 5% per year
    let num-unemployed round ( 0.004 * num-households )
    ask n-of num-unemployed households with [ budget > 0 ] [ 
      set budget 0 
    ]
  ]

  ask households [
    
    set available-this-month budget
    
    ;; Spend on basic needs
    ;; --------------------

    ;; If budget greater than expenditure for basic needs (set as a percentage of minimum budget), then spend from it
    if ( budget >= minimum-expenditure ) [ 
      spend-basic
    ]
    
    ;; If budget less than minimum expenditure, then withdraw from deposits, if any
    if ( budget < minimum-expenditure ) [ 

      ;; Calculate the amount needed to meet basic needs
      set amount-needed ( minimum-expenditure - budget )
      
      ;; If the needed funds are available, withdraw them from deposits and meet basic needs
      if ( amount-needed < deposit-amount ) [ 
        
        withdraw-deposit
        spend-basic
        
      ]
      
      if ( amount-needed > deposit-amount ) [ 
        set color grey ;; To show that this household is economically inactive
      ]
      
    ]
    
    ;; Collect interest on deposits, if any
    if has-deposits? [ 
      collect-interest
    ]
    
    ;; Repay existing loan
    ;; -------------------
    
    ;; If budget exceeds loan payment then repay loan
    if ( available-this-month > monthly-loan-payment ) [ 
      pay-loan       
      set defaulter? false
    ]
    
    ;; If budget is less than monthly loan payment withdraw from deposits, if any
    if ( available-this-month < monthly-loan-payment ) [
      
      ;; Calculate the amount needed to meet the loan payment
      set amount-needed ( monthly-loan-payment - available-this-month )
      
      ;; If the needed funds are available, withdraw them from deposits and pay the loan
      if ( amount-needed < deposit-amount ) [ 
        
        withdraw-deposit
        pay-loan
        
        set color orange ;; To show that this household uses deposits to meet its needs
        set defaulter? false
        
      ]

      ;; If the needed funds are not available, declare household as a defaulter
      if ( amount-needed > deposit-amount ) [
        set defaulter? true
        set color red
      ]
      
    ]
    
    ;; Ask for a new loan
    ;; ------------------
    
    ;; Flip a coin to whether or not ask for a loan
    if random-float 1.0 < 0.5 and not has-loan? and not defaulter? [
      ask-for-loan
      ]
    
    ;; Decide action with remaining budget
    ;; -----------------------------------
    
    ;; Flip a coin to decide whether to spend or deposit the remaining budget, if any
    if available-this-month > 0 [ 
      ifelse random-float 1.0 < 0.5 [ deposit ] [ spend-extra ]
    ]
    
    ;; Spend some of the deposits (thus simulating e.g. some luxury needs, buys etc)
    ;; If the economic conditions are adverse then the overall amount spent on luxuries is lower
    ;; -----------------------------------------------------------------------------------------
    
    ;; Normal conditions
    if pcolor = white [ 
      if random-float 1.0 < 0.5 [
        let deposit%-to-spend random-float 0.3
        set expenditure-this-month ( deposit%-to-spend * deposit-amount )
        set deposit-amount ( 1 - deposit%-to-spend ) * deposit-amount
        ]
      ]
      
    ;; Adverse conditions
    if pcolor != white [
      if random-float 1.0 < 0.5 [
        let deposit%-to-spend random-float 0.1
        set expenditure-this-month deposit%-to-spend * deposit-amount
        set deposit-amount ( 1 - deposit%-to-spend ) * deposit-amount
        ]
      ]
    ]
  
  ;; For househodlds that have repaid their loan
  ask households with [ outstanding-borrowing <= 0 ] [
     set outstanding-borrowing 0
     set has-loan? false 
     set monthly-loan-payment 0
     set monthly-loan-rate 0
     set capital-repayment 0
     set borrower-interest-payment 0
     set color blue 
  ]  
    
    ;; (income) Inequality measure estimation
    update-gini
    percentile-calc
    
  tick
end

;; ------------------
;;     PROCEDURES
;; ------------------

to spend-basic
  
  set expenditure-this-month minimum-expenditure
  set available-this-month ( available-this-month - minimum-expenditure )
  
end

to deposit
  
  set deposit-amount ( deposit-amount + available-this-month )
  set available-this-month 0
  set has-deposits? true
  
end

to collect-interest
  
  let savers-interest-payment 0
  
  set annual-savers-rate% 1.5
  set monthly-savers-rate annual-savers-rate% / ( 12 * 100 )

  set savers-interest-payment deposit-amount * monthly-savers-rate
  set deposit-amount ( deposit-amount + savers-interest-payment )  
  
end

to spend-extra
  
  set expenditure-this-month ( expenditure-this-month + available-this-month )
  set available-this-month 0
  
end

to ask-for-loan
  
  let annual-loan-rate% 0
  let loan-term-months 0
  let monthly-payment 0

  let loan-size 5000 + random 20000

  ifelse loan-size <= 10000 [
    set annual-loan-rate% 5
    set monthly-loan-rate ( annual-loan-rate% / 12 ) / 100
    set loan-term-months 60 ;; 5 years
    ]
  
  [
    set annual-loan-rate% 3
    set monthly-loan-rate ( annual-loan-rate% / 12 ) / 100
    set loan-term-months 120 ;; 10 years
    ]
  
  set monthly-payment ( loan-size * monthly-loan-rate ) / (1 - (( 1 + monthly-loan-rate) ^ ( - loan-term-months )))
  
  ;; Reserve ratio requirement must be met
  if ( total-outstanding-borrowing + loan-size ) <= ( 1 - reserve-ratio ) * total-deposit-amount [
   
    ;; Affordabiity test for loans. In its simplest form, budget should exceed monthly-payment + amount for basic needs
    ifelse affordability-test [
      if not has-loan? and ( budget-payment-ratio * budget ) >= ( monthly-payment + minimum-expenditure ) [
        set has-loan? true
        set outstanding-borrowing loan-size
        set monthly-loan-payment monthly-payment
        set deposit-amount deposit-amount + outstanding-borrowing
        set has-deposits? true
        set color green
      ]    
   ]
    [
    if not has-loan? [ 
      set has-loan? true
      set outstanding-borrowing loan-size
      set monthly-loan-payment monthly-payment
      set deposit-amount deposit-amount + outstanding-borrowing
      set has-deposits? true
      set color green
     ]
   ] 
  ]

end

to pay-loan

  if outstanding-borrowing > 0 and not defaulter? [
    set borrower-interest-payment ( outstanding-borrowing * monthly-loan-rate )
    set capital-repayment ( monthly-loan-payment - borrower-interest-payment )
    set outstanding-borrowing ( outstanding-borrowing - capital-repayment )
    set available-this-month ( available-this-month - monthly-loan-payment )
    set expenditure-this-month ( expenditure-this-month + monthly-loan-payment )
    ]
  
end

to withdraw-deposit
  
  set amount-withdrawn amount-needed
  set deposit-amount ( deposit-amount - amount-withdrawn )
  set available-this-month ( available-this-month + amount-withdrawn )
  
end

to update-gini
  
  let sorted-income sort [ budget ] of households
  let min-income min [ budget ] of households
  let total-income sum sorted-income
  let income-sum-so-far 0
  let index 0
  set gini 0

  if total-income = 0 [set gini 0]
  if min-income < 0 [
    set total-income total-income - num-households * min-income
    set index 0
    repeat num-households [
      set sorted-income replace-item index sorted-income ( item index sorted-income - min-income )
      set index index + 1
    ]
  ]
  set index 0
  if total-income > 0
  [
    repeat num-households [
      set income-sum-so-far ( income-sum-so-far + item index sorted-income )
      set index (index + 1)
      set gini gini + (index / num-households) - (income-sum-so-far / total-income )
    ]
  ]
  set gini (gini / num-households ) / 0.5
  
end

to percentile-calc
  
  let sorted-income sort [ budget ] of households
  set p10 item ( round ( 0.1 * num-households ) ) sorted-income
  set p20 item ( round ( 0.2 * num-households ) ) sorted-income
  set p30 item ( round ( 0.3 * num-households ) ) sorted-income
  set p40 item ( round ( 0.4 * num-households ) ) sorted-income
  set p50 item ( round ( 0.5 * num-households ) ) sorted-income
  set p60 item ( round ( 0.6 * num-households ) ) sorted-income
  set p70 item ( round ( 0.7 * num-households ) ) sorted-income
  set p80 item ( round ( 0.8 * num-households ) ) sorted-income
   
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
649
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
9
167
72
200
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
111
167
174
200
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
10
182
43
num-households
num-households
100
1000
500
100
1
NIL
HORIZONTAL

CHOOSER
9
47
181
92
income-distribution
income-distribution
"uniform" "normal" "exponential" "gamma"
2

PLOT
3
296
203
446
Overall income distribution
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 5.0 1 -16777216 true "" "set-plot-x-range min [ precision budget 0 ] of households with [ budget > 0] max [ precision budget 0 ] of households with [ budget > 0]\nset-plot-y-range 0 0.001 * num-households\nhistogram [ budget ] of households with [ budget > 0]"

MONITOR
3
449
106
494
Average budget
mean [ budget ] of households with [ budget > 0 ]
1
1
11

SWITCH
9
216
159
249
affordability-test
affordability-test
0
1
-1000

SLIDER
7
254
159
287
budget-payment-ratio
budget-payment-ratio
1
1.3
1.3
0.05
1
NIL
HORIZONTAL

PLOT
905
11
1177
190
# of borrowers (by type)
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Defaulted" 1.0 0 -2674135 true "" "plot count households with [ defaulter? ]"
"Servicing" 1.0 0 -10899396 true "" "plot count households with [ color = green ]"
"Service w deposit" 1.0 0 -955883 true "" "plot count households with [ color = orange ]"
"No loan" 1.0 0 -13345367 true "" "plot count households with [ color = blue ]"
"Inactive" 1.0 0 -7500403 true "" "plot count households with [ color = grey ]"
"Shock time" 1.0 0 -5825686 false "" "if scenario != \"normal\" [\nlet maxy plot-y-max \nplotxy shock-t 0\nplotxy shock-t maxy ]"

PLOT
655
11
899
190
Total non-performing loans
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"Non-performing loans" 1.0 0 -2674135 true "" "plot total-non-performing-loans"
"Shock time" 1.0 0 -5825686 false "" "if scenario != \"normal\" [\nlet maxy plot-y-max \nplotxy shock-t 0\nplotxy shock-t maxy ]"

PLOT
907
193
1177
360
Total expenditure
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot sum [ expenditure-this-month ] of households"
"Shock time" 1.0 0 -5825686 false "" "if scenario != \"normal\" [\nlet maxy plot-y-max \nplotxy shock-t 0\nplotxy shock-t maxy ]"

PLOT
659
367
900
525
Unemployment rate
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot 100 * count households with [ budget = 0 ] / num-households"
"Shock time" 1.0 0 -5825686 false "" "if scenario != \"normal\" [\nlet maxy plot-y-max \nplotxy shock-t 0\nplotxy shock-t maxy ]"

CHOOSER
7
99
125
144
scenario
scenario
"normal" "mild shock" "severe shock"
2

INPUTBOX
131
96
181
156
shock-t
200
1
0
Number

MONITOR
907
366
1028
411
Unemployment rate
100 * count households with [ budget = 0 ] / num-households
3
1
11

PLOT
656
194
899
360
Income distribution of defaulters
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -2674135 true "" "if count households with [ defaulter? ] > 0 [ \n\nlet maxx 1 + max [ precision budget 0 ] of households with [ defaulter? ] \nset-plot-x-range min [ precision budget 0 ] of households with [ defaulter? ] maxx\nhistogram [ budget ] of households with [ defaulter? ]\n\n ]"

PLOT
212
475
412
630
Gini index
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot gini"
"pen-1" 1.0 0 -5825686 false "" "if scenario != \"normal\" [\nlet maxy plot-y-max \nplotxy shock-t 0\nplotxy shock-t maxy ]"

MONITOR
114
450
202
495
Gini index
gini
3
1
11

MONITOR
113
499
202
544
p80 : p20 ratio
p80 / p20
3
1
11

PLOT
425
477
626
630
p80 : p20 ratio
NIL
NIL
0.0
10.0
0.0
5.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot p80 / p20"
"pen-1" 1.0 0 -5825686 true "" "if scenario != \"normal\" [\nlet maxy plot-y-max \nplotxy shock-t 0\nplotxy shock-t maxy ]"

@#$#@#$#@
## WHAT IS IT?

This is a simple model to explore the effect of income distribution on the transmission of shocks and the evolution of non-performing loans and household expenditure (as a representative macroeconomic variable) under normal and adverse economic conditions.

## HOW IT WORKS

### Agents

The model has one type of agent named _household_

#### Properties

Households have the following properties:

* _shape_ is fixed to be _person_.
* _color_ assumes a different value according to the loan-servicing status of the Household.
    * _blue_ the Household doesn't have a loan.
    * _green_ the Household has a loan and services it from its budget.
    * _orange_ the Household has a loan and services it from its deposits.
    * _red_ the Household does not service its loan.
    * _grey_ the Household does not have a loan nor can meet its basic needs. Typically this category is very small.
* _budget_ is the monthly income a Household receives.
* _has-loan?_ is True when the Household has a loan, False otherwise.
* _has-deposits?_ is True when the Household has deposits, false otherwise.
* _deposit-amount_ keeps record of the deposits a Household owns.
* _amount-withdrawn_ keeps record of the deposit amount withdrawn from a Household.
* _amount-needed_ denotes the extra amount (if the budget alone is not enough) a Household needs to meet either its basic needs or its loan payments.
* _available-this-month_ is the available money for spending each month. At the beggining of each month has the value of _budget_ and it is reduced as the Household spends it for basic needs and/or loan payments.
* _expenditure-this-month_ keeps record of the current month expenditure of each Household
* _monthly-loan-payment_ is the calculated monthly loan payment for each Household. It depends on the size, the interest rate and the duration (term) of the loan.
* _monthly-loan-rate_ is the (monthly) loan interest rate.
* _capital-repayment_ keeps track of the capital repayments of each Household.
* _borrower-interest-payment_ keeps track of the interest repayments of each Household.
* _outstanding-borrowing_ is the amount of loans each Household has.
* _defaulter?_ is True when the Household doesn't service its loan, false otherwise.

#### Actions

Households can take the following actions during the simulation:

* Spend money to meet their basic needs. This money can come either from their _budget_, from their deposits or both.
* Deposit any remaining money (after they've met all their obligations) into their savings account.
* Collect interest on their deposits. For this simulation, the annual savings interest rate is set to 1.5%.
* Spend any remaining money (after they've met all their obligations) at the end of the month.
* Ask for a loan. Each Household can have one loan at a time. There are two types of loans based on their size (which is randomly decided by the requesting Household and is between 5000 and 25000 currency units):
    * Loans less than or equal to 10000 currency units. Their duration (also known as "term") is 5 years (60 months) and their annual interest rate is 5%.
    * Loans greater than 10000 currency units. Their duration is 10 years (120 months) and their annual interest rate is 3%.
* Pay a loan. This action consists of repaying a loan's interest and capital by using any funds available (_budget_ and/or deposits).
* Withdraw from deposits. This action is taken either when a Household cannot meet any of its needs (basic or loan paymennt) with its budget only, or when it wants to spend some amount from its savings (eg. for some luxury buys).

### Environment

The environment consists of the patches and it reflects the overall "economic conditions sentiment" experienced by the Households. The environment affects Househods in their decisions to spend a proportion of their deposits (simulating some luxury buys etc.).

This environment-agent interaction is controled by the color of the patches. It can assume three different values:

* If the economic conditions are benign (**_scenario_** is set to _normal_) the patches are white.
* If the economic conditions are mildly adverse (**_scenario_** is set to _mild shock_) the patches are orange-ish.
* If the economic conditions are severely deteriorated (**_scenario_** is set to _severe shock_) the patches are red-ish.

### Order of events

#### Initialization (SETUP)

1. Every Household property is initialized to zero or to false if it is boolean except for the _budget_.
2. Budgets are randomly assigned to each Household according to the selected distribution (_uniform_, _normal_, _exponential_) following Hamill & Gilbert (2016). The mean and minimum values are approximately the same and are 1000 and 350 currency units respectively.
3. A minimum expenditure is set to 80% of the minimum _budget_ to simulate the minimum amount that a Household will need to meet its basic needs. Notice that this value remains fixed and does not change under the various scenarios.
4. The reserve ratio is set to 10%. Therefore, (1 - reserve ratio) times the deposits is the available amount of money for borrowing. For more information on the reserve ratio see the respective Wikipedia lemma on [reserve requirement] (https://en.wikipedia.org/wiki/Reserve_requirement).
5. All patches are set to be initially white. Thus, their color will change after the implementation of a given shock and remain changed for as long as the shock lasts (here until the end of the simulation).

#### Execution (GO)

The simulation starts by checking which **_scenario_** is selected and runs for **_shock-t_** + 36 ticks. Assuming that each tick corresponds to one month, the 3 year duration of the simulation emulates the typical time-frame used for bank stress testing exercises (Schuermann, 2014). The following scenarios are implemented:

* In the _normal_ scenario no shocks are implemented and it can be used as a baseline for comparison with the other scenarios.
* In the _mild shock_ scenario, the _budget_ of every Household decreases by 0.2% each month or by approximately 2.4% per year. In addition, unemployment (randomly choosing Households and setting their _budget_ = 0) rises by 2.4% per year.
* The _severe shock_ scenario reflects the 2010 sovereign debt crisis experienced in Greece (based on [Eurostat] (http://ec.europa.eu/eurostat/data/database) data). In this case _budget_ decreases by 0.5% each month or by approximately 5.8% per year and unemployment increases by approximately 5% per year.

Next, each Household takes the following actions:

1. Spend on basic needs
  1.1 If the _budget_ is greater than expenditure for basic needs, then spend from it.
  1.2 Else, if _budget_ is less than minimum expenditure and there are enough deposits, withdraw from deposits.
  1.3 Else, if there are no sufficient deposits to meet basic needs, mark this Household as inactive.
2. Collect interest on deposits, if any
3. Repay existing loan
  3.1 If _budget_ exceeds loan payment then repay loan.
  3.2 Else, if budget is less than monthly loan payment and there are enough deposits, withdraw from deposits.
  3.3 If the needed funds are not available, declare Household as a defaulter.
4. Ask for a new loan (if it doesn't have one nor is a defaulter)
  4.1 Flip a coin to whether or not ask for a loan and proceed accordingly.
5. Decide action with remaining budget
  5.1 Flip a coin to decide whether to spend or deposit the remaining _budget_, if any, and proceed accordingly.
6. Spend some of the deposits (thus simulating e.g. some luxury buys etc.)
  6.1 If the conditions are normal (patch color is white), flip a coin to decide whether or not to spend some deposits and if so, randomly choose an amount up to 30% of the deposits.
  6.2 If the conditions are adverse (patch color is orange or red) flip a coin to decide whether or not to spend some deposits and if so, randomly choose an amount up to 10% of the deposits.
7. Finally, if a Household has repaid its loan, set all related properties to zero or false.

## HOW TO USE IT

### Inputs

The modeller is required to provide the following inputs:

1. Set the number of households using the slider **_num-households_**.
2. Select the desired income distribution from the chooser **_income-distribution_**. The available options are _uniform_, _normal_, _exponential_ and _gamma_.
3. Select the scenario from the chooser **_scenario_**. The available options are _normal_, _mild shock_ and _severe shock_.
4. Set the time when the shock will begin by typing an integer in the input box **_shock-t_**. Remember the following: a) The shock will be applied from the moment specified until the end of the simulation, 36 ticks later; b) Even when the _normal_ scenario is selected (meaning that there is no shock applied) the **_shock-t_** needs to be specified in order for the simulation to terminate. This way a standard simulation time for each scenario is ensured, thus facilitating comparison among them.

After the previous core settings, there is the option to switch on or off the **_affordability-test_** and set its value using the slider **_budget-payment-ratio_**.

1. The switch **_affordability-test_** (de)activates a check to be applied before granting a loan. The check consists of testing whether or not the monthly loan repayment plus the minimum monthly expenditure is less or equal to **_budget-payment-ratio_** times the monthly budget. If activated and this condition holds, then the loan is granted otherwise it is not.

After the aforementioned initial settings, the button **_go_** starts the simulation. 

### Outputs

The plot **_Overall income distribution_** and the monitor **_Average budget_** serve mainly as model verification aids. The plot displays the monthly budget distribution of all Households and the monitor shows its average value. They both assist to confirm that the distribution of income has the desired (shape and mean) properties.

There are two plots with their accompanying monitors related to measuring income inequality and its evolution. The measures used are the [ commonly] ( https://www.equalitytrust.org.uk/notes-statistical-sources-and-methods ) studied **_Gini index_** and the **_p80:p20 ratio_**. A detailed description of the former can be found in the Wikipedia [lemma] ( https://en.wikipedia.org/wiki/Gini_coefficient ), whereas the latter displays how many times richer is the upper 20% of the households compared to the bottom 20% of the households. For the estimation of the Gini index was used the respective part of the code from NetLogo Wealth Distribution model.

The plot **_Total non-performing loans_** is visualizing the main output of the model, the total amount of non-performing loans. A vertical line in this and other plots denotes the start of the stress scenario. If the _normal_ scenario is selected, then no vertical line is drawn.

The plot **_# of borrowers (by type)_** displays the number of Households according to their loan-servicing status. There are 4 types of borrowers; _Defaulted_, _Servicing_, _Servicing w deposits_, _No loan_ and _Inactive_. The first and fourth types are the ones who don't service their loan and don't have a loan respectively. The difference between the _Servicing_ and _Servicing w deposits_ is that the former pay their loan using only their budget, while the latter resort to their deposits to meet their payments. The _Inactive_ type is always very small compared to the other types and denotes the Households which do not have a loan nor can meet their basic needs.

The plot **_Income distribution of deafulters_** shows the distribution of income of the defaulted Households. It is expected that most defaults will be from zero income (unemployed) Households.

The plot **_Total expenditure_** shows the behaviour of the total Household expenditure as an example of a macroeconomic variable and its evolution under normal and stress conditions.

The purpose of the **_Unemployment rate_** plot and its accompanying monitor is to provide a verification tool to check that the implementation of the shock is as intended. They both display the number of zero income (unemployed) Households as a percentage of all Households.

## THINGS TO TRY

* Select a given income distribution and examine the impact of different shocks to the total amount of non-performing loans and Household expenditure.
* Select a given shock and examine the impact of different income distributions to the total amount of non-performing loans and Household expenditure.
* Examine the effect that different shocks have to income inequality.
* Perform the previous analyses with the **_affordability-test_** switched on/off.

## EXTENDING THE MODEL

The simple model presented here can be extended in various ways such as:

* Implement a recovery period after the end of the shock period. Examine under which conditions and/or how long will it take to return to the pre-shock levels of non-performing loans and/or Household expenditure.
* Create a local "economic conditions sentiment" environment where agents experience the shock in a heterogeneous manner and influence their neighbours accordingly.
* Add a breed for Bank agents so as Households can ask loans from different Banks.
* Create a more detailed banking sector with variables such as Equity, Profits, Liquid Assets, Risk Weighted Assets etc.
* Elaborate on Banks' behaviour by adding capital adequacy, liquidity and other regulatory constraints.

## RELATED MODELS

This model is inspired by and utilizes parts from the Banking model of Hamill, L. & Gilbert, N. (2016). http://cress.soc.surrey.ac.uk/web/sites/default/files/user-uploads/u1/Chapter%209-Banking.nlogo

## CREDITS AND REFERENCES
Built and runs on NetLogo 5.2.

Hamill, L. & Gilbert, N. (2016) Agent-based Modelling in Economics. Wiley: Chapter 9. http://cress.soc.surrey.ac.uk/web/publications/books/agent-based-modelling-economics/more-information

Schuermann, T. (2014). Stress testing banks. International Journal of Forecasting, 30(3), 717-728.

Wilensky, U. (1998).  NetLogo Wealth Distribution model.  http://ccl.northwestern.edu/netlogo/models/WealthDistribution.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2016 Georgios Papadopoulos

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International License. To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/4.0/
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-non-performing-loans</metric>
    <enumeratedValueSet variable="num-households">
      <value value="600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="affordability-test">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="scenario">
      <value value="&quot;severe shock&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="budget-payment-ratio">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock-t">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;uniform&quot;"/>
      <value value="&quot;normal&quot;"/>
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
