
; the regular traders in the market
breed [traders trader]
; traders who don't hold stock, and only attempt
; to make money off hijacking trades between traders
breed [pred-traders pred-trader]
; wealth is money owned not in stock
; stock is money owned in stock
; wps is the sum of the two
; gain is the difference from initial cost of stock to current cost
; bid-ask is the bid price or ask price for that turtle
traders-own [stock wps gain bid-ask]

turtles-own [wealth]
; avg-wealth is the average wealth of traders
; avg-stock is the average stock of the traders
; num-trades is the total number of trades that have occured
; avg-pred-wealth is the average wealth of pred-traders
; total-wealth is total wealth of the market
globals [avg-wealth avg-stock num-trades avg-pred-wealth total-wealth]

to setup
  clear-all
  ask patches [set pcolor white]
  ifelse not predatory-trading? [
    create-traders number-people [setxy random-xcor random-ycor set shape "circle" set color green]
    set amount-predatory-traders 0 ; make slider reflect no predatory traders
  ]
  [
    ; as the system sets amount-predatory-traders to 0 if the switch is off
    ; reset the slider to default value of 5 in case user forgets to (otherwise there will be errors)
    if amount-predatory-traders = 0 [ set amount-predatory-traders 5 ]
    ; keep number of people same, but make amount-predatory-traders % pred-traders
    create-traders (number-people * (1 - (amount-predatory-traders / 100))) [setxy random-xcor random-ycor set shape "circle" set color green]
    create-pred-traders (number-people * (amount-predatory-traders / 100)) [setxy random-xcor random-ycor set shape "circle" set color red]
  ]
  ask traders [
    set-initial-wealth
    set-initial-stock
    set gain 0
    set wps wealth + stock
  ]
  if predatory-trading? [
    ask pred-traders [
      set-initial-wealth
    ]]
  set avg-wealth person-initial-wealth
  ifelse predatory-trading? ; don't need avg-pred-wealth value for market when they don't exist
  [ set avg-pred-wealth person-initial-wealth ]
  [ set avg-pred-wealth "N/A" ]
  set total-wealth person-initial-wealth * number-people
  set num-trades 0
  reset-ticks
end

to go
  ask traders [
    if wealth > 0 [ ; can't buy stock without money
      buy-stock
      appreciate-stock
      set wps wealth + stock
    ]
  ]

  if one-of links != nobody [ ; get rid of links showing current trades, as they have finished
    ask links [die]
  ]
  get-avg-wealth
  get-avg-stock
  get-total-wealth
  if predatory-trading? [get-avg-pred-wealth]
  tick
end

;; handles the entire buy/sell transaction between 2 traders
to buy-stock ;; turtle procedure
  if random 100 < 10 [
    let amt ((random 15) / 100) * wealth ; amount of stock turtle wishes to purchase
    set bid-ask (random-normal 1 (bid-ask-spread / 10000)) * amt ; set bid price to a small spread around the amount
    let success? false
    ; variables to handle the predatory trading
    let bid-price bid-ask
    let asking-price 0
    let hijacked? false
    let pred-turtle nobody
    ; only want traders with that much stock to sell it - avoid negative stock
    ask other traders with [stock > amt] [
      set bid-ask (random-normal 1 (bid-ask-spread / 10000)) * amt ; set offer price
      ; not success? is to make sure only one turtle gets rid of its stock
      ; the gain checks are if turtle has either made money or lost a good amount from stock
      if random 100 < 10 and not success? and (gain > 0 or gain < -1 * .10 * stock) and bid-ask <= bid-price [
        set asking-price bid-ask ; makes handling scope easier
        ifelse predatory-trading? and amt > .05 * avg-wealth [ ; if the trade is large enough, the predatory traders will intercept
          ask one-of pred-traders [
            create-link-from myself [set color yellow] ; different color to highlight trade hijacking
            set wealth wealth + (bid-price - asking-price) ; predatory traders pocket the spread
            set hijacked? true
            set pred-turtle self
          ]
          set wealth wealth + asking-price ; seller gains money equal to their offer value for the stock
        ]
        [
          set wealth wealth + asking-price + (bid-price - asking-price) ; if there wasn't an interception, seller can benefit from buyer crossing the spread
          create-link-to myself
        ]
        set stock stock - amt ; amt of stock is ultimately sold
        set num-trades num-trades + 1
        set success? true ; trade was successful
      ]
    ]
    ; if seller found, update turtle's wealth and stock
    if success? [
      set gain 0
      set wealth wealth - bid-price ; buyer pays money equal to their bid price, as they are crossing the spread in this model
      if hijacked? [
        create-link-from pred-turtle [set color yellow] ; if trade is intecepted, complete the hijacked trade link
        set num-trades num-trades + 1 ; predatory trader performs 2 trades, first buying, then selling
      ]
      set stock stock + amt ; amt of stock is ultimately purchased regardless of the bid-ask spread
    ]
  ]

end

; this method gives each turtle's stock a different growth rate
; which simulates traders holding different stocks in the market
to appreciate-stock ; turtle procedure
  ; convert growth rate to daily rate
  let growth-difference (market-annual-growth-rate / (100 * 250))
  let growth-rate 1 + growth-difference
  set gain gain - stock ; this subtracts the stock's initial amount from gain
  ; want a normal distribution around the growth rate
  set stock (random-normal growth-rate .02 ) * stock ; .02 is taken from S&P 500 volatility
  set gain gain + stock ; add the new stock amount to complete (new - old) calculation
end


;;; SETUP PROCEDURES ;;;
;; set traders initial wealth to around initial wealth specified
to set-initial-wealth
  set wealth (random-normal person-initial-wealth (.05 * person-initial-wealth))
end

; set initial stock to be a random number around a normal distribution of 30% of wealth
to set-initial-stock
  set stock (random-normal (.3 * wealth) (.1 * wealth))
  set wealth wealth - stock
end


;;; AVERAGE OUTPUT PROCEDURES ;;;
;; get average wealth of traders for display
to get-avg-wealth
  set avg-wealth 0
  ask traders [
    set avg-wealth avg-wealth + wealth + stock
  ]
  set avg-wealth (avg-wealth / (count traders))
end

; get average wealth of predatory traders
to get-avg-pred-wealth
  set avg-pred-wealth 0
  ask pred-traders [
    set avg-pred-wealth avg-pred-wealth + wealth
  ]
  set avg-pred-wealth avg-pred-wealth / (count pred-traders)
end

; get average stock for each trader
to get-avg-stock
  set avg-stock 0
  ask traders [
    set avg-stock avg-stock + stock
  ]
  set avg-stock (avg-stock / (count traders))
end

; get total wealth in the market
to get-total-wealth
  set total-wealth 0
  ask traders [
    set total-wealth total-wealth + wealth + stock
  ]
  if predatory-trading? [
    ask pred-traders [
      set total-wealth total-wealth + wealth
  ]]
end

; report the average wealth of all agents
to-report avg-total-wealth
  let total 0
  ask traders [
    set total total + wealth + stock
  ]
  if predatory-trading? [
    ask pred-traders [
      set total total + wealth
    ]]
  ifelse predatory-trading? [
    set total total / ((count traders) + (count pred-traders))
  ]
  [
    set total total / (count traders)
  ]
  report total
end
@#$#@#$#@
GRAPHICS-WINDOW
264
10
703
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
days
30.0

SLIDER
15
12
224
45
number-people
number-people
50
200
100
1
1
NIL
HORIZONTAL

BUTTON
15
210
225
243
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
15
250
225
283
market-annual-growth-rate
market-annual-growth-rate
-4
8
2.5
.1
1
%
HORIZONTAL

BUTTON
15
290
226
323
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
15
50
224
83
person-initial-wealth
person-initial-wealth
100
500
150
1
1
NIL
HORIZONTAL

PLOT
728
11
1113
271
Global wealth
Days
Wealth
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"avg wealth" 1.0 0 -16777216 true "" "plot avg-total-wealth"
"avg stock" 1.0 0 -14439633 true "" "plot avg-stock"

MONITOR
15
340
104
385
Trades
num-trades
1
1
11

PLOT
725
285
1115
435
Wealth of traders
Wealth
Traders
0.0
500.0
0.0
10.0
true
true
"" ""
PENS
"Pred Traders" 5.0 1 -2674135 true "" "histogram [ wealth ] of pred-traders"
"Traders" 5.0 1 -7500403 true "" "histogram [ wps ] of traders"

SWITCH
15
90
225
123
predatory-trading?
predatory-trading?
0
1
-1000

SLIDER
15
170
225
203
bid-ask-spread
bid-ask-spread
0
100
100
1
1
Pts
HORIZONTAL

MONITOR
140
340
222
385
Trades/Day
num-trades / ticks
2
1
11

MONITOR
15
400
130
445
Avg Trader Wealth
avg-wealth
2
1
11

MONITOR
140
400
252
445
Avg Pred Wealth
avg-pred-wealth
2
1
11

MONITOR
15
455
102
500
NIL
total-wealth
1
1
11

SLIDER
15
130
225
163
amount-predatory-traders
amount-predatory-traders
0
50
5
1
1
%
HORIZONTAL

MONITOR
140
455
222
500
Wealth Gain
total-wealth - (person-initial-wealth * number-people)
2
1
11

@#$#@#$#@
## WHAT IS IT?

This project models the behavior of a stock market with predatory trading occuring in the market. Normal traders will hold stock, and try to buy and sell stock with others. Predatory traders try to get in between a buyer and a seller to take advantage of the bid-ask spread.

## HOW IT WORKS

In the model, traders have different levels of wealth and stock. Stock is appreciated for each turtle on each day based on a random-normal value around the annual growth rate (converted to daily growth rate).

Trader rules followed in a single tick:

  * If I have any wealth, I consider buying stock
  * If I decide to buy stock, I decide the amount of stock I want to buy, and the price at which I am willing to buy it
  * Try to find another trader that has that amount of stock, and have that trader set the price at which they are willing to sell
  * If I am the seller, I only sell if I have already made money on the stock or lost at least 10%, and if the bid price is higher than my ask price, I will sell the stock
  * If a trade takes place, create a grey link between the two traders

Predatory Trader rules followed in a single tick:

  * If the amount of stock being traded is high enough, I will hijack the trade, gaining the spread between the bid and the ask price as profit, and allowing the trade to be completed
  * If I hijack a trade, create two yellow links, one between the buyer and me, and one between the seller and me


## HOW TO USE IT

The Setup button sets up the model, using the current values of the sliders. It changes patch color for a white background, resets ticks and global variables, and creates agents spread out randomly across the world.

The Go button starts the model, and causes agents to begin trying to trade with each other.

The number-people slider adjusts how many people are in the model. Select the value before clicking Setup.

The market-annual-growth-rate is how much stock is expected to grow, on average, in a year. Select the value before clicking Setup.

The person-initial-wealth slider adjusts how much wealth each person starts with. A person will start out with a small random amount of stock taken from this initial wealth value. Select the value before clicking Setup.

The predatory-trading? switch determines whether there are predatory traders in the market or not. Select the value before clicking Setup.

The amount-predatory-traders slider adjusts what percentage of people in the market are predatory traders. Select the value before clicking Setup.

## THINGS TO NOTICE

Notice how the average wealth of predatory traders increases. Follow the histogram as well as the monitor to see how the process is riskless

Notice how the wealth of regular traders quickly becomes distributed. See the skewed right distribution quickly form once trading begins.

Notice how after long enough growth, the market can start to experience some large swings in growth.

## THINGS TO TRY

The market-annual-growth-rate sets the annual growth rate of stock in the market. Try setting the value initially, and adjusting the slider as the model runs.

Adjust the amount-predatory-traders value to see how a saturation of predatory traders affects the market.

Adjust the bid-ask-spread to see how a tighter spread can affect profits for predatory trading, and reduction of wealth in the market.

Adjust the person-initial-wealth slider to see how smaller amounts of wealth can lead to smaller negative effects from predatory trading.

Run the model both with predatory-trading? on and off. Do you notice a difference after 1200 ticks, 2500?

## EXTENDING THE MODEL

Right now the model focuses on the basic effect of predatory trading on the basic market. The model doesn't have a way of increasing or decreasing stability of the market, so it could be extended to include this.

In addition, this model could be extended to include random shocks to the market, such as recessions or booms. This would make the market more accurate, and allow observation of predatory trading during these specialized periods. Currently, only manually changing the market-annual-growth-rate slider during running can simulate this

## NETLOGO FEATURES

The most interesting netlogo feature used for me was the random-normal function, a built-in way to get a random number from a normal distribution with specified mean and standard deviation. I had no idea Netlogo had this functionality, and the model didn't work nearly as well without it. Bryan Head guided me to this feature.

One more interesting feature is the ability to change global slider variables in the code, and have it be reflected in the interface. I didn't realize that altering a variable in the Setup code would actually show the updated value in the slider without the user adjusting the value manually.

## RELATED MODELS

Exponential Growth
Wealth Distribution

## CREDITS AND REFERENCES

Inspiration for this model, as well as background information on high frequency trading from _Flash Boys_ by Michael Lewis.

Volatility value from http://finance.yahoo.com/q?s=%5EVIX and was accurate for day of writing.

This model would not have been possible without the help of Dr. Uri Wilensky, Bryan Head, Arthur Hjorth, and Elham Beheshti

Link to model: http://modelingcommons.org/browse/one_model/4663
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 6.0-M5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="test" repetitions="50" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2500"/>
    <metric>num-trades</metric>
    <metric>avg-wealth</metric>
    <metric>avg-pred-wealth</metric>
    <metric>total-wealth</metric>
    <enumeratedValueSet variable="predatory-trading?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bid-ask-spread">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-people">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="amount-predatory-traders">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="person-initial-wealth">
      <value value="150"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="market-annual-growth-rate">
      <value value="2.5"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
