;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global variables ;;;
;;;;;;;;;;;;;;;;;;;;;;;;
globals
[
  tie_counter         ;; when adding random ties, this varibale monitors the number of random ties already added
  random_agent_1      ;; used when one or more agents are randomly selected
  random_agent_2      ;; used when one or more agents are randomly selected
  links_in_network    ;; sets the number of links in the network
  iteration           ;; sets the number of iterations to a maximum (in case of falling into a cycle).
  nash_network        ;; used when the action profile of all agents is and equilibrium best response
]

;;;;;;;;;;;;;;;;;;;;;;;
;;; Types of agents ;;;
;;;;;;;;;;;;;;;;;;;;;;;
breed [bops bop]      ;; bop agents prefer A=0
breed [bips bip]      ;; bip agents prefer A=1

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Turtle variables ;;;
;;;;;;;;;;;;;;;;;;;;;;;;
turtles-own
[
  action                  ;; agent's choice (1 or 0)
  threshold_high          ;; agent's threshold to choose their disliked_action, as a function of their number of connections
  threshold_low           ;; agent's threshold to choose their liked_action, as a function of their number of connections
  alpha                   ;; payoff earned for coordinating in the liked_action
  beta                    ;; payoff earned for coordinating in the disliked_action 
  tie_check               ;; auxiliary variable that is needed to check whether two agents maintain a link
  set_ties                ;; indicates the set of links an agent maintains
  total_ties              ;; indicates the cardinality of the set of links an agent maintains: agent's degree
  set_A1                  ;; set of agents who choose A=1
  set_A0                  ;; set of agents who choose A=0 
  A0_neighbors            ;; set of neighbors who choose A=1, for the focal agent
  A1_neighbors            ;; set of neighbors who choose A=1, for the focal agent
  total_A0_neighbors      ;; cardinality of the set of neighbors A_0
  total_A1_neighbors      ;; cardinality of the set of neighbors A_1
  liked_action            ;; this is the choice that gives higher payoffs to the focal agent
  disliked_action         ;; this is the choice that gives lower payoffs to the focal agent
  prob_like_first         ;; probability of liking the first choice
  happy                   ;; variable that shows if an agent's choice == liked_choice
  nash                    ;; variable that shows if an agent's choice is stable when fixing the choices of all her neighbors
  set_nash                ;; this is the set of agents that don't want to change their choice
  total_set_nash          ;; this is the cardinality of the nash set
]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setting up the world ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to setup
  clear-all               ;; clears the world   
  reset-ticks             ;; ticks start from 0 after every setup
  setup-link_number       ;; calculates the number of links in the network
  setup-bops              ;; creates the bops
  setup_bips              ;; creates the bips
  make_network            ;; sets the random formation prior to the coordination game
  setup-liked_action      ;; sets the value of the favorite action for each breed
  setup-disliked_action   ;; sets the value of the non-favorite action for each breed
  setup-first_choice      ;; sets the random initial choice agents make
  set iteration 0         ;; sets the iteration counter: relevant for the length of run
  set nash_network 0      ;; sets the nash network variable
  setup_color             ;; sets the initial color of the turtles
end

to setup-link_number  
  set links_in_network ((proportion-links * ((number_nodes * (number_nodes - 1)) / 2 )) / 100)
end
;; The initial number of links is given by the input of the proportion-links slider

to setup-bops
  set-default-shape bops "circle"
  create-bops (number_nodes)
  [
  set size 1
  ]
end
;; All nodes created are initially bops

to setup_bips
  if number_nodes = 0 [stop]   
  if number_bips > number_nodes [stop] 
  set-default-shape bips "triangle"
  loop [                       
  if (number_bips > 0) [      
    ask one-of turtles [ set breed bips]
  ]
  if (count bips = number_bips) [stop] 
  ]
  end
;; if there are no nodes in the network there can be no bips
;; if the number of bips to be created is greater than the number of nodes, all nodes will remain as bops
;; the loop process randomly chooses a node to change her breed from bop to bip
;; the loop process starts only if at least one bip is to be created, as given by the input from the number_bips slider
;; the process stops when the number of bips is the one selected in the setup


to make_network
  repeat links_in_network [ layout-spring turtles links 0.4 6 1]  
  set tie_counter 0            
  while [tie_counter < links_in_network] 
  [
    set random_agent_1 0 
    set random_agent_2 0
    ask turtle random_agent_1 [ask link-neighbors [set tie_check 1]]
    while [(random_agent_1 = random_agent_2) or ([tie_check] of turtle random_agent_2 = 1)] 
    [                                                                                       
      ask turtle random_agent_1 [ask link-neighbors [set tie_check 0]] 
      set random_agent_1 random number_nodes    
      set random_agent_2 random number_nodes
      ask turtle random_agent_1 [ask link-neighbors [set tie_check 1]] 
    ]
    ask turtle random_agent_1 [create-link-with turtle random_agent_2]
    set tie_counter tie_counter + 1  
  ]
  if (links_in_network = 0) [layout-circle (sort turtles) max-pxcor - 2  ] 
  ask turtles [set set_ties link-neighbors]      
  ask turtles [set total_ties count set_ties]
  ask turtles [
    set alpha 3       
    set beta 2      
  ]
end
;; the spring-out layout arranges the setup given the links, it is ignored if layout-circle is set
;; the tie_counter = 0 sets that initially there are no links in the network
;; the process of link creatiion is repeated as long as there are less than the proportion of links chosen in the setup
;; there are two random agents chosen to create a link between them. The two random agents are initially agent 0
;; if there is a link between two agents the tie_check gives value 1
;; if either both random_agents are the same or are already linked, they are skipped from the formation process
;; for each agent the count of links (link-neighbors) starts in 0 
;; Pairs of agents are randomly selected from the entire population (number_nodes) and if they are not connected an undirected link is formed
;; and the tie_counter of links is updated, by adding a new link into it. 
;; The update will stop when the total number of links counted is the one setup
;;
;; If there are no links to be formed, the layout-circle locates all agents on a circle
;;
;; The set_ties is the set of neighbors for an agent
;; The total_ties is the cardinality of the neighborhood
;; Alpha is set to 3 and beta is set to 2

  
to setup-liked_action  
  ask turtles [
  ifelse breed = bops
  [set liked_action 0] 
  [set liked_action 1]
  ]
end
;; this is the parameter for liked actions given an agents breed
;; bops like action 0 and bips like action 1

to setup-disliked_action 
  ask turtles [
  ifelse breed = bops
  [set disliked_action 1]  
  [set disliked_action 0]
  ]
end
;; this is the parameter for disliked actions given an agents breed
;; bops dislike action 1 and bips dislikeaction 0

to setup-first_choice    
  ask turtles [
    set prob_like_first random 100   
    ifelse prob_like_first < I_like_first_choice 
    [set action liked_action]
    [set action disliked_action]
    ]                        
end
;; this setups the choice agents start with (prior to "go")
;; First, it sets a random number between 0 and 100 as the probability to like the first choice
;; if the probability set is lower than the probability set as an input in the I_like_first_choice slider, 
;; an the focal agent starts choosing the action she likes. This is repeated for all agents
;; OJO: given the inequality/equality in 0 or 100% there is still a chance for liked/disliked action????

to setup_color
  ask bops [
    ifelse action = liked_action
    [set color pink]  
    [set color cyan]  
  ]
  ask bips [
    ifelse action = liked_action
    [set color cyan]
    [set color pink]
  ]
end
;; this sets the initial color of the agents given their initial choice
;; action 0 is assigned color pink (independently of who is choosing it)
;; action 1 is assigned color cyan (independently of who is choosing it)


;;;;;;;;;;;;;;;;;;;;;;
;;; Here starts GO ;;;
;;;;;;;;;;;;;;;;;;;;;;
to go
  if not any? turtles [stop]            ;; if there are no nodes in the network the model stops
  if ticks >= max_iterations [stop]     ;; if the maximum_iterations is reached the model stops
    set_threshold_high                  ;; sets the threshold_high for every agent
    set_threshold_low                   ;; sets the threshold_low for every agent
    set_A1_choice                       ;; makes a set of turtles choosing action 1
    set_A0_choice                       ;; makes a set of turtles choosing action 0
    set_happy                           ;; makes a set of turtles choosing the action they like
    set_A0_neighbors                    ;; makes a set of neighbors choosing action 0, for the focal agent
    set_A1_neighbors                    ;; makes a set of neighbors choosing action 1, for the focal agent
    check_nash                          ;; sets if an agent's choice is stable (nash)
    set_nash_set                        ;; makes a set of agents whose choice is nash
    report_nash                         ;; displays a label 1 if an agent is nash, and 0 otherwise
    check_nash_network                  ;; checks wether all agents are in the nash set (1) or not (0)
  if nash_network = 1 [stop]            ;; if all agents are in the nash set the model stops
    choose_action                       ;; agents that are not in nash change their action
    set_color                           ;; the color is set of all agents given their choice
  tick                                  ;; a new tick (period of time) runs
end

to set_threshold_high
  ask turtles [
    ifelse links_in_network = 0
    [set threshold_high 0 ]
    [ifelse total_ties = 0
      [set threshold_high 0]
    [set threshold_high (((total_ties * alpha) + alpha - beta ) / (alpha + beta ))]
    ]
  ]
end
;; The threshold_high is a function of an agents connections. It determines the tipping point to choose the disliked action.

to set_threshold_low
  ask turtles [
    ifelse links_in_network = 0
    [set threshold_low 0 ]
    [ifelse total_ties = 0
      [set threshold_low 0]
    [set threshold_low (((total_ties * beta) - alpha + beta ) / (alpha + beta ))]
    ]
  ]
end
;; The threshold_low is a function of an agents connections. It determines the tipping point to choose the liked action.

to set_A1_choice
  ask bops [
    ifelse action = liked_action
    [set set_A1 0]
    [set set_A1 1]
  ]
  ask bips [
  ifelse action = liked_action
  [set set_A1 1]
  [set set_A1 0]
]
end
;; The set_A1 is composed by all agents choosing action 1. It is defined given the agents' breed and liked/disliked actions

to set_A0_choice
  ask bops [
    ifelse action = liked_action
    [set set_A0 1]
    [set set_A0 0]
  ]
  ask bips [
  ifelse action = liked_action
  [set set_A1 1]
  [set set_A1 0]
]
end
;; The set_A0 is composed by all agents choosing action 0. It is defined given the agents' breed and liked/disliked actions

to set_happy
  ask turtles [
    ifelse action = liked_action
    [set happy 1]
    [set happy 0]
  ]
end
;; The happy is composed by all agents choosing the action they like

to set_A0_neighbors
  ask turtles [
    set A0_neighbors link-neighbors with [set_A0 = 1]
    set total_A0_neighbors count A0_neighbors
  ]
end
;; The set_A0_neighbors is composed by all neighbors of the focal agents choosing action 0. Its cardinality is total_A0_neighbors

to set_A1_neighbors
  ask turtles [
    set A1_neighbors link-neighbors with [set_A1 = 1]
    set total_A1_neighbors count A1_neighbors
  ]
end
;; The set_A1_neighbors is composed by all neighbors of the focal agents choosing action 1. Its cardinality is total_A1_neighbors

to check_nash
  ask turtles[
  ifelse complements?
  [
  ask bops[
    ifelse action = liked_action  
    [ifelse total_A0_neighbors >= threshold_low
      [set nash 1]
      [set nash 0]
    ]
    [ifelse total_A1_neighbors > threshold_high
      [set nash 1]
      [set nash 0]
    ]
  ]
  ask bips[
    ifelse action = liked_action  
    [ifelse total_A1_neighbors >= threshold_low
      [set nash 1]
      [set nash 0]
    ]
    [ifelse total_A0_neighbors > threshold_high
      [set nash 1]
      [set nash 0]
    ]
  ]
  ]
   [
  ask bops[
    ifelse action = liked_action  
    [ifelse total_A1_neighbors >= threshold_low
      [set nash 1]
      [set nash 0]
    ]
    [ifelse total_A0_neighbors > threshold_high
      [set nash 1]
      [set nash 0]
    ]
  ]
  ask bips[
    ifelse action = liked_action  
    [ifelse total_A0_neighbors >= threshold_low
      [set nash 1]
      [set nash 0]
    ]
    [ifelse total_A1_neighbors > threshold_high
      [set nash 1]
      [set nash 0]
    ]
  ]
  ]
  ]
  
  end
;; An agent is nash depending on the relation between her neighbors' choices and the thresholds.
  
to set_nash_set
  ask turtles [ 
    set set_nash turtles with [nash = 1]
    set total_set_nash count set_nash
  ]
end
;; All agents that are nash are part of the nash set. The cardinality of the set is total_set_nash

to report_nash
  ask turtles[set label nash]
end
;; This displays a label (1) on top if every agent when nash, and 0 otherwise
    
 to check_nash_network
   ask turtles [
   ifelse total_set_nash = number_nodes
   [set nash_network 1]
   [set nash_network 0]
   ]
 end  
;; The nash_network takes value 1 if all agents are part of the nash set. If at least one agent is not nash, it takes value 0 
    
to choose_action
  ask turtles[
    ifelse action = liked_action
    [ifelse nash = 1
      [set action liked_action]
      [set action disliked_action]
    ]
    [ifelse nash = 1
      [set action disliked_action]
      [set action liked_action]
    ]
  ]
end
;; Agents that are not in the nash set change their action. Agents in the nash set keep their same choice as in the previous period.

to set_color
  ask bops [
    ifelse action = liked_action
    [set color pink]
    [set color cyan]
  ]
  ask bips [
    ifelse action = liked_action
    [set color cyan]
    [set color pink]
  ]
end
;; If an agent changed her action her color is updated, otherwise her color is kept the same.

;;;;;;;;;;;;;;;;;;;;;;;;
;;; End of the model ;;;
;;;;;;;;;;;;;;;;;;;;;;;;
    
    
    

 
@#$#@#$#@
GRAPHICS-WINDOW
463
10
930
498
16
16
13.85
1
10
1
1
1
0
0
0
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
145
189
211
222
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
286
189
349
222
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
217
189
280
222
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
41
87
213
120
number_nodes
number_nodes
0
50
30
1
1
NIL
HORIZONTAL

SLIDER
36
137
217
170
number_bips
number_bips
0
50
29
1
1
NIL
HORIZONTAL

SLIDER
242
135
454
168
I_like_first_choice
I_like_first_choice
0
100
50
1
1
%
HORIZONTAL

SLIDER
259
85
431
118
proportion-links
proportion-links
0
100
7
1
1
%
HORIZONTAL

MONITOR
54
242
148
287
formed_links
tie_counter
17
1
11

INPUTBOX
266
10
421
70
max_iterations
100
1
0
Number

MONITOR
159
242
216
287
A0
count turtles with [action = 0]
17
1
11

MONITOR
225
242
282
287
A1
count turtles with [action = 1]
17
1
11

MONITOR
291
242
348
287
happy
count turtles with [happy = 1]
17
1
11

MONITOR
361
243
418
288
nash
count turtles with[nash = 1]
17
1
11

PLOT
243
305
451
470
equilibrium
time
agents
0.0
5.0
0.0
50.0
true
true
"" ""
PENS
"happy" 1.0 0 -14439633 true "" "plot count turtles with [happy = 1]"
"nash" 1.0 0 -955883 true "" "plot count turtles with [nash = 1]"

PLOT
19
303
230
468
Choices
time
agents
0.0
5.0
0.0
50.0
true
true
"" ""
PENS
"A0" 1.0 0 -2064490 true "" "plot count turtles with [action = 0]"
"A1" 1.0 0 -13791810 true "" "plot count turtles with [action = 1]"

SWITCH
56
31
202
64
complements?
complements?
0
1
-1000

@#$#@#$#@
## WHAT IS IT?

This model studies social influence in randomly formed networks for two classes of games. Games with strategic complements where agents want to coordinate in the same action, and games with strategic substitutes, where agents want to anti-coordinate.

Agents in the model can be of two types: bips and bops. Bips are agents that prefer action 1 and are represented as triangles. Bops prefer action 0 and are represented as circles. Independently of the type, if an agent chooses action 1 her color will be cyan, and if 0 her color will be pink.

The first choice made is randomly assigned to players. From there, in every period of time, an agent myopically best responds to the choices made by her neighbors in the previous period. When an agent has no incentives to change her choice, she is in the nash set and will display the number 1, otherwise she displays 0. If all agents reach nash the simulation stops, if not, it will continue until the maximum number of iterations is reached.

Agents that choose the action they prefer are said to be happy. This is used to characterize the final outcome (most likely in equilibrium) as satisfactory if all agents are happy, and frustrated if at least one is not. Also as specialize if all agents choose the same action, or hybrid if both actions coexist.  

## HOW TO USE IT

There are 6 parameters you can decide on before initializing the model. (1) The COMPLEMENTS? switch that sets games as strategic complements if ON and sets games as strategic substitutes if OFF. (2) The NUMER_NODES slider controls the size of the network. (3) The NUMBER_BIPS slider controls how many of the total agents are bips. Note that if the number of bips you choose is greater than the total number of agents, by default all agents will be bops.(4) The PROPORTION_OF_LINKS slider controls the level of connectivity in the network. You cannot decide the exact number of links you want, but the percentage of the total possible links [(n*(n-1))/2]. (5) The I_LIKE_FIRST_CHOICE slider controls the probability that all agents first choice is their preferred action. If 100% all agents will choose with probability 1 the action they like: bips choose 1 and bops choose 0. (6) The MAX_ITERATIONS monitor controls the number of iterations the model will run in case it has not reached equilibrium. Choose the values for the five controllers and then press SETUP.

Pressing the GO ONCE button sets the revision of the first initial choice. To repeatedly make agents revise their choices, press GO.

The FORMED_LINKS monitors shows the exact number of connections formed by the proportion of links you determined. This is a fixed value along the run of the model. A1 and A0 monitors show how many agents choose action 1 or 0 repectively, in each round. Monitors HAPPY and NASH, show how many agents choose the action they like and how many agents are in the nash set and don't want to change their choice in the next round, respectively.

The ACTION and EQUILIBRIUM plots illustrate the entire path of choices 1 and 0 and of happy and nash agents along the run of the model.

## THINGS TO TRY

Let the model run until the end. Does it always stop before the maximum iterations? Does it always reach an equilibrium where all agents are in the nash set?

Run the model again, this time slowly, a step at a time. Watch how agents change their choices. What is happening when the network is highly connected? What is happening when there are no links in the network? 

Run it with a small proportion of bips (or bops). What happens when all agents are identical? What happens when there is a large majority of one type and few agents of the other type? What happens when both types of agents are close to 50%?

## EXTENDING THE MODEL

Right now the probability of any two nodes getting connected to each other is the same. Can you think of ways to making any two nodes more attractive to be connected to each other? Perhaps (1) that the more connected an agent is the more likely to create a new link, or (2) the more likely two nodes to be connected if their types is the same, or (3) or the more likely to nodes to be connected if their type is the opposite, or (4) even that all agents will necessarily have the same number of connections. How would that impact the likelihood of reaching equilibrium? How would that impact the class of equilibria reached?

## RELATED MODELS

See other models in the Networks section of the Models Library.

## CREDITS AND REFERENCES

This model is adapted from:
Hernandez, Penalope, Munoz-Herrera, Manuel, Sanchez, Angel, 2013. Heterogeneous Network Games: Conflicting Preferences. Games and Economic Behavior. Forthcoming.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

person construction
false
0
Rectangle -7500403 true true 123 76 176 95
Polygon -1 true false 105 90 60 195 90 210 115 162 184 163 210 210 240 195 195 90
Polygon -13345367 true false 180 195 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285
Circle -7500403 true true 110 5 80
Line -16777216 false 148 143 150 196
Rectangle -16777216 true false 116 186 182 198
Circle -1 true false 152 143 9
Circle -1 true false 152 166 9
Rectangle -16777216 true false 179 164 183 186
Polygon -955883 true false 180 90 195 90 195 165 195 195 150 195 150 120 180 90
Polygon -955883 true false 120 90 105 90 105 165 105 195 150 195 150 120 120 90
Rectangle -16777216 true false 135 114 150 120
Rectangle -16777216 true false 135 144 150 150
Rectangle -16777216 true false 135 174 150 180
Polygon -955883 true false 105 42 111 16 128 2 149 0 178 6 190 18 192 28 220 29 216 34 201 39 167 35
Polygon -6459832 true false 54 253 54 238 219 73 227 78
Polygon -16777216 true false 15 285 15 255 30 225 45 225 75 255 75 270 45 285

person farmer
false
0
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Polygon -1 true false 60 195 90 210 114 154 120 195 180 195 187 157 210 210 240 195 195 90 165 90 150 105 150 150 135 90 105 90
Circle -7500403 true true 110 5 80
Rectangle -7500403 true true 127 79 172 94
Polygon -13345367 true false 120 90 120 180 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 180 90 172 89 165 135 135 135 127 90
Polygon -6459832 true false 116 4 113 21 71 33 71 40 109 48 117 34 144 27 180 26 188 36 224 23 222 14 178 16 167 0
Line -16777216 false 225 90 270 90
Line -16777216 false 225 15 225 90
Line -16777216 false 270 15 270 90
Line -16777216 false 247 15 247 90
Rectangle -6459832 true false 240 90 255 300

person service
false
0
Polygon -7500403 true true 180 195 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285
Polygon -1 true false 120 90 105 90 60 195 90 210 120 150 120 195 180 195 180 150 210 210 240 195 195 90 180 90 165 105 150 165 135 105 120 90
Polygon -1 true false 123 90 149 141 177 90
Rectangle -7500403 true true 123 76 176 92
Circle -7500403 true true 110 5 80
Line -13345367 false 121 90 194 90
Line -16777216 false 148 143 150 196
Rectangle -16777216 true false 116 186 182 198
Circle -1 true false 152 143 9
Circle -1 true false 152 166 9
Rectangle -16777216 true false 179 164 183 186
Polygon -2674135 true false 180 90 195 90 183 160 180 195 150 195 150 135 180 90
Polygon -2674135 true false 120 90 105 90 114 161 120 195 150 195 150 135 120 90
Polygon -2674135 true false 155 91 128 77 128 101
Rectangle -16777216 true false 118 129 141 140
Polygon -2674135 true false 145 91 172 77 172 101

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
