globals [
  %gay              ;; Percent of homosexual people in the population (set by slider)
  %bi               ;; Percent of bisexual people in the population (determined by slider)
  
  max-coupling-factor       ;; Maximum coupling tendency value
                            ;; Used as an upper bound for generating random chance of coupling
                            ;; (default is scale 0 - 10)
  max-condom-factor         ;; Maximum condom use value
                            ;; Used as an upper bound for generating random chance of using a condom
                            ;; (default is scale 0 - 10)
                            
  average-coupling-tendency ;; Average tendency of a person to couple with another person
  average-commitment        ;; Number of ticks a couple will stay together, on average
  ;average-condom-use       ;; Average frequency that a person uses protection (set by slider)
  
  slider-check-1    ;; Temporary variable to use to check for slider adjustments
                    ;; during the simulation and adjust model appropriately
  infection-chance ;; The chance out of 100 that an infected person will transmit infection during one week of couplehood
  ]

turtles-own [
  male?              ;; true or false for genders M or F
  orientation        ;; gay straight or bi
  
  infected?          ;; If true, the person is infected.
                     
  coupled?           ;; If true, the person is in a sexually active couple.
  partner            ;; The person that is our current partner in a couple.
  couple-length      ;; How long the person has been in a couple.
  
  commitment         ;; How long the person will stay in a couple-relationship.
  coupling-tendency  ;; How likely the person is to join a couple.
  condom-use         ;; The percent chance a person uses protection
                     ;; (determined by slider & normal distribution)
]

;;;
;;; SETUP PROCEDURES
;;;

to setup
  clear-all
  setup-globals
  setup-people
  reset-ticks
end

to setup-globals
  
  set max-coupling-factor 10.0
  set max-condom-factor 11.0
  
  set slider-check-1 average-condom-use
  
  ;; In the AIDS model, these variables are set with sliders
  ;; For simplicity, we set them to predetermined values
  ;; Individual values in each turtle are still set using 
  ;; a random value following a normal distribution
  
  set average-coupling-tendency 5 ;should be less than max-coupling-factor
  set average-commitment 20
  set infection-chance 50 ;; %50 chance of being infected by having unprotected sex with infected partner
end


to setup-people
  crt initial-people       ;; create the specified initial number of people
    [ 
      set male? true                ;; default person straight male... just like society -_-
      set orientation "straight"    ;; set sexual orientation
      setxy random-xcor random-ycor ;; place every person in a random position
      set coupled? false
      set partner nobody
      set infected? false 

      ;; The below variables may be different in each turtle, and the values
      ;; follow an approximately normal distribution
      assign-commitment
      assign-coupling-tendency
      assign-condom-use
      ]
    
   ;; set gender of turtles to be 50% male, 50% female
   
   ask n-of (initial-people / 2) turtles [set male? false ]
   
   ;; set orientations of turtles based on slider values
   
   set %gay (100 - %straight) * (%gay-in-pop / 100)
   set %bi (100 - %straight) * (1 - %gay-in-pop / 100)
  
   ask n-of (initial-people * (%bi / 100)) turtles [set orientation "bi"] 
   ;; when setting the second orientation, make sure to change only formerly straight turtles
   ask n-of (initial-people * (%gay / 100)) turtles with [orientation = "straight"] [set orientation "gay"]
  
   
   ask turtles [
     assign-color    ;; color is determined by orientation
     assign-shape    ;; shape is determined by gender and sick status
     set size 1.5
     ]
end


;; Color of turtle indicates their mate gender preference
;; i.e. likes men is blue, likes women is pink, both is purple

to assign-color  ;; turtle procedure
  if likesboys? [ set color blue]
  if likesgirls? [ set color pink]
  if likesboys? and likesgirls? [set color violet]
end


 ;; Set shape based on gender (male or female)
 ;; and whether or not infected (includes a red dot)
 
to assign-shape ;; turtle procedure
  ifelse infected?
  [ ifelse male?
    [set shape "male sick"]
    [set shape "female sick"]
  ]
  [ ifelse male?
    [set shape "male"]
    [set shape "female"]
  ]
end



;;;
;;; SELECT PROCEDURE
;;;
;; User chooses an initial patient zero

to select
  let picked? false
   if mouse-down? [
    let candidate min-one-of turtles [distancexy mouse-xcor mouse-ycor]
    if [distancexy mouse-xcor mouse-ycor] of candidate < 1 [
        ask candidate [ 
          set shape word shape " sick"
          display
          set picked? true
          set infected? true ] ] ]
  if picked?
  [stop]
  
end 


;; People move about the simulation at random.

to move  ;; turtle procedure
  rt random-float 360
  fd 1
end


;;;
;;; GO PROCEDURES
;;;

to go
  ;; stop when % of infected people reaches specified limit
  ;; allows user to examine what genders/orientations are last to be infected
  if (%infected > stop%) [ stop ] 
     
  check-sliders ;; check if any sliders impacting behavior have been changed
  
  ask turtles
    [ ifelse coupled?
        [ set couple-length couple-length + 1 ]
        [ move ] ;; move if not coupled
    ]

   ;; Any turtle can initiate mating if they are not coupled
   ;; (and random chance permits)
  ask turtles
    [ if not coupled? and (random-float max-coupling-factor < coupling-tendency)
        [ couple ] ]
  ask turtles [ uncouple ]
  ask turtles [ infect ]
  tick
end



;; On each tick a check is made to see if sliders have been changed.
;; If one has been, the corresponding turtle variable is adjusted

to check-sliders
  
  if (slider-check-1 != average-condom-use)
    [ ask turtles [ assign-condom-use ]
      set slider-check-1 average-condom-use ]
end


;; The following procedure assigns core turtle variables.  They use
;; the helper procedure RANDOM-NEAR so that the turtle variables have an
;; approximately "normal" distribution around the average values.

to assign-commitment  ;; turtle procedure
  set commitment random-near average-commitment
end

to assign-coupling-tendency  ;; turtle procedure
  set coupling-tendency random-near average-coupling-tendency
end

to assign-condom-use  ;; turtle procedure
  set condom-use random-near average-condom-use
end


;; Helper procedure to approximate a "normal" distribution
;; around the given average value

;; Generate many small random numbers and add them together.
;; This produces a normal distribution of tendency values.
;; A random number between 0 and 100 is as likely to be 1 as it is to be 99.
;; However, the sum of 20 numbers between 0 and 5 is much more likely to be 50 than it is to be 99.

to-report random-near [center]  ;; turtle procedure
  let result 0
  repeat 40
    [ set result (result + random-float center) ]
  report result / 20
end



;;;
;;; COUPLING/UNCOUPLING/INFECTING PROCEDURES
;;;


;; People have a chance to couple depending on their orientation,
;; their tendency to couple/have sex, and if they meet.
;; To better show that coupling has occurred, the patches below
;; the couple turn gray.

to couple  ;; turtle procedure 
  let potential-partner one-of (turtles-at -1 0) with [not coupled?] 
  if potential-partner != nobody
    [ 
      ;; check for sexual orientation compatibility 
      
      if ( ((male? and [likesboys?] of potential-partner) or 
            (not male? and [likesgirls?] of potential-partner ))      ;; if the partner is willing to mate
        and ( (likesboys? and [male?] of potential-partner) or 
              (likesgirls? and not [male?] of potential-partner ) ) ) ;; if the person in question is willing to mate
      [ 
      ;; normal coupling probability
      
      if random-float max-coupling-factor < [coupling-tendency] of potential-partner
      [ set partner potential-partner
        set coupled? true
        ask partner [ set coupled? true ]
        ask partner [ set partner myself ]
        ask partner [ set pcolor gray - 3]
        move-to patch-here ;; move to center of patch
        ask partner [move-to patch-here] ;; partner moves to center of patch
        set pcolor gray - 3 ]]
      ]
end

;; If two persons are together for longer than either person's 
;; commitment variable allows, the couple breaks up.

to uncouple  ;; turtle procedure
  if coupled? 
    [ if (couple-length > commitment) or
         ([couple-length] of partner) > ([commitment] of partner)
        [ set coupled? false
          set couple-length 0
          ask partner [ set couple-length 0 ]
          set pcolor black
          ask partner [set pcolor black]
          ask partner [ set partner nobody ]
          ask partner [ set coupled? false ]
          set partner nobody ] ]
end


;; Note that for condom use to occur, both people must want to use one.  If
;; either person chooses not to use a condom, infection is possible.  Changing the
;; primitive to AND in the third line will make it such that if either person
;; wants to use a condom, infection will not occur.

to infect  ;; turtle procedure
  if coupled? and infected? 
    [ if random-float max-condom-factor > condom-use or
         random-float max-condom-factor > ([condom-use] of partner)
        [ if random-float 100 < infection-chance
            [ ask partner [ set infected? true
                assign-shape
                ]  ]  ]  ]
end



;;;
;;; REPORTER PROCEDURES
;;;

;; turtle procedures
to-report likesboys?
  ifelse ( (male? and orientation = "straight")
           or (not male? and orientation = "gay") )
  [report false] ;; straight men and lesbians don't like men
  [report true]  ;; everyone else likes men
end

to-report likesgirls?
  ifelse ( (male? and orientation = "gay")
           or (not male? and orientation = "straight") )
  [report false] ;; gay men and straight women don't like women
  [report true]  ;; everyone else likes women
end



;;;
;;; MONITOR PROCEDURES
;;;

to-report %infected
  ifelse any? turtles
    [ report (count turtles with [infected?] / count turtles) * 100 ]
    [ report 0 ]
end

to-report %straight-infected
  ifelse any? turtles with [orientation = "straight"]
  [ report 100 * (count turtles with [infected? and orientation = "straight"]) / (count turtles with [orientation = "straight"]) ]
  [ report 0 ]
end

to-report %bi-infected
  ifelse any? turtles with [orientation = "bi"]
  [ report 100 * (count turtles with [infected? and orientation = "bi"]) / (count turtles with [orientation = "bi"]) ]
  [ report 0 ]
end

to-report %gay-infected
  ifelse any? turtles with [orientation = "gay"]
  [ report 100 * (count turtles with [infected? and orientation = "gay"]) / (count turtles with [orientation = "gay"]) ]
  [ report 0 ]
end

to-report %F-infected
  ifelse any? turtles with [not male?]
  [ report 100 * count turtles with [infected? and not male?] / count turtles with [not male?] ]
  [ report 0 ]
end

to-report %M-infected
  ifelse any? turtles with [male?]
  [ report 100 * count turtles with [infected? and male?] / count turtles with [male?] ]
  [ report 0 ]
end


;;
;; Reporters for each combination of male/female gay/straight/bi
;; Not currently on display in the model (to avoid information overload),
;; but readily available if the user wishes to add monitors
;; to view additional demographic information
;;

to-report %M-straight-infected
  ifelse any? turtles with [male? and orientation = "straight"]
  [ report 100 * count turtles with [infected? and male? and orientation = "straight"] / count turtles with [male? and orientation = "straight"]]
  [ report 0 ]
end

to-report %F-straight-infected
  ifelse any? turtles with [not male? and orientation = "straight"]
  [ report 100 * count turtles with [infected? and not male? and orientation = "straight"] / count turtles with [not male? and orientation = "straight"]]
  [ report 0 ]
end

to-report %M-bi-infected
  ifelse any? turtles with [male? and orientation = "bi"]
  [ report 100 * count turtles with [infected? and male? and orientation = "bi"] / count turtles with [male? and orientation = "bi"]]
  [ report 0 ]
end

to-report %F-bi-infected
  ifelse any? turtles with [not male? and orientation = "bi"]
  [ report 100 * count turtles with [infected? and not male? and orientation = "bi"] / count turtles with [not male? and orientation = "bi"]]
  [ report 0 ]
end

to-report %M-gay-infected
  ifelse any? turtles with [male? and orientation = "gay"]
  [ report 100 * count turtles with [infected? and male? and orientation = "gay"] / count turtles with [male? and orientation = "gay"]]
  [ report 0 ]
end

to-report %F-gay-infected
  ifelse any? turtles with [not male? and orientation = "gay"]
  [ report 100 * count turtles with [infected? and not male? and orientation = "gay"] / count turtles with [not male? and orientation = "gay"]]
  [ report 0 ]
end
@#$#@#$#@
GRAPHICS-WINDOW
287
10
722
466
12
12
17.0
1
10
1
1
1
0
1
1
1
-12
12
-12
12
1
1
1
weeks
30.0

BUTTON
9
277
92
310
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
10
363
93
396
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
738
413
821
458
% infected
%infected
2
1
11

SLIDER
7
10
276
43
initial-people
initial-people
50
500
100
1
1
NIL
HORIZONTAL

SLIDER
9
455
278
488
average-condom-use
average-condom-use
0
10
0
1
1
NIL
HORIZONTAL

MONITOR
791
79
863
124
# females
count turtles with [not male?]
17
1
11

MONITOR
896
79
960
124
# males
count turtles with [male?]
17
1
11

MONITOR
729
136
855
181
# infected straight
count turtles with [infected? and orientation = \\"straight\\"]
17
1
11

MONITOR
862
136
951
181
# infected bi
count turtles with [infected? and orientation = \\"bi\\"]
17
1
11

MONITOR
960
135
1056
180
# infected gay
count turtles with [infected? and orientation = \\"gay\\"]
17
1
11

PLOT
735
225
1050
405
Percentage of Populations Infected
weeks
percentage
0.0
10.0
0.0
100.0
true
true
"" ""
PENS
"Straight" 1.0 0 -2674135 true "" "plot %straight-infected"
"Bi" 1.0 0 -8630108 true "" "plot %bi-infected"
"Gay" 1.0 0 -13840069 true "" "plot %gay-infected"

BUTTON
358
472
452
505
hide infected
ask turtles[if infected? [hide-turtle]]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
579
473
670
506
show infected
ask turtles [if infected? [show-turtle]]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
8
207
252
240
stop%
stop%
0
100
90
1
1
NIL
HORIZONTAL

MONITOR
828
413
904
458
% F infected
%F-infected
2
1
11

MONITOR
912
413
990
458
% M infected
%M-infected
2
1
11

SLIDER
8
118
252
151
%gay-in-pop
%gay-in-pop
0
100
50
1
1
NIL
HORIZONTAL

SLIDER
7
47
252
80
%straight
%straight
0
100
50
1
1
NIL
HORIZONTAL

BUTTON
11
317
78
350
NIL
select
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
93
320
243
348
Select one individual to have an STD
11
0.0
1

TEXTBOX
8
86
290
123
Out of the non-heterosexual population, indicate what percentage will be homosexual (vs. bisexual)
11
0.0
1

MONITOR
178
156
267
201
% homosexual
(100 - %straight) * (%gay-in-pop / 100)
2
1
11

MONITOR
91
156
168
201
% bisexual
(100 - %straight) * (1 - %gay-in-pop / 100)
2
1
11

MONITOR
9
155
79
200
NIL
%straight
2
1
11

TEXTBOX
11
242
212
273
Simulation will stop running once this percentage is infected
11
0.0
1

TEXTBOX
337
507
487
535
Once stopped, show who has not been infected
11
0.0
1

TEXTBOX
551
512
701
530
Show the entire population
11
0.0
1

@#$#@#$#@
## WHAT IS IT? 

This model simulates the spread of sexually transmitted infections (STIs) throughout a population. These populations can include members of any sexual orientation; therefore, this model serves to demonstrate how an infection/disease can permeate through a population even if it only starts with one person (regardless of the sexual orientation or gender of that initial _patient zero_). 

Although STIs may be transmitted through avenues other than sexual behavior, as in drug needles, childbirth, or breastfeeding, this model focuses on the sexual interactions, as they are most common form of transmission. 

The model examines the effects of infection transmission through different demographic percentages. The user can select the size of the initial population, the initial percentage of heterosexual people, and the initial percentage of homosexual people (the remainder of the population will be bisexual). In doing so, users can observe any potential patterns that emerge with different populations. 

## HOW IT WORKS 

People in this model are color coded to indicate the gender of their preferred partner. Blue people are attracted to males, pink people are attracted to females, and purple people are attracted to either/both. As the individuals move around randomly, they may come across a potential sexual partner. If both partners are attracted to the gender of the other partner, there is a chance these two will have a sexual relationship; it is not guaranteed. If a couple is formed, they stay next to each other (stationary) for a pre-determined length of time and their respective patches turn gray in order to more easily identify them. If one partner in the couple is infected, the other partner is at risk for infection if both partners do not use condoms. 

An infected person is denoted with the addition of a red dot on their body. If they happen to infect their partner, the partner will then have a red dot as well. The sexual relationship lasts for a limited time, and the two individuals move randomly about once again until they find another potential partner. 

The model stops when the desired percentage of the population is infected. 

## HOW TO USE IT 

Using sliders, determine the initial population, including percentages of representations of different sexual orientations. Users also determine the likelihood of an individual using a condom and the point at which the model should stop. Indicating a stop percentage allows the user to see who has (or has not) been infected after the determined percentage has been reached. 

The SETUP button will generate a population based on the user_s determined values. Once the population is initialized, the user presses the SELECT button and clicks on an individual in the simulation to infect with a sexually transmitted infection. This allows the user to observe any patterns that arise from an initial _patient zero_. 

To start the simulation, the user should press the GO button. The simulation will run until the GO button is pressed again or the determined stop-percentage has been met, whichever happens first. 

Monitors indicate the percentage of the total population that has been infected, as well as counts and percentages for some demographics. The graph shows the percentage of each sexual orientation that is infected.

Once the simulation has reached a stopping point, whether by pressing GO a second time or the stop-percentage has been met, the user can press the HIDE INFECTED button. This will only display the remaining members of the population who have yet to be infected. This information could be used when noticing any possible patterns. To return to the entire population, select the SHOW INFECTED button. 

The user can affect the likelihood of safe sex being practiced,  i.e. how likely an individual is to use a condom. If both partners use a condom, the model prevents the infection from being transmitted. However, if one partner chooses to use a condom while the other does not, the model simulates no use of a condom, and both parties are potentially at risk of infection. This could be explored and altered in possible extensions of this model. 

## THINGS TO NOTICE 

If the INITIAL PEOPLE slider is set relatively high (300-500), you can easily notice that couples start forming on top of each other. It appears as though there are 3 or more individuals in a given sexual relationship, but in actuality, some individuals may be occupying the same patch and only one is visible. 

All members of the population can become infected regardless of the qualities of the user-selected initial infector. Who is yet to be infected after the designated percentage of infected population has been met? Is this different with initial infectors of different genders and/or sexual orientations?

Before it was properly understood, HIV/AIDS (formerly known as Gay-Related Immune Deficiency) was thought of as a "gay" disease. Based on your observations of the model, comment on the likelihood of a disease only infecting persons of one particular sexual orientation.

## THINGS TO TRY 

After pressing the SELECT button, _infect_ any individual you want and run the simulation. Keep the same variables and press the SETUP button again. This time SELECT an individual with a different orientation and/or gender. Try all possible initial individuals and look for any patterns that might emerge. 

Adjust the sliders for %STRAIGHT and %GAY. Does the difference in distribution alter the outcomes at all? 

## EXTENDING THE MODEL 

Symptoms of sexually transmitted infections aren_t always visible or known, and some STIs display symptoms differently in different genders. These factors impact how often a particular gender might choose to get tested or use protection in sexual encounters. To better simulate real-life behaviors, implement the chance that females have a high likelihood of experiencing symptoms, while males do not. If a person experiences symptoms, they can become treated and cured of the infection in some defined amount of time. You can also implement the condition that if a person thinks they are infected, they will definitely use protection. See how these changes impact the outcome of the model.

In different relationships, condom use may vary. Additionally, condoms are not always effective or properly used. To more accurately account for likelihood of condom use and consequent transmission of infections, create different condom-use tendencies for each sexual orientation and create a probability that a condom is ineffective. 

The culture and sexual behavior habits might alter likelihood of transmission for couples depending on their sexual orientation, i.e. heterosexual vs. bisexual vs. homosexual. Choose different probabilities of transmission for couples of different sexual orientations. Notice if higher transmission rates and lower population percentages alters the outcomes. 

## NETLOGO FEATURES 
n-of is used to divide the turtle population into genders and sexual orientations.

Every time-step (tick), a slider_s value is checked against a global variable that holds the value of what the slider_s value was the time-step before. If the slider_s current value is different than the global variable, NetLogo knows to call procedures that adjust the population_s tendencies. Otherwise, those adjustment procedures are not called. 

The random-near function generates many small random numbers and adds them together to determine individual tendencies. This produces an approximately normal distribution of values.

## RELATED MODELS 

Disease Solo
Virus
AIDS 

## CREDITS AND REFERENCES 

This model is inspired by the AIDS model from the Models Library.
Special thanks to Dave Weintrop. 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

female
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 105 285 105 300 135 300 150 225 165 300 195 300 195 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
Polygon -7500403 true true 90 195 75 255 225 255 180 105 120 105 135 180 135 165 120 105

female sick
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 105 285 105 300 135 300 150 225 165 300 195 300 195 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
Polygon -7500403 true true 90 195 75 255 225 255 180 105 120 105 135 180 135 165 120 105
Circle -2674135 true false 113 98 72

female1
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 105 285 105 300 135 300 150 225 165 300 195 300 195 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
Polygon -7500403 true true 120 195 75 255 225 255 180 195 120 195

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

male
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

male sick
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
Circle -2674135 true false 120 105 60

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

person lefty
false
0
Circle -7500403 true true 170 5 80
Polygon -7500403 true true 165 90 180 195 150 285 165 300 195 300 210 225 225 300 255 300 270 285 240 195 255 90
Rectangle -7500403 true true 187 79 232 94
Polygon -7500403 true true 255 90 300 150 285 180 225 105
Polygon -7500403 true true 165 90 120 150 135 180 195 105

person righty
false
0
Circle -7500403 true true 50 5 80
Polygon -7500403 true true 45 90 60 195 30 285 45 300 75 300 90 225 105 300 135 300 150 285 120 195 135 90
Rectangle -7500403 true true 67 79 112 94
Polygon -7500403 true true 135 90 180 150 165 180 105 105
Polygon -7500403 true true 45 90 0 150 15 180 75 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
