globals [
  colors            ;; list that holds the colors used for students' turtles
  color-names       ;; list that holds the names of the colors used for students' turtles
  used-colors       ;; list that holds the shape-color pairs that are already being used
  max-possible-colors
]

breed [ students student ]
breed [ birds bird ]

students-own [
  user-id            ;; entered by student, unique identifier
  my-birdies              ;; birds controlled by this student
  max-separate-turn
  max-align-turn
  max-cohere-turn
  vision
  minimum-separation
  color-name
  trim
  speed
]

birds-own [
  flockmates         ;; agentset of nearby turtles
;  nearmates
  nearest-neighbor   ;; closest one of our flockmates
  my-student            ;; student controlling this bird
  max-separate-turn
  max-align-turn
  max-cohere-turn
  vision
  minimum-separation
  flockedness
]

to startup
  set colors      [ white   gray   brown   yellow   green   lime   turquoise
                    cyan   sky   blue   violet ]
  set color-names ["white" "gray" "brown" "yellow" "green" "lime" "turquoise"
                   "cyan" "sky" "blue" "violet" "orange" "magenta" "pink"]
  set max-possible-colors (length colors)
  set used-colors []
  create-robot-student
  hubnet-set-client-interface "COMPUTER" []
  hubnet-reset
end

to login
  listen-clients
end

to go
  if ( not any? birds ) [ stop ]
  ask birds [ flock measure-flock]
  ask students with [ user-id != "robo-student" ] [ update-interface ]
end

to flock  ;; turtle procedure
  find-flockmates
  ;find-nearmates
  if any? flockmates
    [ find-nearest-neighbor
      ifelse distance nearest-neighbor < minimum-separation
        [ separate ]
        [ align
          cohere ] ]
  fd 1 ;;+ value-from my-student [ speed ]
  ;; right value-from my-student [ trim ]
end

to reset
  ask birds [ die ]
  ask students with [ user-id != "robo-student" ]
    [ create-birds-for-student birds-per-student ]
end

to find-nearmates
;  set nearmates other turtles in-radius 3
end


to find-flockmates  ;; turtle procedure
  set flockmates other turtles in-radius vision
end

to find-nearest-neighbor ;; turtle procedure
  set nearest-neighbor min-one-of flockmates [distance myself]
end

;;; SEPARATE

to separate  ;; turtle procedure
  turn-away ([heading] of nearest-neighbor) max-separate-turn
end

;;; ALIGN

to align  ;; turtle procedure
  turn-towards average-flockmate-heading max-align-turn
end

to-report average-flockmate-heading  ;; turtle procedure
  ;; We can't just average the heading variables here.
  ;; For example, the average of 1 and 359 should be 0,
  ;; not 180.  So we have to use trigonometry.
  ;; Theoretically this could fail if both sums are 0
  ;; since atan 0 0 is undefined, but in practice that's
  ;; vanishingly unlikely.
  report atan sum [sin heading] of flockmates
              sum [cos heading] of flockmates
end

;;; COHERE

to cohere  ;; turtle procedure
  turn-towards average-heading-towards-flockmates max-cohere-turn
end

to-report average-flockmate-x ;; turtle-procedure
  report mean [ xcor ] of flockmates
end

to-report average-flockmate-y ;; turtle-procedure
  report mean [ ycor ] of flockmates
end

to cohere2 ;; turtle procedure
   turn-towards towardsxy average-flockmate-x average-flockmate-y max-cohere-turn
end

to-report average-heading-towards-flockmates  ;; turtle procedure
  ;; "towards myself" gives us the heading from the other turtle
  ;; to me, but we want the heading from me to the other turtle,
  ;; so we add 180
  report atan mean [sin (towards myself + 180)] of flockmates
              mean [cos (towards myself + 180)] of flockmates
end

;;; HELPER PROCEDURES

to turn-towards [new-heading max-turn]  ;; turtle procedure
  turn-at-most (subtract-headings new-heading heading) max-turn
end

to turn-away [new-heading max-turn]  ;; turtle procedure
  turn-at-most (subtract-headings heading new-heading) max-turn
end

;; turn right by "turn" degrees (or left if "turn" is negative),
;; but never turn more than "max-turn" degrees
to turn-at-most [turn max-turn]  ;; turtle procedure
  ifelse abs turn > max-turn
    [ ifelse turn > 0
        [ rt max-turn ]
        [ lt max-turn ] ]
    [ rt turn ]
end

to measure-flock ; turtle procedure
  let heading-limit 10
  let real-flockmates flockmates with
    [ abs subtract-headings heading [heading] of myself < heading-limit ]
  ifelse any? real-flockmates  [
    set flockedness count real-flockmates
  ] [
    set flockedness 0
  ]
end

to auto-flock
  ask birds
    [ setxy random 10 random 10
      turn-towards ( towardsxy-nowrap max-pxcor max-pycor ) 360      ]
end

;;; CLIENT HANDLING

to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ show "new client connection"  ]
    [
      ifelse hubnet-exit-message?
      [ remove-student ]
      [ execute-command hubnet-message-tag ]
    ]
  ]
end

to create-new-student
  create-students 1
    [ set user-id hubnet-message-source
      set-unique-color
      setup-student-vars
      create-birds-for-student birds-per-student
      hubnet-send user-id "Your birds are:" color-name
    ]
end

to create-robot-student
  create-students 1
    [ set user-id "robo-student"
      set color red
      setup-student-vars
    ]
end

to create-robot-bird
  show "Adding robot"
  ask students with [ user-id = "robo-student" ]
    [ create-birds-for-student 1 ]
end

to setup-student-vars
  set hidden? true
  set max-separate-turn default-max-separate-turn
  set max-align-turn default-max-align-turn
  set max-cohere-turn default-max-cohere-turn
  set vision default-vision
  set minimum-separation default-minimum-separation
  set my-birdies []

end

to set-unique-color
  let code 0

  set code random max-possible-colors
  while [member? (item code colors) used-colors and count students < max-possible-colors]
  [
    set code random max-possible-colors
  ]
  set color item code colors
  set color-name item code color-names
  set used-colors (lput color used-colors)

end

to update-birds [ student ]
  ask birds with [ my-student = student ] [ update-variables-from-student student ]
end

to update-variables-from-student [ student ]
      set hidden? false
      set my-student student            ;; student controlling this bird
      set max-separate-turn ([max-separate-turn] of student)
      set max-align-turn ([max-align-turn] of student)
      set max-cohere-turn ([max-cohere-turn] of student)
      set vision ([vision] of student)
      set minimum-separation ([minimum-separation] of student)
end

to create-birds-for-student [ num ]
  hatch-birds num
    [ update-variables-from-student myself
      rt random 360
      fd random 50
  ]
end


to turn-birds [ adj ] ;; turtle procedure
  ask birds with [ my-student = myself ]
    [ right adj ]
end

to accelerate-birds [ adj ] ;; turtle procedure
  ask birds with [ my-student = myself ]
    [ fd adj ]
end


to remove-student
  ask students with [ user-id = hubnet-message-source ]
  [
    ask birds with [ my-student = myself ] [ die ]
    set used-colors remove color used-colors
    die
  ]
end

to execute-command [command]

  if command = "login"
  [ if (not any? students with [ user-id = hubnet-message-source ])
     [ create-new-student ]
  ]

  if command = "max-separate-turn"
  [
    ask students with [user-id = hubnet-message-source]
    [ set max-separate-turn hubnet-message
      update-birds self ]
    stop
  ]
  if command = "max-align-turn"
  [
    ask students with [user-id = hubnet-message-source]
    [ set max-align-turn hubnet-message
      update-birds self
    ]
    stop
  ]
  if command = "max-cohere-turn"
  [
    ask students with [user-id = hubnet-message-source]
      [ set max-cohere-turn hubnet-message
        update-birds self
      ]
    stop
  ]

  if command = "vision"
  [
    ask students with [user-id = hubnet-message-source]
    [ set vision hubnet-message
      update-birds self
    ]
    stop
  ]

  if command = "minimum-separation"
  [
    ask students with [user-id = hubnet-message-source]
    [ set minimum-separation hubnet-message
      update-birds self]
    stop
  ]

  if command = "left"
    [ ask students with [ user-id = hubnet-message-source ]
      ;;[ set trim max (list -10 (trim - 1)) ]
      [ turn-birds -5 ]
    ]

  if command = "right"
    [ ask students with [ user-id = hubnet-message-source ]
      ;;[ set trim min (list 10 (trim + 1)) ]
      [ turn-birds 5 ]
    ]

  if command = "faster"
    [ ask students with [ user-id = hubnet-message-source ]
      ;;[ set speed precision min (list 0.5 (speed + 0.1)) 1 ]
      [ accelerate-birds 1 ]
    ]

  if command = "slower"
    [ ask students with [ user-id = hubnet-message-source ]
      ;;[ set speed precision max (list -0.5 (speed - 0.1)) 1 ]
      [ accelerate-birds -.5 ]
    ]
end

to update-interface
  if (not any? birds ) [ stop ]
  hubnet-send user-id "Your flock value:"
    precision ( mean [ flockedness ] of birds with [ my-student = myself ] ) 1
  hubnet-send user-id "trim" [ trim ] of self
  hubnet-send user-id "speed" [ speed ] of self
end
@#$#@#$#@
GRAPHICS-WINDOW
250
10
704
485
18
18
12.0
1
10
1
1
1
0
1
1
1
-18
18
-18
18

CC-WINDOW
5
499
713
594
Command Center
0

BUTTON
149
10
230
43
NIL
go
T
1
T
OBSERVER
T
G
NIL

SLIDER
9
88
232
121
birds-per-student
birds-per-student
1
30
1
1
1
NIL

SLIDER
7
133
241
166
default-max-separate-turn
default-max-separate-turn
0
20
1.5
0.25
1
NIL

SLIDER
8
187
219
220
default-max-align-turn
default-max-align-turn
0
20
5
0.25
1
NIL

SLIDER
9
232
233
265
default-max-cohere-turn
default-max-cohere-turn
0
20
3
0.25
1
NIL

SLIDER
7
314
179
347
default-vision
default-vision
0
10
3
0.5
1
NIL

SLIDER
7
273
243
306
default-minimum-separation
default-minimum-separation
0
5
0.5
0.5
1
NIL

BUTTON
155
49
221
82
NIL
reset\\n
NIL
1
T
OBSERVER
T
R
NIL

MONITOR
7
357
142
406
Global Flock Value
mean [flockedness] of birds
1
1

BUTTON
21
11
84
44
NIL
login
T
1
T
OBSERVER
NIL
L
NIL

BUTTON
21
51
124
84
Add Robot
create-robot-bird
NIL
1
T
OBSERVER
NIL
NIL
NIL

BUTTON
8
412
108
445
NIL
auto-flock
NIL
1
T
OBSERVER
T
NIL
NIL

@#$#@#$#@
VERSION
-------
$Id: HubFlocking.nlogo 37529 2008-01-03 20:38:02Z craig $


WHAT IS IT?
-----------

This model is a HubNet enabled variation of the flocking module.  Each student controls a number of birds, identified by color.  It works precisely as the Flocking model, but each student can control the parameters of their bird(s) flocking behavior.  A "flock value" is also tracked for all birds, and for each student.

HOW TO USE IT
-------------

First, determine the number of birds you want each student to control in the model and set the BIRDS-PER-STUDENT slider to that value.  The press the LOGIN button to allow students to enter the model.  Press GO to have them start flying around.  Birds will be created and removed as students connect and disconnect from the hub.

Each studentThe default settings for the sliders will produce reasonably good flocking behavior.  However, you can play with them to get variations:

Three TURN-ANGLE sliders control the maximum angle a bird can turn as a result of each rule.

VISION is the distance that each bird can see 360 degrees around it.

SEPARATION is the distance each bird maintains from other birds.

THINGS TO NOTICE
----------------
Central to the model is the observation that flocks form without a leader.

There are no random numbers used in this model, except to position the birds initially.  The fluid, lifelike behavior of the birds is produced entirely by deterministic rules.

Also, notice that each flock is dynamic.  A flock, once together, is not guaranteed to keep all of its members.  Why do you think this is?

After running the model for a while, all of the birds have approximately the same heading.  Why?

Sometimes a bird breaks away from its flock.  How does this happen?  You may need to slow down the model or run it step by step in order to observe this phenomenon.


THINGS TO TRY
-------------
Play with the sliders to see if you can get tighter flocks, looser flocks, fewer flocks, more flocks, more or less splitting and joining of flocks, more or less rearranging of birds within flocks, etc.

You can turn off a rule entirely by setting that rule's angle slider to zero.  Is one rule by itself enough to produce at least some flocking?  What about two rules?  What's missing from the resulting behavior when you leave out each rule?

Will running the model for a long time produce a static flock?  Or will the birds never settle down to an unchanging formation?  Remember, there are no random numbers used in this model.


EXTENDING THE MODEL
-------------------
Currently the birds can "see" all around them.  What happens if birds can only see in front of them?

Is there some way to get V-shaped flocks, like migrating geese?

What happens if you put walls around the edges of the screen that the birds can't fly into?

Can you get the birds to fly around obstacles in the middle of the screen?

What would happen if you gave the birds different velocities?  For example, you could make birds that are not near other birds fly faster to catch up to the flock.  Or, you could simulate the diminished air resistance that birds experience when flying together by making them fly faster when in a group.

Are there other interesting ways you can make the birds different from each other?  There could be random variation in the population, or you could have distinct "species" of bird.


NETLOGO FEATURES
----------------
Notice the need for the SUBTRACT-HEADINGS primitive and special procedure for averaging groups of headings.  Just subtracting the numbers, or averaging the numbers, doesn't give you the results you'd expect, because of the discontinuity where headings wrap back to 0 once they reach 360.


CREDITS AND REFERENCES
----------------------
This model is inspired by the Boids simulation invented by Craig Reynolds.  The algorithm we use here is roughly similar to the original Boids algorithm, but it is not the same.  The exact details of the algorithm tend not to matter very much -- as long as you have alignment, separation, and cohesion, you will usually get flocking behavior resembling that produced by Reynolds' original model.  Information on Boids is available at http://www.red3d.com/cwr/boids/.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.0pre3
@#$#@#$#@
set population 200
setup
repeat 200 [ go ]
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
SLIDER
8
144
246
177
max-separate-turn
max-separate-turn
0
20
1.5
0.25
1
degrees

SLIDER
8
223
223
256
max-align-turn
max-align-turn
0
20
5
0.25
1
degrees

SLIDER
8
184
235
217
max-cohere-turn
max-cohere-turn
0
20
3
0.25
1
degrees

SLIDER
8
103
239
136
minimum-separation
minimum-separation
0
5
1.5
0.5
1
patches

SLIDER
9
65
181
98
vision
vision
0
10
3.5
0.5
1
patches

MONITOR
10
10
115
59
Your birds are:
NIL
3
1

MONITOR
126
10
242
59
Your flock value:
NIL
3
1

BUTTON
360
69
430
102
faster
NIL
NIL
1
T
OBSERVER
NIL
W

BUTTON
352
164
448
197
slower
NIL
NIL
1
T
OBSERVER
NIL
S

BUTTON
321
120
384
153
left
NIL
NIL
1
T
OBSERVER
NIL
A

BUTTON
407
119
470
152
right
NIL
NIL
1
T
OBSERVER
NIL
D

MONITOR
273
65
330
114
trim
NIL
0
1

MONITOR
470
65
527
114
speed
NIL
2
1

BUTTON
259
11
322
44
login
NIL
NIL
1
T
OBSERVER
NIL
NIL

@#$#@#$#@
