breed [electrons electron]
globals [
  current         ;; current measurement, accumulates for 100 ticks
  last-current    ;; current measurement from previous 100 tick period
]

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setup Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;
to setup
  clear-all
  set-default-shape electrons "circle 2"

  ;; create wire & terminals
  ask patches
    [ set pcolor gray ]
  ask patches with [pxcor >= max-pxcor - 4 ]
    [ set pcolor red ]
  ask patches with [pxcor <= min-pxcor + 4 ]
    [ set pcolor black ]

  ;; create electrons
  create-electrons total-electrons
  [
    setxy random-xcor random-ycor
    set color orange - 2
    set size 1.5
  ]

  ;; create labels for the battery terminals
  ask patch (min-pxcor + 4) 0
    [ set plabel "+" ]
  ask patch (max-pxcor - 1) 0
    [ set plabel "-" ]

  ;; since the battery-positive is a sink for negative charges,
  ;; move electrons immediately to the battery-negative
  ask electrons [ if pcolor = black [ set xcor xcor - 7 ] ]

end


;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Runtime Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;
to go
  ;; Rules for electrons
  ask electrons
  [
    ;; wiggle randomly, performing simple point collisions with invisible
    ;; nuclei in the wire and steadily drifting forward due to the electric field
    move
    ;; contribute to current measurement
    add-current
  ]
  ;; advance tick counter and update the view
  tick
  ;; measure and plot current
  if ticks mod 100 = 0
    [ do-plot ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;
;; rules for electrons ;;
;;;;;;;;;;;;;;;;;;;;;;;;;

to move
  ;; perform collisions with (invisible) atoms
  if random-float 1.0 < collision-rate-with-nuclei
    [ rt random 360 ]
  fd 0.3
  ;; drift due to electric field towards the battery-positive
  set xcor xcor - 0.5 * voltage / collision-rate-with-nuclei
end

;;;;;;;;;;;;;;;;;;;;;;;;;
;; Plotting procedures ;;
;;;;;;;;;;;;;;;;;;;;;;;;;

to do-plot
  set-current-plot "Current"
  plotxy (ticks - 100) current / 100
  set last-current current / 100
  set current 0
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Procedures for measuring current ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; count number of electrons arriving at the battery-positive per tick
to add-current
  if pcolor = black [
    set current current + 1
    set xcor xcor - 6
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
211
10
905
173
85
16
4.0
1
30
1
1
1
0
1
1
1
-85
85
-16
16
1
1
1
ticks

CC-WINDOW
5
356
918
451
Command Center
0

BUTTON
21
62
105
102
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
107
62
192
102
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
12
107
201
140
voltage
voltage
0.5
2
1.5
0.5
1
NIL
HORIZONTAL

SLIDER
11
25
201
58
total-electrons
total-electrons
5
2000
500
5
1
NIL
HORIZONTAL

TEXTBOX
214
176
326
194
Battery-positive
11
0.0
0

TEXTBOX
817
176
909
204
Battery-negative\\n
11
0.0
0

SLIDER
12
143
201
176
collision-rate-with-nuclei
collision-rate-with-nuclei
0.1
1
0.5
0.1
1
NIL
HORIZONTAL

PLOT
355
175
627
333
Current
time
current
0.0
0.0
0.0
0.35
true
false
PENS
"default" 1.0 0 -16777216 true

MONITOR
630
176
691
221
current
last-current
2
1
11

BUTTON
14
275
206
308
color & trace one electron
ask one-of electrons with [color = orange - 2]\\n  [ set color one-of [blue yellow violet]\\n    pen-down ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
14
195
175
228
watch one electron
ask one-of electrons \\n[ set color blue\\n  watch-me\\n]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
14
229
175
262
stop watching
reset-perspective\\nask electrons [ set color orange - 2]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
14
309
206
342
stop tracing
ask electrons [ pen-up ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
VERSION
-------
$Id: Ohm's Law.nlogo 37529 2008-01-03 20:38:02Z craig $


WHAT IS IT?
-----------
This model illustrates how a steady current as well as resistance emerge from simple interactions between electrons and atoms in a wire and battery terminals.  It shows how the linear relationship between current (I), resistance(R) and voltage (V) as expressed in Ohm's Law (i.e., V = I * R) emerges from these interactions.


HOW IT WORKS
------------
The wire in this model (gray region) is composed of atoms, which in turn are made of negatively charged electrons and positively charged nuclei.  According to the Bohr model of the atom, these electrons revolve in concentric shells around the nucleus.  However, in each atom, the electrons that are farthest away from the nucleus (i.e., the electrons that are in the outermost shell of each atom) behave as if they are free from the nuclear attraction.  These outermost electrons from each atom are called "free electrons".   These free electrons obey some rules that are specified in the Procedures tab of the model.  The applied electric field due to the battery imparts a steady velocity to the electrons in the direction of the positive terminal.  The positive battery terminal (black region), which is actually an enormous collection of positive charges, acts as a sink for negative charges or electron, whereas, the negative battery terminal (red region) is a large source of negative charges or electrons.

Note that electrons reappear on the other side at the negative terminal after crossing into the positive terminal of the battery.  This is a greatly simplified representation of the continuous process of charge generation in the battery maintaining a constant voltage (or potential difference) between the two terminals.

Voltage in the battery gives rise to a constant electric field throughout the wire, imparting a steady drift to the electrons while they move in the wire connecting the two terminals.   In addition to this drift, the electrons also collide with the atomic nuclei in the wire, giving rise to electrical resistance in the wire.  However, to reduce the complexity of the model, these nuclei are not shown in the model.


HOW TO USE IT
-------------
The TOTAL-ELECTRONS slider allows selecting the total number of free electrons in both the wires. This number is kept constant throughout a single run of the model.

The VOLTAGE slider indicates the magnitude of voltage between the battery terminals.  It imparts a steady velocity to the electrons. However, as the next paragraph explains, this velocity also depends on the rate at which the electron collides with the atomic nuclei in the wires.

The COLLISION-RATE-WITH-NUCLEI slider is inversely proportional to how far an electron travels on average without colliding with atomic nuclei. The collision rate of electrons in a wire causes resistance. The collision-rate affects the motion of electrons in it in another way: the net velocity of the electrons is also reduced in proportion to the collision rate.

The button COLOR & TRACE ONE ELECTRON asks an electron to change color and trace its path.  If you want to go back to the default settings (with all electrons red and no traces), you need to press SETUP again.

The button WATCH ONE ELECTRON highlights a single electron (chosen randomly) so that you can observe its motion in particular.  If you want to go back to the default settings (with all electrons red and no traces), you need to press SETUP again.


THINGS TO NOTICE
----------------
When you observe the trace of the path of an electron by pressing the COLOR & TRACE ONE ELECTRON button, how does the path change when you increase or decrease the COLLISION-RATE-WITH-NUCLEI ?


THINGS TO TRY
-------------
1) Run the model with the default settings. Note the current and number of electrons in the wire.

2a) Increase the collision-rate of electrons with the nuclei. How does the value of current in the wire change? How does the motion of electrons change?
2b) Reduce the collision-rate of electrons with the nuclei. How does the value of current in the wire change? How does the motion of electrons change?

3a) Press WATCH ONE ELECTRON. Using the tick counter, note how much model time the electron takes to travel through the wire. Repeat this observation several times for the same value of collision-rate-with-nuclei. Now calculate the average speed of the electron from these observations.
3b) Repeat 3a for a few different values of the slider COLLISION-RATE-WITH-NUCLEI. What do you notice?
3c) How can you calculate current in the wire from the average speed of an electron through it?

4) Look for the "Procedures for measuring current" section in the Procedures tab. How is current in each wire calculated in this model?  Are this method and 3(b) equivalent to each other?

5) Based on your interactions with the model, how would now explain the concepts of current and resistance?


EXTENDING THE MODEL
-------------------
Can you create a series circuit (with two wires in series) by extending this model?


NETLOGO FEATURES
----------------
Electrons wrap around the world both vertically and horizontally.


RELATED MODELS
--------------
Electrostatics
Series Circuit
Parallel Circuit


CREDITS AND REFERENCES
----------------------
This model is a part of the NIELS curriculum. The NIELS curriculum is currently under development at Northwestern's Center for Connected Learning and Computer-Based Modeling. For more information about the NIELS curriculum please refer to http://ccl.northwestern.edu/NIELS.

Thanks to Daniel Kornhauser for his work on the design of this model.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300
Circle -7500403 false true -45 -45 180
Circle -16777216 false false -2 -2 304

circle 2
false
0
Circle -16777216 true false 0 0 300
Circle -7500403 true true 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.0beta8
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment 1" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <metric>current</metric>
    <enumeratedValueSet variable="total-electrons">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Electric-field">
      <value value="1.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="collision-rate-with-nuclei-left">
      <value value="1"/>
      <value value="1.5"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="collision-rate-with-nuclei-right">
      <value value="2"/>
      <value value="2.5"/>
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
