;~~~~~~~~~~~~ These are globals
breed [Mosquitos Mosquito]
breed [Reindeers Reindeer]

extensions [
  csv
  time
]

globals [
  grass-growth-rate          ; the rate at which grass gows
  grass-density              ; the density of patches that is vegetation
  grass-probability          ; the probability that grass has not been overgrazed
  grass-regrowth-rate        ; the rate at which grass grows back
  grass-energy               ; amount of energy grass provides
  reproductive-energy        ; amount of energy needed for reproduction
  new-turtle                 ; spawn a new turtle
  herd-size                  ; the size of a herd
  initial-energy             ; We assume each reindeer will live for 12 years (365 ticks * 12 years = 4380)
  mortality-rate             ; the rate at which reindeer die
  risky-reindeer             ; the rate at which reindeer will explore
  cluster-center             ; this is the center of the blue water body
  mosquito-start
  reindeer-density           ; This is how many reindeer can occupy any one patch at a given time
  snow-density
  wise-one                   ; oldest reindeer

  patch-radius               ; This is the radius of the lakes
  elevation                  ; this is the elevation of the terrain
  elevation-max              ; This is the maximum elevation of the terrain in the model


  current-date               ; This is the current date of the tick
  current-month              ; This is the current month of the date of the tick, useful for categorizing meterological seasons
  time-series                ; This is if user uploads a .csv file with the first column being a datetime (YYYY-MM-DD) and the second column is temperature in degrees Celcius (must have 1 row of headers)

  temperature                ; This is the daily temperature for the simulation
  index                      ; This is to keep track of the list in the time-series

  rainy-days                ; This is to track how many consecutive days of rain there has been
  cold-warm-days            ; This is to track how many consecutive warm days above 0 C there have been


]

patches-own

[
  grass-amount              ; This is how much units of grass there is, 1 unit of grass can feed 1 turtle


]

;~~~~~~~~~~~~These are turtle properties
Reindeers-own
[
  energy               ; enrgy of the turtle
  hunger               ; minimum amount of grass need to be eaten for turtle survival
  age                  ; the age of turtles in ticks
  reproductiveness     ; the reproductive ability for turtles, based on age and body condition
  infected?            ; if this is true, the turtle is infected
  immunity-remaining   ; how many weeks of immunity the turtle has left
  flockmates           ; agentsset of nearby turtles
  nearest-neighbor     ; closest flockmate
  immune?              ; if this is true a turtle is immune
  recovered?           ; if this is true a turtle has recovered
  sick-time            ; when a turtle became sick
  dead?                ; when true, a turtle is dead
  larvae-count         ; how many larvae are in each reindeer


]

Mosquitos-own
[
 age                 ; age of mosquito (22-76 ticks, days), adult after 20 ticks
 infected            ; infected with m.f.
 infected?
 blood-meal          ; cumulative count of blood for oviposit



]

to clear-run

  clear-all
end

to setup

  clear-output
  clear-all
  reset-ticks

  ifelse file-option? [

  ;set current-time time:create "2010-01-01"
    setup-file
  ][
    set current-date time:create Date_opt
  ]

  setup-patches
  setup-reindeer
  setup-mosquitos

  set cold-warm-days 0
  set rainy-days 0

end

; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~ This is to setup the world
to setup-file

  set time-series csv:from-file File-name

  let start-date first item 1 time-series
  let end-date first item (length time-series - 1) time-series

  set current-date start-date

  set index 1

end

to setup-patches


  resize-world 0 world-size 0 world-size                     ; set up world size. This is in m. 1 patch is 100 m x 100 m


  ask patches [
    ifelse random-float 1.0 < 0.2              ; Adjust the probability (0.2 in this case) for green patches
    [ set pcolor green
    set grass-amount 5]
    [ set pcolor brown
    set grass-amount 0]
  ]

  create-blue-clusters                          ; Create irregular-shaped clusters of blue patches


end

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~ This is to set up a water body ~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

to create-blue-clusters

ifelse patch-sizes = "small" [
    set patch-radius world-size * 0.05
  ][
  ifelse patch-sizes = "medium" [
    set patch-radius world-size * 0.15
  ][
    set patch-radius world-size * 0.3
  ]]

  repeat patches-amt [
    set cluster-center patch random-xcor random-ycor
    ask cluster-center [
      set pcolor blue
      ask patches in-radius patch-radius [ set pcolor blue ]
    ]
  ]



; Define mosquito-start patches at the edge of blue patches
  let edge-patches patches with [
    pcolor = blue and any? neighbors with [pcolor != blue]
  ]

  ifelse any? edge-patches [
    set mosquito-start edge-patches
  ][
    user-message "No suitable mosquito start patches found"
  ]

end



;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~ This is to set up the reindeer;~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


to setup-reindeer

  let non-blue-patches patches with [pcolor != blue] ; consider only non-blue patches
  ; Set initial positions closer together near the nearest non-blue patch
  let p one-of non-blue-patches

  create-Reindeers Reindeer-pop [
    set shape "reindeer"
    set size 1.5
    set color 33
    set age 0
    set energy initial-energy
    set immune? false
    set dead? false
    set energy 300
    set infected? false
    set larvae-count 0

    if random-float 1 < initial-infected-rate [
     set infected? true
     set larvae-count larvae-count + 1
     set color violet
    ]

    set reindeer-density 5
    move-to p ; this is the initial point
    let i 0
    let q patch-ahead random-float (herd-spread * Reindeer-pop)
    while [ i < reindeer-density] [
      rt random 360 ;cw: we were missing random turns so it was only doing random distance below
      set q patch-ahead random-float (herd-spread * Reindeer-pop)  ;cw: copied this inside the while so q is a new patch each loop of the while, also changed to * reindeer to match herd-movement
      if q != nobody and [pcolor] of q != blue and [pcolor] of q != white and [count Reindeers-here] of q < reindeer-density [
        move-to q
        set i reindeer-density ;cw: if they move, break the while
      ]
      set i i + 1
    ]
    ;cw: optional check to more fully scatter over-crowded reindeer (or increase reindeer-density above)
    if count Reindeers-here > reindeer-density [
      move-to one-of patches with [pcolor != blue and pcolor != white and count Reindeers-here < reindeer-density]
    ]
  ]
end

;~~~~~~~~~~~~~~~~~~~ This is to set up mosquitos
to setup-mosquitos

  create-Mosquitos Mosquitos-pop[
    set shape "mosquito"
    set size 0.45
    set color white
    set age 0
    ;set age random-float max-age
    set blood-meal 0
    let initial-patch one-of mosquito-start ;patches with [pcolor = blue]
    move-to initial-patch
  ]
end
;~~~~~~~~~~~~~~~~~~~ This is to set up mosquitos





;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   ++++    +++   ++  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  +       +   +  ++  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  +   ++  +   +  ++  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  +    +  +   +      ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   ++++    +++   ++  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


to go

  ;;;;;;;;;;;;;;; PATCHES STUFF ;;;;;;;;;;;;;;;
  grow-grass
  eat-grass
  herd-movement

  ;;;;;;;;;;;;;;; REINDEER STUFF ;;;;;;;;;;;;;;;
  if reindeer-life-cycle? [
    reproduce
    age-and-die
  ]

  infect-reindeer

  ;;;;;;;;;;;;;;; MOSQUITO STUFF ;;;;;;;;;;;;;;;
  mosquito-age
  mosquito-move
  mosquito-bite
  mosquito-reproduce-die


  ;;;;;;;;;;;;;;; TEMP STUFF (FROM FILE OPTION) ;;;;;;;;;;;;;;;

  ifelse file-option? [

    if index < length time-series [
      set temperature last item index time-series
      set current-date first item index time-series
    ]
    set index index + 1

    if index > length time-series [
      user-message "The simulation has reached the end of your file points"
      stop
    ]

  ][

    temp

      ifelse temperature > 0 [set cold-warm-days cold-warm-days + 1] [set cold-warm-days 0]
      set current-date time:plus current-date 1 "days"
  ]

   ;;;;;;;;;;;;;;; RAIN STUFF ;;;;;;;;;;;;;;;

  if climate-on? [precipitation]

  tick

end

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

to grow-grass ; grass grows on dirt patches at a constant rate

  ; Within 14 ticks, theres a threshold for 5 reindeer grazing
  ; Grass will regrow

  set grass-regrowth-rate 0.0025


  ask patches with [pcolor = brown] [

    if random-float 1 < grass-regrowth-rate [
      set pcolor green
      set grass-amount 5
      ; later I want to add a variation on green depending on grass-amount
      ;Last visited by reindeer over a certain amount of time, will regrow. Healthier grass patch after an amount of time
    ]

  ]
end



;~~~~~ ++ ++      ++++++++  ++++++++   +++++      ++++  ++++       ++++++++  ++++++++  ++ ++    ~~~~~~~~~~
;~~~~~ +++  ++    ++           ++      ++  ++      ++   ++  ++     ++        ++        ++   ++  ~~~~~~~~~~
;~~~~~ ++  ++     ++           ++      ++   ++     ++   ++    ++   ++        ++        ++  ++   ~~~~~~~~~~
;~~~~~ +++        ++++         ++      ++    ++    ++   ++     ++  ++++      ++++      +++      ~~~~~~~~~~
;~~~~~ ++ ++      ++           ++      ++     ++   ++   ++     ++  ++        ++        ++ ++    ~~~~~~~~~~
;~~~~~ ++   ++    ++           ++      ++      ++  ++   ++    ++   ++        ++        ++   ++  ~~~~~~~~~~
;~~~~~ ++     ++  ++           ++      ++       ++ ++   ++  ++     ++        ++        ++    ++ ~~~~~~~~~~
;~~~~~ ++     ++  ++++++++  ++++++++   ++        ++++   ++++       ++++++++  ++++++++  ++    ++ ~~~~~~~~~~

to butcher
  ask n-of butcher-amt Reindeers [die]
  print (word "You have butchered " butcher-amt " reindeer")
end



to eat-grass
  ask reindeers [
    if pcolor = green and grass-amount > 0 [
      set energy energy + 10
      ask patch-here [
        set grass-amount grass-amount - 1
        if grass-amount = 0 [
          set pcolor brown
        ]
      ]
    ]
  ]
end

to reproduce
  ;

  ask Reindeers with [energy > 150 and random-float 1 < reproduction-rate] [
    let nearby-Reindeers other Reindeers in-radius vision
    let parent-Reindeers min-one-of nearby-Reindeers with [self != myself] [distance myself]
    if parent-Reindeers != nobody [
      hatch-reindeers 1 [
        let mean-x mean [xcor] of nearby-Reindeers
        let mean-y mean [ycor] of nearby-Reindeers
        setxy mean-x mean-y
        set age 1
        set energy initial-energy
      ]
      face parent-Reindeers
      fd 1
    ]
  ]
end



to age-and-die

  ;Adult reindeer max age is 20 years. Each tick is a day (20 years is 7300 days)
  ;Reindeer become adults at 5 (Females) and 6 (Males). For now, will set for 5.

  ask Reindeers [
    set age age + 1
    set energy energy - 1
    if age > 7300 or energy <= 0 [
      die]
  ]

  if wise-one = nobody [
    set wise-one min [who] of Reindeers ; this is the minimum ID of all turtles, the turtle with the lowest identifier
    ask Reindeer wise-one [set color yellow] ]

end


to infect-reindeer

  ask reindeers [
    if any? turtles-here with [breed = mosquitos and infected > 0] [
      if random-float 1 < 0.5 [
        set infected? true
        set larvae-count larvae-count + 1
      ]
    ]
  ]
end


;to herd-movement-stationary
;
;  ; Ask all turtles to perform herd movement logic
;  ; Check if the turtle is a cow
;
;  ask turtles [
;
;    if shape = "cow" [
;      let all-turtles other turtles
;      let nearby-turtles all-turtles with [distance myself < vision]
;
;    ; check if turtle is on a non-frozen patch next to a water patch
;
;      let risky? false
;      let neighboring-patches patches in-radius 1 with [pcolor = blue]
;      if pcolor != white and any? neighboring-patches [
;        set risky? (random-float 1 < risky-reindeer)
;    ]
;
;    ; move towards mean position if not risky
;    if not risky? and any? nearby-turtles [
;      let mean-x mean [xcor] of nearby-turtles
;      let mean-y mean [ycor] of nearby-turtles
;      let mean-position patch mean-x mean-y
;
;      let dispersion-factor herd-spread * Reindeer-pop
;      let max-dispersion round(vision * 0.5)
;      let min-dispersion round(vision * 0.25)
;      let radius random-float (max-dispersion - min-dispersion) + min-dispersion
;
;      ; find all patches within radius that are not frozen or water and have fewer than 5 turtles
;      let non-frozen-patches patches with [
;          pcolor != white and
;          pcolor != blue and
;          distance myself <= radius and
;          count Reindeers-here < 5
;        ]
;
;      ; move towards a non-frozen patch if there is one, otherwise move randomly
;      ifelse any? non-frozen-patches [
;        let destination one-of non-frozen-patches
;        face destination
;        fd 1
;      ] [
;        rt random 360
;        fd 1
;      ]
;]]]
;
;end


to herd-movement

  let non-frozen-patches patches with [ ;Patches that are not snow or water
    pcolor != 105
  ]

  set wise-one min [who] of Reindeers ; this is the minimum ID of all turtles, the turtle with the lowest identifier
  ask Reindeer wise-one [set color yellow]
  let dispersion-factor herd-spread * Reindeer-pop ; this is how many patches wide herded reindeer can move around in
  let herd-patches non-frozen-patches with [distance Reindeer wise-one < dispersion-factor] ;  contain a subset of non-frozen-patches that are within the specified distance from the turtle wise-one

  let lead-x [xcor] of Reindeer wise-one
  let lead-y [ycor] of Reindeer wise-one
  let lead-distance patch lead-x lead-y ; The location of wise-one will dictate movements of all turtles

  ask Reindeer wise-one [
    face one-of herd-patches in-radius vision
    if patch-ahead 1 != nobody and [pcolor] of patch-ahead 1 != blue [
      fd 1
  ]
  ]

  ask Reindeers [

    ; if the distance of the turtle to the lead-turtle is less than the dispersion factor, then it will move randomly within the "radius" of the herd
    ; However, if the distance of the turtle to the lead turtle is more than the dispersion factor, then it will move forward one towards the leader
    ; We also want to make sure there are only 5 turtles on a patch

    ifelse distance Reindeer wise-one < dispersion-factor [
      rt random 360
      if patch-ahead 1 != nobody and member? patch-ahead 1 non-frozen-patches and [count Reindeers-here] of patch-ahead 1 < 5 [ ; rename to land
        fd 1];[if count turtles-here > 5 [type "a"]]
    ][
      ; this is asking if the patch ahead the turtle satifies the following conditions. If it does it moves fd 1. If not it faces the wise-one and moves fd 1 towards them
      ;[1] That the patch is a herd-patch, meaning it is grass and not frozen
      ;[2] that there are less than 5 turtles on that patch
      face Reindeer wise-one
      ifelse patch-ahead 1 != nobody and member? patch-ahead 1 herd-patches and [count Reindeers-here] of patch-ahead 1 < 5 and [pcolor] of patch-ahead 1 != blue [
        fd 1
      ]
      [
          rt 90 - random 180
          if patch-ahead 1 != nobody and member? patch-ahead 1 non-frozen-patches and [count Reindeers-here] of patch-ahead 1 < 5 and [pcolor] of patch-ahead 1 != blue [
          fd 1 ];[if count turtles-here > 5 [type "b"]]
      ]
    ]

    if patch-here = blue [
      rt 180
      fd 1
    ]

    ;if count turtles-here > 5 [type "c"]
  ]

;    if [pcolor] of patch-here = blue [
;      print "Turtle is on a blue patch."
;      stop ]

  ; CHECK HOW MANY REINDEER PER PATCH
;  if max [count turtles-here] of patches > 5 [
;      print word ("There are more than 5 turtles on a patch:") max [count turtles-here] of patches
;  ]

end

;~~~~~~~~~~  +++         +++     +++++     ++++++         +++++     ~~~~~~~~~~
;~~~~~~~~~~  ++  +     +  ++    ++   ++    ++            ++   ++    ~~~~~~~~~~
;~~~~~~~~~~  ++   +   +   ++   ++     ++     ++         ++     ++   ~~~~~~~~~~
;~~~~~~~~~~  ++    + +    ++  ++       ++        ++    ++       ++  ~~~~~~~~~~
;~~~~~~~~~~  ++     +     ++  ++       ++          ++  ++       ++  ~~~~~~~~~~
;~~~~~~~~~~  ++           ++   ++     ++          ++    ++     ++   ~~~~~~~~~~
;~~~~~~~~~~  ++           ++    ++   ++         ++       ++   +++   ~~~~~~~~~~
;~~~~~~~~~~  ++           ++     +++++     ++++++         +++++ ++  ~~~~~~~~~~

to mosquito-age


  ; Age adult mosquitos
  ask Mosquitos with [color != white] [
    set age age + 1]

  ; Mosquito larvae and pupa will grow above 5 degrees
  if temperature > 4 and temperature < 40 [
    if member? current-month [3 4 5 6 7 8 9] [
      ask mosquitos with [color = white] [
        set age age + 1
        if age >= 20 [
          set color 1 ]
      ]
    ]
  ]
  ; but half will die if temperatures hit above 40
  if temperature > 40 [
    ask n-of (0.5 * count mosquitos with [color = white]) mosquitos with [color = white] [
      die]
  ]



end

to mosquito-move

  ;This is set up so that all mosquitos move around the water body
  ;as they are flood-water mosquitos.

  ask Mosquitos with [color != white] [
    ifelse blood-meal < blood-max [
      let r min-one-of Reindeers [distance myself]
      ifelse distance r < mosquito-vision [
        face r
        fd random-float mosquito-move-day] [
        rt random 360
        fd random-float mosquito-move-day]
    ] [
      let t min-one-of mosquito-start [distance myself]
      face t
;      print "t"
      ifelse distance t < mosquito-move-day [
        move-to t][
        fd random-float mosquito-move-day
      ]

    ]
  ]

end

to mosquito-bite

  ; this model will follow the SIRS
  ;Susceptible
  ;Infected


 ; Only consider mosquitoes that are either healthy (color 1) or already infected (color red)
  ask Mosquitos with [color = 1 or color = red] [

; If there are reindeer on the same patch as the mosquito
    if count Reindeers-here != 0 [
      let lunch 0                                                 ; Initialize a variable 'lunch' to store that the mosquito has had a blood meal or partial blood meal
      ifelse count Reindeers-here <= mosquito-bite-day            ; If the number of reindeer on the patch is less than or equal to the number of bites a mosquito can take in a day
        [set lunch Reindeers-here]
        [set lunch n-of mosquito-bite-day Reindeers-here ]        ; if there are 50 reindeer, they will bite all 50, if 100 reindeer, they will bite random 50 if mosquito bite is 50

      set blood-meal blood-meal + (count lunch * random-float blood-meal-size)       ; Increase the mosquito's blood meal by an amount proportional to the number of reindeer bitten and a random factor of the blood-meal-size
;      if blood-meal >= blood-max [
;        print "blood!"
;        print blood-meal]

      let infected-lunch lunch with [infected? = true]           ; Create a subset of 'lunch' that includes only the reindeer that are already infected

      let i 0
      while [i < count infected-lunch] [                         ; Create a subset of 'lunch' that includes only the reindeer that are already infected
        if random-float 1 < infected-probability [               ; If a randomly generated number between 0 and 1 is less than the infected-probability
          set infected 1                                         ; can add m.f. load here (avg 3-6 per mosquito)
          set color red
          ]

        set i i + 1
      ]

      print infected-lunch


    ]


  ]

end


to mosquito-reproduce-die


; Only reproduction happening from April to September

  if member? current-month [4 5 6 7 8 9] [

    ask mosquitos with [blood-meal >= blood-max] [
      if member? patch-here mosquito-start [
        hatch-mosquitos eggs [
          set infected 0
          set color white
          set age 0
          set blood-meal 0
        ]

;        print "babies!"]
;      print count mosquitos with [color = white]
      set blood-meal 0
    ]


; Some percentage of adult mosquitos die after reproduction
    if random-float 1 < percent-die-oviposit [
      die ]
  ]
  ]

; Mosquitoes have a short life expectancy
  ask mosquitos with [age >= max-age] [
    die]

; Mosquitos die off after september
  ask mosquitos with [color != white] [
    if member? current-month [10 11 12 1 2 3] and temperature < 1 [
      die ]
  ]


end

;~~~~~~ ++++++++   ++        ++++++++   +++++  +++++   ++++++++   ++++++++  +++++++  ~~~~~~~~~~~~
;~~~~~~ ++         ++           ++      ++ ++ ++  ++   ++    ++      ++     ++       ~~~~~~~~~~~~
;~~~~~~ ++         ++           ++      ++  ++++  ++   ++    ++      ++     ++       ~~~~~~~~~~~~
;~~~~~~ ++         ++           ++      ++   ++   ++   ++    ++      ++     +++++    ~~~~~~~~~~~~
;~~~~~~ ++         ++           ++      ++        ++   ++++++++      ++     ++       ~~~~~~~~~~~~
;~~~~~~ ++         ++           ++      ++        ++   ++    ++      ++     ++       ~~~~~~~~~~~~
;~~~~~~ ++         ++           ++      ++        ++   ++    ++      ++     ++       ~~~~~~~~~~~~
;~~~~~~ ++++++++   ++++++++  ++++++++   ++        ++   ++    ++      ++     +++++++  ~~~~~~~~~~~~

to temp

  ; This is to set random temperatures for the meterological seasons of DJF, MAM, JJA, SON
  ; DJF is between -30 and 00
  ; MAM is between -05 and 05
  ; JJA is between  10 and 15
  ; SON is between  00 and 05
  set current-month time:get "month" current-date

  ifelse member? current-month [1 2 12] [
    set temperature random -30
  ][
    ifelse member? current-month [3 4 5] [
      set temperature -5 + random 5
    ][
      ifelse member? current-month [6 7 8] [
        set temperature 10 + random 15
      ][
        set temperature 0 + random 5
      ]
    ]
  ]

end



to precipitation


; procedure to create random snow when the temperature is below 0.  for now if the month is DJF, snowfall has a probability of 30%
; if the month is MAM, snowfall has a probability of 15%
; if the month is JJA, snowfall has a 5%
; if the month is SON, snowfall has a 10%

; the snow should "disappear" after 3 consecutive days of temp over 0 degrees

  let precip-prob 0
  let precip-dens 0
  let snow-prob 0
  let snow-dens 0


  set current-month time:get "month" current-date

  ifelse member? current-month [12 1 2] [
    set precip-prob 5
    set precip-dens 0.05 * world-size
    set snow-prob 30
    set snow-dens 0.5 * world-size
  ]
  [ ifelse member? current-month [3 4 5] [
     set precip-prob 10
     set precip-dens 0.15 * world-size
     set snow-prob 15
     set snow-dens 0.3 * world-size
    ]
    [ ifelse member? current-month [6 7 8] [
      set precip-prob 30
      set precip-dens 0.40 * world-size
      set snow-prob 5
      set snow-dens 0.1 * world-size
      ]
      [ set precip-prob 15
        set precip-dens 0.25 * world-size
        set snow-prob 10
        set snow-dens 0.2 * world-size
      ]
    ]
  ]

  ask n-of snow-dens patches with [pcolor = brown or pcolor = green] [
    if temperature < 1 [
      set pcolor white]
  ]

    ; Snow melts after 3 consecutive warm days (days above 0)
  ask patches with [pcolor = white] [
    if temperature > 0 and cold-warm-days > 2 [
      set pcolor brown]
  ]
  ;Rain

  if temperature > 0 [
    ;Rain on grass or dirt
    ifelse random-float 100 < precip-prob [
      set rainy-days rainy-days + 1
      ask n-of precip-dens patches with [pcolor != blue] [
        set pcolor 104
      ]
    ][
      set rainy-days 0
    ]

    ;Expand or contract ephemeral water depending on how much it rained
    ;if it rains more than 2 consecutive days, the area will pond randomly
    ifelse rainy-days < 2 [
      ask patches with [pcolor = 104] [
        set pcolor brown
      ]
    ][
      ask patches with [pcolor = 104] [
        if any? neighbors with [pcolor = brown or pcolor = green] [
          ask one-of neighbors with [pcolor = brown or pcolor = green] [
            set pcolor 104
            ]
          ]
        ]
      ]
      ]
  ;if it rains for 5+ consecutuve days, then the permanent waterbodies will flood and expand
  if rainy-days > 4 [
    ask patches with [pcolor = green or pcolor = brown] [
      if any? neighbors with [pcolor = blue] [
        set pcolor 107]
    ]
  ]

  ; if it doesn't rain for 2 days, then the permenant water bodies that previously flooded will contract. (I will have to fix this)
  if rainy-days < 2 [
    ask patches with [pcolor = 107] [
      set pcolor brown]
  ]



end
@#$#@#$#@
GRAPHICS-WINDOW
938
10
1504
577
-1
-1
5.525
1
10
1
1
1
0
0
0
1
0
100
0
100
0
0
1
ticks
30.0

BUTTON
204
107
321
140
Setup Reindeer
setup-reindeer
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
7
90
142
123
Setup Patches
setup-patches
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
425
108
539
141
Setup Mosquitoes
setup-mosquitos
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
204
144
338
204
Reindeer-pop
500.0
1
0
Number

SLIDER
204
208
339
241
herd-spread
herd-spread
0
.5
0.15
0.01
1
NIL
HORIZONTAL

INPUTBOX
204
243
340
303
reproduction-rate
0.9
1
0
Number

SLIDER
204
305
340
338
vision
vision
0
100
17.0
1
1
NIL
HORIZONTAL

INPUTBOX
425
147
541
207
Mosquitos-pop
1000.0
1
0
Number

BUTTON
93
10
156
43
Go
go
T
1
T
OBSERVER
NIL
G
NIL
NIL
1

SLIDER
424
210
544
243
mosquito-vision
mosquito-vision
1
100
20.0
1
1
NIL
HORIZONTAL

SLIDER
424
245
545
278
mosquito-bite-day
mosquito-bite-day
10
100
49.0
1
1
NIL
HORIZONTAL

SLIDER
423
282
546
315
blood-meal-size
blood-meal-size
0
1
0.45
0.01
1
NIL
HORIZONTAL

SLIDER
424
318
550
351
blood-max
blood-max
1
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
424
354
551
387
infected-probability
infected-probability
0
1
0.99
.01
1
NIL
HORIZONTAL

SLIDER
203
341
341
374
initial-infected-rate
initial-infected-rate
0
1
0.04
0.01
1
NIL
HORIZONTAL

SLIDER
422
393
556
426
mosquito-move-day
mosquito-move-day
0
10
10.0
1
1
NIL
HORIZONTAL

SLIDER
421
470
559
503
eggs
eggs
20
150
92.0
1
1
NIL
HORIZONTAL

SLIDER
421
507
561
540
max-age
max-age
1
500
140.0
1
1
NIL
HORIZONTAL

MONITOR
791
372
931
417
No. of Adult Mosquitos
count mosquitos with [color != white]
0
1
11

SLIDER
423
434
556
467
percent-die-oviposit
percent-die-oviposit
0
1
0.86
.01
1
NIL
HORIZONTAL

BUTTON
9
10
88
44
Setup All
setup
NIL
1
T
OBSERVER
NIL
S
NIL
NIL
1

SWITCH
195
73
347
106
reindeer-life-cycle?
reindeer-life-cycle?
0
1
-1000

CHOOSER
7
221
145
266
patch-sizes
patch-sizes
"small" "medium" "large"
0

SLIDER
7
186
143
219
patches-amt
patches-amt
1
50
5.0
1
1
NIL
HORIZONTAL

INPUTBOX
7
124
144
184
world-size
100.0
1
0
Number

MONITOR
826
129
926
174
No. of Reindeer
count Reindeers
17
1
11

SWITCH
16
423
130
456
climate-on?
climate-on?
0
1
-1000

SWITCH
16
460
127
493
file-option?
file-option?
1
1
-1000

INPUTBOX
15
498
250
558
File-name
time_series_test.csv
1
0
String

INPUTBOX
17
355
125
415
Date_opt
2000-08-15
1
0
String

MONITOR
723
10
924
55
Current Date
time:show current-date \"EEEE, MMMM d, yyyy\"
17
1
11

MONITOR
787
66
925
111
Daily Temperature (C)
round temperature
17
1
11

MONITOR
789
321
931
366
No. of Larvae
count Mosquitos with [color = white]
17
1
11

MONITOR
714
523
929
568
NIL
count patches with [pcolor = white]
17
1
11

BUTTON
171
10
234
43
Clear
clear-run
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
206
381
277
414
butcher
butcher
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
206
417
378
450
butcher-amt
butcher-amt
0
50
19.0
1
1
NIL
HORIZONTAL

PLOT
596
128
796
278
Larvae Count per Reindeer
reindeer id
larvae-count
0.0
500.0
0.0
100.0
true
false
"" "plot-pen-reset\nforeach sort reindeers [ [t] -> ask t [plot larvae-count] ]"
PENS
"pen-0" 1.0 0 -10873583 true "" ""

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

The next phase will incorporate mosquito reproduction in the more rain-produced ephemeral water bodies and building the relationship between temperature/flooding/ and mosquito hatching speed.

Do Reindeer consume more energy depending on temperature or insect harassment?

Incorporate seasonality in reindeer partorition rates. 

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

moose
false
0
Polygon -7500403 true true 196 228 198 297 180 297 178 244 166 213 136 213 106 213 79 227 73 259 50 257 49 229 38 197 26 168 26 137 46 120 101 122 147 102 181 111 217 121 256 136 294 151 286 169 256 169 241 198 211 188
Polygon -7500403 true true 74 258 87 299 63 297 49 256
Polygon -7500403 true true 25 135 15 186 10 200 23 217 25 188 35 141
Polygon -7500403 true true 270 150 253 100 231 94 213 100 208 135
Polygon -7500403 true true 225 120 204 66 207 29 185 56 178 27 171 59 150 45 165 90
Polygon -7500403 true true 225 120 249 61 241 31 265 56 272 27 280 59 300 45 285 90

mosquito
true
0
Polygon -16777216 true false 168 141 206 96 204 72 235 44 201 72 201 95 165 137
Polygon -16777216 true false 132 148 111 128 109 135 68 131 109 139 113 133 129 150
Polygon -16777216 true false 168 148 189 128 191 135 232 131 191 139 187 133 171 150
Polygon -16777216 true false 165 167 197 205 205 268 223 272 208 265 201 200 167 164 167 164 167 164 165 163
Polygon -16777216 true false 141 164 109 202 101 265 83 269 98 262 105 197 139 161 139 161 139 161 141 160
Polygon -16777216 true false 132 141 94 96 96 72 65 44 99 72 99 95 135 137
Circle -7500403 true true 127 126 46
Circle -7500403 true true 136 101 28
Line -7500403 true 147 103 121 50
Polygon -7500403 true true 137 118 143 143 163 148
Polygon -7500403 true true 146 150 161 154 161 118 141 120 143 151
Polygon -7500403 true true 129 147 137 252 144 262 153 267 161 264 165 255
Polygon -7500403 true true 133 151 154 255 165 254
Polygon -7500403 true true 133 152 158 256 166 254 170 152
Polygon -7500403 true true 137 248 138 260 143 265 153 265 152 246
Polygon -7500403 true true 145 259 151 277 157 259
Line -7500403 true 153 103 179 50
Polygon -16777216 true false 149 99 151 42 152 100
Polygon -16777216 true false 154 101 161 104 164 110 161 112 155 113 152 108 151 102 152 100 155 104
Polygon -16777216 true false 146 101 139 104 136 110 139 112 145 113 148 108 149 102 148 100 145 104
Polygon -16777216 true false 130 166 149 169 169 166 169 180 150 183 132 181
Polygon -16777216 true false 128 187 150 190 170 186 169 200 151 204 129 199
Polygon -16777216 true false 133 209 151 213 170 209 167 224 152 227 133 223
Polygon -16777216 true false 133 230 151 234 168 230 167 245 152 248 133 244
Polygon -16777216 true false 136 250 154 254 166 252 165 262 154 268 138 265
Line -7500403 true 239 40 202 72
Line -7500403 true 61 40 98 72
Line -7500403 true 208 267 233 274
Line -7500403 true 97 265 69 272
Polygon -1 true false 170 160 271 139 280 144 287 152 291 165 291 174 285 183 271 187 259 192 233 197 210 193 183 175 169 163
Polygon -1 true false 130 160 29 139 20 144 13 152 9 165 9 174 15 183 29 187 41 192 67 197 90 193 117 175 131 163
Line -16777216 false 270 140 171 161
Line -16777216 false 30 140 129 161
Line -16777216 false 183 159 187 174
Line -16777216 false 117 159 113 174
Line -16777216 false 185 161 205 169
Line -16777216 false 115 161 95 169
Line -16777216 false 192 165 211 182
Line -16777216 false 108 165 89 182
Line -16777216 false 251 144 277 147
Line -16777216 false 49 144 23 147
Line -16777216 false 245 146 281 151
Line -16777216 false 55 146 19 151
Line -16777216 false 266 149 242 154
Line -16777216 false 34 149 58 154
Line -16777216 false 247 155 284 159
Line -16777216 false 53 155 16 159
Polygon -16777216 false false 270 140 279 147 286 157 288 168 284 177 279 182 259 189 236 194 214 191 175 163 173 160
Polygon -16777216 false false 30 140 21 147 14 157 12 168 16 177 21 182 41 189 64 194 86 191 125 163 127 160
Polygon -16777216 false false 199 156 206 168 216 176 240 182 263 182 274 181 280 177 285 174 268 177 247 171 238 162 218 162 210 152
Polygon -16777216 false false 101 156 94 168 84 176 60 182 37 182 26 181 20 177 15 174 32 177 53 171 62 162 82 162 90 152
Polygon -16777216 false false 219 150 239 154 253 155 272 165 283 164
Polygon -16777216 false false 81 150 61 154 47 155 28 165 17 164
Line -16777216 false 239 163 273 171
Line -16777216 false 61 163 27 171
Line -16777216 false 204 154 211 166
Line -16777216 false 96 154 89 166
Line -16777216 false 210 166 239 177
Line -16777216 false 90 166 61 177
Line -16777216 false 239 176 270 176
Line -16777216 false 61 176 30 176
Line -16777216 false 211 183 234 189
Line -16777216 false 89 183 66 189
Line -16777216 false 231 188 260 188
Line -16777216 false 69 188 40 188

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

reindeer
false
0
Polygon -16777216 true false 223 114 247 85 241 23 254 78 249 95 279 77 298 39 283 84
Polygon -16777216 true false 244 107 272 73 270 31 302 25 263 29 265 71 252 85
Polygon -7500403 true true 198 228 200 297 182 297 180 244 168 213 138 213 108 213 81 227 75 259 52 257 51 229 40 197 28 168 28 137 48 120 103 122 151 121 183 111 219 121 255 120 292 137 288 157 258 159 237 187 213 188
Polygon -7500403 true true 74 258 87 299 63 297 49 256
Polygon -7500403 true true 27 135 16 156 16 177 27 196 24 183 36 140
Polygon -7500403 true true 257 138 253 100 231 94 213 100 208 135
Polygon -16777216 true false 201 98 173 64 175 22 143 16 182 20 180 62 193 76
Polygon -16777216 true false 249 89 242 50 230 37 228 18 234 34 247 45
Polygon -16777216 true false 196 88 203 49 215 36 217 17 211 33 198 44
Polygon -16777216 true false 222 113 199 84 203 23 190 78 195 95 165 77 146 39 161 84
Polygon -1 true false 211 117 233 153 251 170 254 188 249 200 241 200 228 202 217 205 210 198 212 185 200 169 174 155 166 116 181 112
Polygon -1 true false 25 194 26 183 29 164 52 232 53 204 35 153 26 159
Polygon -16777216 true false 288 135 284 143 289 145 292 138

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

sheep 2
false
0
Polygon -7500403 true true 209 183 194 198 179 198 164 183 164 174 149 183 89 183 74 168 59 198 44 198 29 185 43 151 28 121 44 91 59 80 89 80 164 95 194 80 254 65 269 80 284 125 269 140 239 125 224 153 209 168
Rectangle -7500403 true true 180 195 195 225
Rectangle -7500403 true true 45 195 60 225
Rectangle -16777216 true false 180 225 195 240
Rectangle -16777216 true false 45 225 60 240
Polygon -7500403 true true 245 60 250 72 240 78 225 63 230 51
Polygon -7500403 true true 25 72 40 80 42 98 22 91
Line -16777216 false 270 137 251 122
Line -16777216 false 266 90 254 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
