globals
[ top-row   ; used by drop-marbles
  field     ; the patches that are not border patches
  border    ; the border patches
  bins      ; the patches that make up the bottom bins
  pegs      ; the patches that are pegs
  Fill      ; Use to start/stop filling
  Field-color ; color of the background (field)
  Marble-color ; color of the marbles
  Peg-color ; normal grid color
  Bin-color ; color for bins
  Right-color ; pegs biased to the right
  Left-color ; pegs biased to the left
  Border-color ; color of the border patches
  Distr-color ; color to mark distribution on bins
  Hole-size ; how big is the hole from which marbles fall?
  Bin-height ; how tall are the bins?
  Pegs-top ; where at the top do the pegs begin?
  Pegs-bottom ; where at the bottom do the pegs end?
  total-error ;
  old-d-amp; Use these variables to see if Gaussian distro changed
  old-d-stdev;
  old-d-mean;
]

breed [ marbles marble ]

patches-own
[ void? ; true means patch is empty, and not an obstacle
  my_error ; used by distribution patches
]

to setup
   ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
   clear-all
   ask patches [ set void? true ]
   if use-rseed [ random-seed r-seed ]
   set-color-scheme
   set Bin-height 0.4 * max-pycor
   set Pegs-top 0.95 * max-pycor
   set Pegs-bottom max-pycor * -0.5
   reset-screen-regions
   set top-row patches with [ pycor = max-pycor ]
   set Hole-size 2
   set Fill false
   make-hole
   make-pegs
   make-bins
   reset-ticks
   print "Initialization complete"
end

to go
   ; if using mouse, all else pauses
   ifelse mouse-down?
   [ do-mouse-action ]
   [ ; evaluate the run-time effect switches
     if Fill = true  [ drop-marbles  ]
     ifelse Distribution != "None"
     [ show-dist ]
     [ clear-distr ]
     ; marbles fall
     ; without-interruption
     ask marbles [ fall-down ]
   ]
end

to set-color-scheme
     set Field-color black
     set Marble-color green
     set Bin-color 35
     set Peg-color 3
     set Border-color sky
     set Right-color 15
     set Left-color 95
     set Distr-color white
end

to define-field
   set field patches with
   [     abs pxcor < max-pxcor
     and pycor > min-pycor
   ]
end

to define-border
   set border patches with
   [ abs pxcor = max-pxcor or pycor = min-pycor
   ]
end

to define-bins
   set bins patches with
   [ pycor <= min-pycor + Bin-height and pxcor mod 3 = 0
     and pycor != min-pycor and abs pxcor != max-pxcor
   ]
end

to define-pegs
   set pegs patches with
   [ pycor >= Pegs-bottom and pycor <= Pegs-top and  (pycor + 1) mod 2 = 0
     and floor (pxcor + pycor / 2) mod 2 = 0
     and pycor != min-pycor and abs pxcor != max-pxcor
   ]
end

to-report field-not-defined?
   report not is-patch-set? field
end

to-report border-not-defined?
   report not is-patch-set? border
end

to-report bins-not-defined?
   report not is-patch-set? bins
end

to-report pegs-not-defined?
   report not is-patch-set? pegs
end

to define-screen-regions
   if field-not-defined?   [ define-field  ]
   if bins-not-defined?    [ define-bins ]
   if border-not-defined?  [ define-border ]
   if pegs-not-defined?     [ define-pegs ]
end

to reset-screen-regions
   define-screen-regions
   clear-field
   set-border
end

to clear-field
   ask marbles [ pop ]
   ask field [ clear-grid ]
end

to set-border
   ask border
   [ set void? false
     set pcolor Border-color
   ]
end

to set-fill
  ifelse Fill
  [ set Fill false]
  [ set Fill true ]
end

to drop-marbles
   let open-patches top-row with [ void? = true ]
   if random 100 < drop-rate and count marbles < max-marbles
   [
     if any? open-patches
     [ ask one-of open-patches
       [ make-marble ]
     ]
   ]

end

to make-marble ; patch procedure
     sprout 1
     [ set breed marbles
       set shape "circle"
       set heading 180
       set void? false
       set color Marble-color
     ]
end

to fall-down
   let below patch-at 0 -1
   let below-left patch-at -1 -1
   let below-right patch-at 1 -1
   ;let beside patches at-points [ [ -1 0 ] [ 1 0 ] ]
   ; Adding to fix at-points issue
   let beside-left patch-at -1 0
   let beside-right patch-at 1 0
   let beside (patch-set patch-at -1 0 patch-at 1 0)

   ; if open space directly below, fall straight down
   if [void?] of below = true
   [ set void? true
     set ycor ycor - 1
     set void? false
     stop
   ]

   ; check if there is a biased grid point below.
   if [pcolor] of below = Right-color and [void?] of below-right = true
     and random 100 < bias-strength
     [ set void? true
;       set xcor pxcor-of below-right
;       set ycor pycor-of below-right
       setxy [pxcor] of below-right [pycor] of below-right
       set void? false
       stop
     ]

   if [pcolor] of below = Left-color and [void?] of below-left = true
     and random 100 < bias-strength
     [ set void? true
;       set xcor pxcor-of below-left
;       set ycor pycor-of below-left
       setxy [pxcor] of below-left [pycor] of below-left
       set void? false
       stop
     ]

;   This part gets executed if the space below is occupied, either by an unbiased
;   peg, or by another marble, or by a bin wall. In this case see if there is space
;   below to the left or right. If both are available, choose one randomly.
;   [
     set beside beside with [ void? = true and [void?] of patch-at 0 -1 = true ]
     if count beside > 0
      [ set beside one-of beside
        set void? true
        setxy [pxcor] of beside ( pycor - 1 )
        set void? false
        stop
;      ]

   ]

end

; this procedure can do two things: set some of the pins to have a bias, or draw a distribution on the pegs
to do-mouse-action

   ifelse Distribution != "Manual" or mouse-ycor >= Pegs-bottom
   [
     let biased pegs with [ abs ( mouse-xcor - pxcor ) <= Bias-radius
                          and abs ( mouse-ycor - pycor ) <= Bias-radius ]
     ask biased [
       ifelse Bias-direction = "Left"
       [ set pcolor Left-color ]
       [ ifelse Bias-direction = "Right"
         [ set pcolor Right-color ]
         [ set pcolor Peg-color ]
       ]
     ] ; ask biased
   ]
   [
     let marked bins with [  abs ( pxcor - mouse-xcor ) < 1 ];  and abs ( pycor - mouse-ycor ) < 1 ]
     ask marked [
       ifelse pycor = round mouse-ycor
       [ set pcolor Distr-color ]
       [ set pcolor Bin-color]
     ]
   ]

end

to clear-marbles
   ;without-interruption
   ask marbles [ pop ]
end

to clear-bias
   ;without-interruption
   ask pegs
     [ set-color-grid Peg-color ]
end

to clear-distr
   ;without-interruption
   ask bins with [ pcolor = Distr-color ]
     [ set-color-grid Bin-color ]
end

to make-hole
   ask field with [ pycor = max-pycor and abs ( pxcor ) > Hole-size ]
   [ set-color-grid Border-color ]
end

to make-pegs
   ask pegs
   [ set-color-grid Peg-color ]
end

to make-bins
   ask bins
   [ set-color-grid Bin-color ]
end

; superimpose the distribution on the bin columns
to show-dist
   set total-error 0
   if ( Distribution = "Gaussian" ) and Gaussian-unchanged [ stop ]
   ask bins
    [
     if Distribution = "Dynamic"
      [ ifelse ( [void?] of patch-at -1 0 = false or [void?] of patch-at 1 0 = false )
               and ( [void?] of patch-at -1 1 = true and [void?] of patch-at 1 1 = true
                     or [pcolor] of patch-at 0 1 = Field-color )
        [ set pcolor Distr-color ]
        [ set pcolor Bin-color ]
        stop
      ]

     if Distribution = "Gaussian"
      [
         ifelse pycor = min-pycor +
          floor ( d-amp * Bin-height * exp ( - ( pxcor - d-mean ) * ( pxcor - d-mean ) / ( d-stdev * d-stdev ) ) )
          [ set pcolor Distr-color
            set my_error 0
            let left-col min-pycor
            let right-col min-pycor
            let not-found true
          ]
          [ set pcolor Bin-color
            set my_error 0
          ]
        stop
      ]
   ]

end

; This reporter checks whether the sliders for the Gaussian have changed
to-report Gaussian-unchanged
  ifelse ( old-d-amp = d-amp ) and ( old-d-stdev = d-stdev ) and ( old-d-mean = d-mean )
  [ set old-d-amp d-amp
    set old-d-stdev d-stdev
    set old-d-mean d-mean
    report true
  ]
  [ set old-d-amp d-amp
    set old-d-stdev d-stdev
    set old-d-mean d-mean
    report false
  ]


end

to clear-grid
   set pcolor Field-color
   set void? true
end

to set-color-grid [ gcolor ]
   set pcolor gcolor
   set void? false
end

to pop
   hide-turtle
   set void? true
   die
end
@#$#@#$#@
GRAPHICS-WINDOW
234
10
577
604
-1
-1
5.0
1
10
1
1
1
0
1
1
1
-33
33
-58
58
0
0
1
ticks
30.0

BUTTON
10
139
76
172
Init
Setup
NIL
1
T
OBSERVER
NIL
I
NIL
NIL
1

BUTTON
83
139
146
172
Run
go
T
1
T
OBSERVER
NIL
R
NIL
NIL
1

MONITOR
581
483
700
528
marbles dropped
count marbles
3
1
11

SLIDER
10
235
180
268
drop-rate
drop-rate
0
20
10.0
1
1
NIL
HORIZONTAL

BUTTON
10
199
98
232
Clear marbles
clear-marbles
NIL
1
T
OBSERVER
NIL
C
NIL
NIL
1

SLIDER
11
351
183
384
Bias-radius
Bias-radius
1
20
3.0
1
1
NIL
HORIZONTAL

BUTTON
10
303
98
348
Clear Pegs
clear-bias
NIL
1
T
OBSERVER
NIL
P
NIL
NIL
1

BUTTON
151
139
220
172
Drop
set-fill
T
1
T
OBSERVER
NIL
M
NIL
NIL
1

TEXTBOX
9
283
159
301
Bias settings\n
11
0.0
0

TEXTBOX
10
181
160
199
Marble settings
11
0.0
0

CHOOSER
101
303
223
348
Bias-direction
Bias-direction
"Left" "Right" "None"
1

BUTTON
13
455
110
500
Clear Distr
clear-distr
NIL
1
T
OBSERVER
NIL
D
NIL
NIL
1

TEXTBOX
13
437
163
455
Distribution
11
0.0
0

CHOOSER
116
455
212
500
Distribution
Distribution
"None" "Gaussian" "Dynamic" "Manual"
1

SLIDER
11
387
183
420
bias-strength
bias-strength
0
100
36.0
1
1
%
HORIZONTAL

SLIDER
14
505
106
538
d-mean
d-mean
-21
21
0.0
1
1
NIL
HORIZONTAL

SLIDER
14
542
106
575
d-stdev
d-stdev
0
50
11.0
1
1
NIL
HORIZONTAL

SLIDER
588
570
694
603
r-seed
r-seed
0
10
0.0
1
1
NIL
HORIZONTAL

SLIDER
109
505
201
538
d-amp
d-amp
0.0
1.0
0.9
0.1
1
NIL
HORIZONTAL

SWITCH
588
535
694
568
use-rseed
use-rseed
1
1
-1000

SLIDER
101
199
220
232
max-marbles
max-marbles
10
1000
250.0
10
1
NIL
HORIZONTAL

TEXTBOX
18
28
227
126
* Select the parameters you want\n* Hit \"Run\" then \"Drop\"\n* Hit \"Clear marbles\" to drop more\n* Select Bias-Direction then click-drag to bias some pegs\n* Play around with distributions\nRead the INFO tab for more
11
0.0
1

TEXTBOX
70
10
159
28
INSTRUCTIONS
12
0.0
1

@#$#@#$#@
## WHAT IS IT?

This is a simulation of falling marbles to illustrate certain points about statistics and modeling.

## HOW IT WORKS

Marbles fall from the top and bounce over pegs to the left or to the right. At the bottom they are collected into bins. The distribution of marbles across bins can be observed and modified by applying a "bias" to some of the pegs.

When there are no biases, the resulting distribution approximates a Gaussian (or "Bell-shaped" or "Normal" distribution). Groups of pegs can be biased to the left or to the right by varying degrees.

## HOW TO USE IT

=== Main Control Buttons ===
- Click "Init" to reset the entire simulation and draw the pegs and the bins
- Click "Go" to start/end the simulation.
- Click "Drop" to start/stop dropping marbles through the hole at the top.

=== Marble Settings ===
- Click "Clear Marbles" to erase all Marbles.
- Use the "max-marbles" slider to determine hoe many marbles to drop (good default is 250)
- Adjust the "drop-rate" slider to determine how quickly marbles drop from the hole.

=== Bias Settings ===
Use these controls to change some of the pegs to be biased to the left or right. Click anywhere over the pegs to bias some of them, based on the various settings.
- Use the "Bias-direction" drop-down to select Left, Right or None.
- Use the "Bias-radius" slider to determine how large of a patch will be biased around the point where you click the mouse.
- Use the "bias-strength" slider to set the bias. For example, when a marble hits a red peg (Right bias) with the slider set to 100%, the marble will always go to the right. At a 0% bias, the marble has a 50-50 chance of falling to the right. [Technically, the probability of the marble going to the right is 0.5*bias + 0.5.]

=== Distribution Settings ===
- Click "Clear Distr" to clear the red dots showing the distribution on the bins.
- Use the "Distribution" drop-down to select one of the following modes:
_* None: do not show distribution, or, if a distribution was being shown, do not change it. Once "Init" or "Clear Distr" are pressed, any remaining distribution will be erased.
_* Dynamic: For each bin, place a dot corresponding to the highest marble stacked on its left (not ideal but gets the point across). This changes dynamically as the marbles drop to match the current distribution.
_* Gaussian: Draw a Gaussian distribution based on the sliders below
_* Manual: Use the mouse to draw a distribution by clicking on individual bins. Note that you can also click-and-drag to draw the distribution.

- Use the "d-mean" "d-amp" and "d-stdev" sliders to set the mean, amplitude and standard deviation of the Gaussian distribution.

=== Additional controls ===
- The toggle "use-rseed" determines whether at initialization the random number generator uses a seed or not. If you want to replicate a run, turn this ON and select a desired seed.
- Use the "r-seed" slider to set a random number seed between 0 and 10
- The "# of marbles" box is a counter showing how many marbles have been dropped.

## WHAT THIS IS ABOUT

The main point of this demo is to illustrate the difference between statistical approaches and agent-based or "bottom-up" approaches to modeling.

When there are no biases, it can be shown mathematically that the distribution of falling marbles should approach a Gaussian distribution. In fact, if you run this with about 250 marbles and no biases, you should be able to fit the resulting distribution with a Gaussian with mean 0, amplitude 1 and standard deviation 10.

However, suppose now that someone shows you a distribution which is not Gaussian. Perhaps it is skewed to one side, or even bimodal. A statistical modeling approach would try to find a different mathematical function (perhaps a gamma or beta distribution) and adjust parameters until it minimizes the error between theoretical and empirical distributions. However, this approach doesn't really say anything about WHY the distribution should be the way it is, nor is there any relationship between the mathematical expression for the distribution and the underlying behavior.

In contrast, using a bottom-up modeling approach, we can make speculations about WHY a certain distribution arises. For instance, if you see a distribution where most marbles end up on the right, you can assume that some pegs must be biased to the right. By selecting different pegs you can make adjustments until you get close to the observed distribution.

Conversely, you can make hypotheses about the impact of different biases. For instance, what happens if all the pegs on the right are biased to the right and all the pegs on the left are biased to the left? Is there a difference between biasing the first five rows of pegs versus biasing the bottom five rows of pegs?

Now imagine the pegs representing people expressing opinions about a product. Each "person" decides whether he or she thinks the product is good (right) or bad (left). Each person then expresses that opinion to another person. Now imagine one person is biased in one direction. If that person can also influence many of its friends through word-of-mouth, a big cluster of biased individuals can cause the opinion of the entire population to shift.

Modeling these kinds of effects is nearly impossible with statistical approaches. You might even imagine a more complex situation, in which the bias of individuals is changed gradually over time by watching what other people decided. Bottom-up modeling gives you the ability to test different conditions, and to do so based on direct, intuitive terms, such as "what happens if this group is biased?" or "where is the most effective group to bias?"

By comparison, with a statistical approach you might be able to ask questions like "what if the standard deviation of the distribution increases?" or "what if I use a Gamma distribution instead of a Gaussian?"

In a nutshell, bottom-up modeling gives you the power to explore and understand causality, while statistical approaches at best can tell you about correlations between variables.

## BUGS ETC

The "manual" distribution should place dots at the average between the highest marbles on the left and right sides.

It would be nice to let the bias change as a function of several other factros, such as what other patches are doing, or what information the marbles carry.

## COPYRIGHT

This work was created by Paolo Gaudiano and is copyright of Icosystem Corporation. This may not be used for commercial purposes without prior consent of Icosystem. For additional information please visit our web site (http://www.icosystem.com) or send mail to info@icosystem.com
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

ant
true
0
Polygon -7500403 true true 136 61 129 46 144 30 119 45 124 60 114 82 97 37 132 10 93 36 111 84 127 105 172 105 189 84 208 35 171 11 202 35 204 37 186 82 177 60 180 44 159 32 170 44 165 60
Polygon -7500403 true true 150 95 135 103 139 117 125 149 137 180 135 196 150 204 166 195 161 180 174 150 158 116 164 102
Polygon -7500403 true true 149 186 128 197 114 232 134 270 149 282 166 270 185 232 171 195 149 186
Polygon -7500403 true true 225 66 230 107 159 122 161 127 234 111 236 106
Polygon -7500403 true true 78 58 99 116 139 123 137 128 95 119
Polygon -7500403 true true 48 103 90 147 129 147 130 151 86 151
Polygon -7500403 true true 65 224 92 171 134 160 135 164 95 175
Polygon -7500403 true true 235 222 210 170 163 162 161 166 208 174
Polygon -7500403 true true 249 107 211 147 168 147 168 150 213 150

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bee
true
0
Polygon -1184463 true false 152 149 77 163 67 195 67 211 74 234 85 252 100 264 116 276 134 286 151 300 167 285 182 278 206 260 220 242 226 218 226 195 222 166
Polygon -16777216 true false 150 149 128 151 114 151 98 145 80 122 80 103 81 83 95 67 117 58 141 54 151 53 177 55 195 66 207 82 211 94 211 116 204 139 189 149 171 152
Polygon -7500403 true true 151 54 119 59 96 60 81 50 78 39 87 25 103 18 115 23 121 13 150 1 180 14 189 23 197 17 210 19 222 30 222 44 212 57 192 58
Polygon -16777216 true false 70 185 74 171 223 172 224 186
Polygon -16777216 true false 67 211 71 226 224 226 225 211 67 211
Polygon -16777216 true false 91 257 106 269 195 269 211 255
Line -1 false 144 100 70 87
Line -1 false 70 87 45 87
Line -1 false 45 86 26 97
Line -1 false 26 96 22 115
Line -1 false 22 115 25 130
Line -1 false 26 131 37 141
Line -1 false 37 141 55 144
Line -1 false 55 143 143 101
Line -1 false 141 100 227 138
Line -1 false 227 138 241 137
Line -1 false 241 137 249 129
Line -1 false 249 129 254 110
Line -1 false 253 108 248 97
Line -1 false 249 95 235 82
Line -1 false 235 82 144 100

bird1
false
0
Polygon -7500403 true true 2 6 2 39 270 298 297 298 299 271 187 160 279 75 276 22 100 67 31 0

bird2
false
0
Polygon -7500403 true true 2 4 33 4 298 270 298 298 272 298 155 184 117 289 61 295 61 105 0 43

boat1
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 158 33 230 157 182 150 169 151 157 156
Polygon -7500403 true true 149 55 88 143 103 139 111 136 117 139 126 145 130 147 139 147 146 146 149 55

boat2
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 157 54 175 79 174 96 185 102 178 112 194 124 196 131 190 139 192 146 211 151 216 154 157 154
Polygon -7500403 true true 150 74 146 91 139 99 143 114 141 123 137 126 131 129 132 139 142 136 126 142 119 147 148 147

boat3
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 158 37 172 45 188 59 202 79 217 109 220 130 218 147 204 156 158 156 161 142 170 123 170 102 169 88 165 62
Polygon -7500403 true true 149 66 142 78 139 96 141 111 146 139 148 147 110 147 113 131 118 106 126 71

box
true
0
Polygon -7500403 true true 45 255 255 255 255 45 45 45

butterfly1
true
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7500403 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7500403 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7500403 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7500403 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7500403 true 167 47 159 82
Line -7500403 true 136 47 145 81
Circle -7500403 true true 165 45 8
Circle -7500403 true true 134 45 6
Circle -7500403 true true 133 44 7
Circle -7500403 true true 133 43 8

circle
false
0
Circle -7500403 true true 35 35 230

line
true
0
Line -7500403 true 150 0 150 300

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

person
false
0
Circle -7500403 true true 155 20 63
Rectangle -7500403 true true 158 79 217 164
Polygon -7500403 true true 158 81 110 129 131 143 158 109 165 110
Polygon -7500403 true true 216 83 267 123 248 143 215 107
Polygon -7500403 true true 167 163 145 234 183 234 183 163
Polygon -7500403 true true 195 163 195 233 227 233 206 159

sheep
false
15
Rectangle -1 true true 90 75 270 225
Circle -1 true true 15 75 150
Rectangle -16777216 true false 81 225 134 286
Rectangle -16777216 true false 180 225 238 285
Circle -16777216 true false 1 88 92

spacecraft
true
0
Polygon -7500403 true true 150 0 180 135 255 255 225 240 150 180 75 240 45 255 120 135

thin-arrow
true
0
Polygon -7500403 true true 150 0 0 150 120 150 120 293 180 293 180 150 300 150

truck-down
false
0
Polygon -7500403 true true 225 30 225 270 120 270 105 210 60 180 45 30 105 60 105 30
Polygon -8630108 true false 195 75 195 120 240 120 240 75
Polygon -8630108 true false 195 225 195 180 240 180 240 225

truck-left
false
0
Polygon -7500403 true true 120 135 225 135 225 210 75 210 75 165 105 165
Polygon -8630108 true false 90 210 105 225 120 210
Polygon -8630108 true false 180 210 195 225 210 210

truck-right
false
0
Polygon -7500403 true true 180 135 75 135 75 210 225 210 225 165 195 165
Polygon -8630108 true false 210 210 195 225 180 210
Polygon -8630108 true false 120 210 105 225 90 210

turtle
true
0
Polygon -7500403 true true 138 75 162 75 165 105 225 105 225 142 195 135 195 187 225 195 225 225 195 217 195 202 105 202 105 217 75 225 75 195 105 187 105 135 75 142 75 105 135 105

wolf
false
0
Rectangle -7500403 true true 15 105 105 165
Rectangle -7500403 true true 45 90 105 105
Polygon -7500403 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7500403 true true 105 120 263 195
Rectangle -7500403 true true 108 195 259 201
Rectangle -7500403 true true 114 201 252 210
Rectangle -7500403 true true 120 210 243 214
Rectangle -7500403 true true 115 114 255 120
Rectangle -7500403 true true 128 108 248 114
Rectangle -7500403 true true 150 105 225 108
Rectangle -7500403 true true 132 214 155 270
Rectangle -7500403 true true 110 260 132 270
Rectangle -7500403 true true 210 214 232 270
Rectangle -7500403 true true 189 260 210 270
Line -7500403 true 263 127 281 155
Line -7500403 true 281 155 281 192

wolf-left
false
3
Polygon -6459832 true true 117 97 91 74 66 74 60 85 36 85 38 92 44 97 62 97 81 117 84 134 92 147 109 152 136 144 174 144 174 103 143 103 134 97
Polygon -6459832 true true 87 80 79 55 76 79
Polygon -6459832 true true 81 75 70 58 73 82
Polygon -6459832 true true 99 131 76 152 76 163 96 182 104 182 109 173 102 167 99 173 87 159 104 140
Polygon -6459832 true true 107 138 107 186 98 190 99 196 112 196 115 190
Polygon -6459832 true true 116 140 114 189 105 137
Rectangle -6459832 true true 109 150 114 192
Rectangle -6459832 true true 111 143 116 191
Polygon -6459832 true true 168 106 184 98 205 98 218 115 218 137 186 164 196 176 195 194 178 195 178 183 188 183 169 164 173 144
Polygon -6459832 true true 207 140 200 163 206 175 207 192 193 189 192 177 198 176 185 150
Polygon -6459832 true true 214 134 203 168 192 148
Polygon -6459832 true true 204 151 203 176 193 148
Polygon -6459832 true true 207 103 221 98 236 101 243 115 243 128 256 142 239 143 233 133 225 115 214 114

wolf-right
false
3
Polygon -6459832 true true 170 127 200 93 231 93 237 103 262 103 261 113 253 119 231 119 215 143 213 160 208 173 189 187 169 190 154 190 126 180 106 171 72 171 73 126 122 126 144 123 159 123
Polygon -6459832 true true 201 99 214 69 215 99
Polygon -6459832 true true 207 98 223 71 220 101
Polygon -6459832 true true 184 172 189 234 203 238 203 246 187 247 180 239 171 180
Polygon -6459832 true true 197 174 204 220 218 224 219 234 201 232 195 225 179 179
Polygon -6459832 true true 78 167 95 187 95 208 79 220 92 234 98 235 100 249 81 246 76 241 61 212 65 195 52 170 45 150 44 128 55 121 69 121 81 135
Polygon -6459832 true true 48 143 58 141
Polygon -6459832 true true 46 136 68 137
Polygon -6459832 true true 45 129 35 142 37 159 53 192 47 210 62 238 80 237
Line -16777216 false 74 237 59 213
Line -16777216 false 59 213 59 212
Line -16777216 false 58 211 67 192
Polygon -6459832 true true 38 138 66 149
Polygon -6459832 true true 46 128 33 120 21 118 11 123 3 138 5 160 13 178 9 192 0 199 20 196 25 179 24 161 25 148 45 140
Polygon -6459832 true true 67 122 96 126 63 144
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
