extensions [table]

globals [
  ec-added ;;keeps track of contaminants added
  ms ;;model speed adjuster. Not fully implemented
  Kr ;;Enzyme Kinetics Options
  Kc
  Kd
  view-number ;;Used to name the image downloaded with 'export view'
  interface-number
  complexes-formed ;;Keeps track of Enzyme + Substrate complexes formed. Used to calculate % dissociate
  dissociated ;; Count of dissociated complexes
  fungi-option ;;stores name of fungi species selected
  status-1 ;; (Comment) I can do this more efficiently
  status-2
  status-3
  my-list1 ;;Used by clock. The code is inside the plot options
  my-list2
  intracellularly ;; Used to count EC's degraded intracellularly
  fungi-dict ;;Dictionary/Table of values unique to each fungi species
  adjuster-enzymes ;; adjusts production and efficiency of enzymes based on amount of them. Capped at 280
  adjuster-fungi ;;Used to decide amount of starting NH4 given to offsprings
  ;;eph
  ;;fph
  ;;ftemp
]

fungi-own [
  age
  energy
  NH4
  partner
]

patches-own [
  glucose
  O2 ;; (Comment) Remove eventually
  lignin
]

enzymes-own [
  age
  partner
  turn
  pause-time ;; Enzyme won't go into complex again, right after dissociating
]

ECs-own [
  partner
  locked-time ;; Counts time in Enzyme Substrate complex or Intracellular process
  taken ;; True if being degraded Intracellularly
]

breed [fungi fungus]
breed [ECs EC]
breed [enzymes enzyme]


to setup
  clear-all
  setup-globals
  setup-patches
  setup-fungi
  setup-ECs
  reset-ticks
end

to go
  if ticks > 400 and not any? ECs [
    output-print (word "Success! All Contaminants Removed in " ticks " ticks.")
    stop]
  if ticks > 1000 [
    output-print (word (ec-added - count ECs)" of " ec-added " Contaminants Removed.")
    stop]
  if fungi-species != fungi-option [
    setup
    stop]

  diffuse glucose (0.5 + (0.01 * temperature)) ;;diffuse glucose in patches
  diffuse O2 (0.5 + (0.01 * temperature))

  set adjuster-enzymes round max(list ((count enzymes)^(1 / 4)) 1)
  set adjuster-fungi round max(list ((count enzymes)^(1 / 5)) 1)
  ;;if background-color [color-patches]
  move

  break-down-lignin
  eat-glucose
  replenish-O2

  reproduce
  ;;produce-enzymes ;; called with if/else in reproduce
  produce-ECs

  age-fungus
  age-enzymes
  check-death
  nutrient-status

  intracellular-degrade
  denature
  change-rate
  degrade-ECs
  tick
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup-globals
  set ms 0.5
  set view-number 1
  set interface-number 1
  set fungi-option fungi-species
  set status-1 false
  set status-2 false
  set status-3 false
  set fungi-dict table:make
end

to setup-patches
  ask patches [
    set pcolor blue
    set glucose 4
    set lignin 100
    set O2 25
    ;;Net values are 32 * 32 * of these
  ]
end

to setup-fungi
  create-fungi 6 [;;set the starting number of fungi
    if fungi-species = "Phanerodontia Chrysosporium" [
    set shape "fungi-tendrils"
    set size 2.25
    table:put fungi-dict "fungi-growth-temp" 30
    table:put fungi-dict "fungi-growth-pH" 2
    table:put fungi-dict "enzyme-production-cost" 1
    table:put fungi-dict "enzyme-production-pH" 4
    ;;intracellular-radius
    ]
    if fungi-species = "Trametes Versicolor" [
    set shape "turkey-tail-3"
    set size 2
    table:put fungi-dict "fungi-growth-temp" 24
    table:put fungi-dict "fungi-growth-pH" 1
    table:put fungi-dict "enzyme-production-cost" 1
    table:put fungi-dict "enzyme-production-pH" 3.75
    ]
    if fungi-species = "Pleurotus Ostreatus" [
    set shape "Button-Mushroom"
    set size 2
    table:put fungi-dict "fungi-growth-temp" 33.5
    table:put fungi-dict "fungi-growth-pH" 4
    table:put fungi-dict "enzyme-production-cost" 2.5
    table:put fungi-dict "enzyme-production-pH" 5
    ]
  ;;table:put fungi-dict "enzyme-denature-pH" 4
  ;;table:put fungi-dict "enzyme-reaction-pH" 4 ;; I'm gonna hard code these in
  set color 1
  setxy random -28 + 14 random -28 + 14 ;;place the fungi at random locations, but not edges (display is -16 to 16)
  set energy 30
  set NH4 12
  set partner nobody
  ]
  ;;set eph table:get fungi-dict "enzyme-production-pH"
  ;;set fph table:get fungi-dict "fungi-growth-pH"
  ;;set ftemp table:get fungi-dict "fungi-growth-temp"
end

to setup-ECs
  set ec-added 40
  create-ECs 40 ;;If changed, change above as well
  [
    set size 1
    set shape "substrate"
    set color 26
    setxy (random -32 + 16) one-of [ -16 16 ] ;;place the ECs at random locations along the bottom or top
    set partner nobody
    set hidden? false
    set taken false
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to color-patches
  ask patches [
    set pcolor (blue + glucose / 3) ;;changes patch color based on amount of glucose
  ]
end

to move
  ask enzymes [
    if partner = nobody [
      if ticks mod 3 = 0 [
        set turn (random 120) - 60 ;; A turn is decided every 3 ticks, and then carried out in 3 pieces over that time. Very smooth
      ]
   		right turn / 3
      ifelse (xcor > 14.5) [ ;; These 4 blocks redirect the enzyme if it is near an edge and heading towards that edge
        if (0 < heading and heading <= 90) [left 20]
        if (90 < heading and heading < 180) [right 20]
      ][
      ifelse (xcor < -14.5) [
        if (270 <= heading and heading < 360) [right 20]
        if (180 < heading and heading < 270) [left 20]
      ][
      ifelse (ycor < -14.5) [
        if (270 > heading and heading > 180) [right 20]
        if (180 >= heading and heading > 90) [left 20]
      ][
      if (ycor > 14.5) [
        if (0 <= heading and heading < 90) [right 20]
        if (270 < heading and heading < 360) [left 20]
      ]
      ]
      ]
      ]
      forward 0.12
  	]
  ]
  ask ECs [
    if partner = nobody [
      ifelse ticks < 150 [
        ;; This moves the EC's towards the center horizontal at the start of the game.
        ;; This is needed as the first EC's start on the top and bottom
        let ec-heading heading
        if ycor > 6 [set heading 180]
        if ycor < -6 [set heading 0]
        forward 0.05
        set heading ec-heading
      ]
      [
        ;;This keeps them closer to the center horizontal during the rest of the game
        let ec-heading heading
        ifelse ycor > 12 [set heading 180][
        ifelse ycor < -12 [set heading 0][
		ifelse xcor < -12 [set heading 90] [
		if xcor > 12 [set heading 270]
		]
		]
		]
        forward 0.02
        set heading ec-heading
      ]

      right -45 + random 90 ;; No turn setup. More erratic

      if (xcor > 14.5) [
        if (0 < heading and heading < 90) [left 20]
        if (90 < heading and heading < 180) [right 20]
      ]
      if (xcor < -14.5) [
        if (270 < heading and heading < 360) [right 20]
        if (180 < heading and heading < 270) [left 20]
      ]
            if (ycor < -14.5) [
        if (270 > heading and heading > 180) [right 20]
        if (180 > heading and heading > 90) [left 20]
      ]
      if (ycor > 14.5) [
        if (0 < heading and heading < 90) [right 20]
        if (270 < heading and heading < 360) [left 20]
      ]
      forward 0.06
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to break-down-lignin
  ask enzymes with [partner = nobody] [
     if lignin >= 2
    [ set lignin (lignin - 2)
      set glucose (glucose + 1) ;;(Comment) later try / adjuster-enzyme
      ]
      ]
  if (count enzymes) < 80 [
    ask fungi [
      if lignin >= 1 [
        set lignin lignin - 0.5
        set glucose glucose + 0.25
      ]
    ]
  ]

end

to eat-glucose
  ask patches [ ;;(Comment) try to simplify or remove this section
    let fungus-count count fungi-here
    if fungus-count > 0 [
        let g glucose / fungus-count
          ask fungi-here [
        set O2 (O2 + (g * 2))
          ]
       ]
    ]
  ask fungi [
        ifelse glucose >= 5 and O2 >= 30 [
      	set glucose (glucose - 5)
      	set O2 (O2 - 30)
      	set energy (energy + (5 * 0.25))
    ][
          if O2 >= glucose * 3 [
          let m (glucose / count fungi-here)
          set glucose (glucose - m)
          set O2 (O2 - (m * 3))
          set energy (energy + (m * 0.25))
          ]
      ]
    ]
end

to replenish-O2
  if (ticks mod 5 = 0) [
  ask patches [
    if O2 <= 50 [
    set O2 (O2 + 30)
    ]
  ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to reproduce
  ask fungi [
    ifelse age > 16  and random(2 * (abs(table:get fungi-dict "fungi-growth-pH" - pH))^ 2)= 0 and energy > (random 8 + 2 + (abs(table:get fungi-dict "fungi-growth-temp" - temperature) ^ 2) * 2 + (abs(2 - table:get fungi-dict "fungi-growth-pH"))^ 2) [
      ;;Affected by age, pH, energy and temp
      let En (energy / 2)
      set energy En
      let L glucose / 2
      set glucose L
      set NH4 NH4 - 0.5
      hatch-fungi 1 [
        let p max-one-of patches in-radius 3 with [pycor < 15 and pycor > -15 and pxcor < 15 and pxcor > -15] [glucose]
        face p
        set energy En + (5.5 - (2 * abs(table:get fungi-dict "fungi-growth-pH" - pH))) + (table:get fungi-dict "enzyme-production-cost" * 1.5)
        set NH4  3.5 / adjuster-fungi
        set age 0
        forward 1
        ;;set glucose L
        set partner nobody
      ]
     ]
    [produce-enzymes] ;;Enzymes can only be produced when the fungi cannot reproduce
  ]
end

to produce-enzymes
    if (sum [glucose] of patches in-radius 2) < 120 and (energy < 20 or ticks < 12) and NH4 >= (max(list 1 (abs(table:get fungi-dict "enzyme-production-pH" - pH) * (table:get fungi-dict "enzyme-production-cost"))) * table:get fungi-dict "enzyme-production-cost") and (random(((abs(table:get fungi-dict "enzyme-production-pH" - pH))^ 3) * 50) = 0  or ticks < 12) and ((random 100) + count enzymes) < 280 [
      ;;Affected by nearby glucose, energy, NH4, ticks, enzyme count and pH
      set NH4 NH4 - (max(list 1 (abs(table:get fungi-dict "enzyme-production-pH" - pH) * (table:get fungi-dict "enzyme-production-cost"))) * table:get fungi-dict "enzyme-production-cost")
      hatch-enzymes 1 [
          set size 1
          set color white
          set shape "enzyme"
          set partner nobody
          set age 0
        ]
    ]
end

to produce-ECs
  if ticks > 32 and ticks < 400 and ticks mod 6 = 0 [ ;;EC's produced every 6 ticks after the first 32
    create-ECs 2 [
      set size 1
      set shape "substrate"
      set color 26
      setxy (random 20 - 10) (random 20 - 10) ;; Placed closer to the center of the display
      set partner nobody
      set hidden? false
      set ec-added ec-added + 1
      set taken false
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to age-fungus
  ask fungi [
    set energy energy - 0.7
    set NH4 NH4 + 0.08 / (adjuster-enzymes * 1.5) ;;All fungus get NH4 at a set rate every tick
    set age (age + 1)
    set color age * 0.15 ;;lighten color as fungi age
    if energy <= 0 [set age age + 0.6]
    if color >  9.9 [
      set color 9.9]
    if energy < 0 [set energy 0]
  ]
end

to age-enzymes
  ask enzymes [
    set age (age + 1) ;;add 1 to enzymes' age
  ]
end

to check-death
  ask fungi [
    if age > 200 * (1 / ms) and partner = nobody [die];;die when age exceeds 400
  ]
  ask enzymes [
    if age > 275 * (1 / ms) and partner = nobody [die] ;;enzymes die when their age exceeds 550
  ]
end

to nutrient-status
  ;;This reports the lignin remaining to the user
  ;;I'm trying to draw attention to the graphs
  if sum [lignin] of patches < (51200) and status-1 = false
  [output-print ("Lignin at 50%")
  ;ask status 1 [set whether true
  set status-1 true]
  if sum [lignin] of patches < (76800) and status-2 = false
    [output-print ("Lignin at 75%")
     set status-2 true]
  if sum [lignin] of patches < (26600) and status-3 = false
    [output-print ("Lignin at 25%")
     set status-3 true]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to intracellular-degrade ;;Some contaminants are removed by intracellular enzymes
  ;;This happens when they approach an unpaired fungus
  ask fungi with [partner = nobody] [
    if any? ECs in-radius 0.5 with [partner = nobody] [
      set partner one-of ECs in-radius 0.5 with [partner = nobody]
      ask partner [
        set partner myself
        set taken true]
    ]
  ]
  Ask ECs with [taken = true] [
      set locked-time locked-time + 1
      if locked-time > 34 [
        set intracellularly intracellularly + 1 ;;Counter for % removed intracellularly
        ask partner [set partner nobody]
        die
      ]
    set size 1 - (locked-time / 34) ;;Shrink each tick and move slightly forward
    fd 0.007  ;;this is 1 / 15 rounded
    ]
end

to denature ;;parameters for denaturing enzymes
  let aging-amount 0.5 + 0.1 * (table:get fungi-dict "enzyme-production-pH" - pH) ^ 2 + (count enzymes / 500) ;;All enzymes age the same amount, based on pH the count of enzymes
  ask enzymes [
    set age age + aging-amount
  ]
  ;;if ticks mod 80 = 0 [ ;;Optional randomizer. Sometimes a lot of enzymes die at once, this would prevent that
    ;;ask enzymes with [partner = nobody] [
      ;;set age age + (random 20) - 10
    ;;]
end

to change-rate
    set Kc round (1) ;;(Comment) If these first two are going to be constant. Define them in setup globals once and for all
    set Kd round (60)
    set Kr round(3 * abs(32 - temperature) + 3) ;;multiply Kr by the multipliers for temperature, pH
end

to degrade-ECs
  ask enzymes [form-complex]
  ask enzymes [dissociate]
  ask ECs [react-forward]
end

to form-complex
  ifelse pause-time < 0 ;;Enzymes won't reform immediately after dissociating
  [set pause-time pause-time + 1]
  [if partner = nobody and (any? other ECs-here with [partner = nobody and not taken]) [ ;;and random Kc = 0 [
    set partner one-of (other ECs-here with [partner = nobody and not taken])
    set complexes-formed complexes-formed + 1
    let h heading
    ask partner [
      set partner myself
      set hidden? true
      move-to partner ;;These two make it so the EC matches the Enzyme position exactly.
      set heading h   ;;if they dissociate it will look like a clean separation
    ]
    set shape "complex"
    ]
  ]
end

to react-forward
  if (partner != nobody and not taken) [
    set locked-time locked-time + 1 ;;Won't react forward right after joining.
    if locked-time > 25 and random Kr = 0
      [ ;set breed products
        ;set color green
        ;set shape "substrate"
        ;set hidden? false
        ask partner [
          set partner nobody
          set shape "enzyme"
          set color white
          set age age - 15] ;;Age lowers so enzyme won't die right after leaving complex. That would be confusing for the user
        set partner nobody
        die
    ]
  ]
end

to dissociate
  if partner != nobody and random Kd = 0
      [set dissociated dissociated + 1
       set pause-time -15
       ask partner [
          set partner nobody
          set hidden? false
          set locked-time 0]
        set partner nobody
        set shape "enzyme"
        set age age - 15 ;;Age lowers so enzyme won't die right after leaving complex. That would be confusing for the user
  			]
end
@#$#@#$#@
GRAPHICS-WINDOW
390
20
835
466
-1
-1
13.242424242424242
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
1055
166
1125
199
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1130
166
1215
199
go/pause
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
25
20
381
237
Nutrients
Time
Quantity per Patch
0.0
800.0
0.0
10.0
true
true
"" ""
PENS
"lignin" 1.0 0 -13345367 true "" "plot sum [lignin] of patches / (32 * 32)"
"glucose" 1.0 0 -4699768 true "" "plot sum [glucose] of patches / (32 * 32)"

MONITOR
390
472
455
517
# Fungi
count fungi
17
1
11

MONITOR
460
472
536
517
# Enzymes
count enzymes
17
1
11

PLOT
25
242
381
466
Organisms and Contaminants
Time
Quantity
0.0
800.0
0.0
10.0
true
true
"" ""
PENS
"fungi" 1.0 0 -16777216 true "" "plot count fungi"
"enzymes" 1.0 0 -8630108 true "" "plot count enzymes"
"contaminant" 1.0 0 -955883 true "" "plot count ECs"

TEXTBOX
630
477
796
575
Key:\nenzymes = white\nfungi = black shapes\nwhiter fungi = older\ncontaiminants = orange triangles
11
0.0
1

SLIDER
1050
206
1215
239
temperature
temperature
10
50
33.5
0.5
1
celsius
HORIZONTAL

MONITOR
541
472
604
517
# ECs
count ECs
17
1
11

SLIDER
1050
246
1214
279
pH
pH
0
7
5.0
0.1
1
pH
HORIZONTAL

MONITOR
390
521
501
566
% ECs Removed
(1 - ( count ECs / ec-added )) * 100
1
1
11

BUTTON
1032
292
1132
325
Watch Enzyme
ifelse any? enzymes \n[ask one-of enzymes [watch-me]]\n[user-message (\"Currently no Enzymes\")]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1142
292
1233
325
Watch EC
ifelse any? ECs\n[ask one-of ECs [watch-me]]\n[user-message (\"Currently no Contaminants\")]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

OUTPUT
26
473
381
553
12

BUTTON
1032
327
1132
360
Export View
export-view (word \"view\" view-number \".jpg\")\nset view-number view-number + 1
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1142
327
1232
360
Export Interface
export-interface (word \"interface\" interface-number \".jpg\")\nset interface-number interface-number + 1
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
1054
116
1214
161
fungi-species
fungi-species
"Phanerodontia Chrysosporium" "Trametes Versicolor" "Pleurotus Ostreatus"
2

PLOT
842
42
1025
447
Clock
NIL
Time Remaining (Ticks)
0.0
1.0
0.0
400.0
true
true
"set my-list1 n-values 400 [0] \nset my-list2 n-values 600 [1]" "ifelse ticks > 0 and ticks <= 400 [\nset my-list1 remove-item 0 my-list1\n][\nif ticks > 400 and ticks <= 1000 [\nset my-list2 remove-item 0 my-list2\n]]\n\n"
PENS
"ECs Appearing" 1.0 1 -955883 true "" "histogram my-list1"
"Remediation Time" 1.0 1 -13791810 true "" "histogram my-list2"

@#$#@#$#@
## WHAT IS IT?

This is a model of fungal bioremediation. In this model, white rot fungus produces enzymes that serve two purposes: breaking down lignin (a complex polymer found in wood) and degrading neonicotinoids (contaminants). 

Lignin is broken down into glucose, which is then consumed by bacteria and fungi in cellular respiration. The fungi and bacteria have a symbiotic relationship. Since bacteria cannot break down lignin, they rely on fungal enzymes to break down lignin to produce glucose in the absence of other glucose. 

The fungal enzymes also degrade neonicotinoids (contaminants in the water). While in reality, the neonicotinoids would be degraded into a potentially less harmful product, in this model, the neonicotinoids are simply modeled as being eliminated. 

For further context, in a real world setting, white rot fungi growing on wood chips would function as a filter. As contaminated water flows through the fungi filter, the fungi convert neonicotinoids in the water into potentially less harmful products. 

This model includes elements of enzyme kinetics and organisms' competition for food.

## HOW IT WORKS

When the experiment is set up, fungi, bacteria, and contaminants are randomly placed in the grid. 

New fungi grow outward from old fungi, and as fungi age, they become less active and eventually die off. The fungi become whiter in color as they age. Fungi consume nutrients, glucose, and dissolved oxygen in the surrounding patches and produce enzymes that degrade lignin and neonicotinoids. 

Bacteria also consume glucose, nutrients, and dissolved oxygen in the surrounding patches. 

Nutrients and dissolved oxygen diffuse throughout the water. 

Sliders control various factors that influence reaction rates, reproduction rates, water speed, oxygen levels, the starting amount of ECs, and the amount of energy gained from glucose consumption.

The first switch allows for the user to turn on nutrient replenishing, and the second switch allows for the user to turn on background coloring so that the background changes colors based on the levels of glucose present.

## HOW TO USE IT

Press "setup" to set up the model. Press "go" to run. The simulation will automatically stop when all of the fungi has died off, but if you'd like to stop the simulation before then, simply press "go" again. Press "setup" to reset the simulation.

The initial populations of fungi and bacteria can be modified in the two green boxes. The slider called "amount-lignin" controls the initial amount of lignin in each patch. 

The interface contains three plots. One monitors lignin, glucose, nutrients, O2, and CO2. Note that the plot for CO2 is set to not show since CO2 is much greater than the other elements, and adding the CO2 plot would condense the other plots. The second monitors fungi, bacteria, and enzyme populations. The last monitors the percent of contaminants removed. 

Several monitors track the populations of contaminants, enzymes, bacteria, fungi, and percent of ECs that escaped. The monitor and plot for percent of ECs escaped isn't particularly useful for water flow = 0. Users should just look at the number of contaminants remaining plot to monitor the progress of contaminant removal.

The green sliders control various aspects of the model.

## THINGS TO NOTICE

Notice the time it takes for a certain amount of contaminants to be removed and the percent of contaminants that are removed before all fungi die off. 

Notice the colors of the patches. What does that indicate about the diffusing behavior of the nutrients and dissolved oxygen?

Notice the behavios of the fungi, bacteria, and enzyme populations. Are the populations always increasing? Always decreasing? Do they rapidly increase and then decrease?

Notice the shapes of the graphs for glucose, nutrients, lignin, and dissolved O2. 

Notice the behavior at low and high water flow.

## THINGS TO TRY

Task 1: There are three different sliders related to the reaction rate: temperature, pH, and uv-light. For each condition, there's an optimal condition where the enzymes function best. Try to figure out what the optimal conditions are. Consider isolating each condition (don't change multiple at once as that will make it difficult to determine how each change affected the experiment). Consider a model where the water speed = 0 and a model where the water speed = 1.

Task 2: There's a cost associated with each of these conditions (e.g. keeping the solution at a constant temperature). Assuming all of the conditions cost the same amount to make optimal, figure out which two conditions at their optimal have the greatest positive impact on the rate. Which condition can be sub-optimal while having the smallest negative impact on the rate? Consider a model where the water speed = 0 and a model where the water speed = 1.

Task 3: Now, figure out which condition should be optimal if you can only have one optimal condition and the other two must be sub-optimal. Can you rank the conditions from least to greatest impact on rate?

Task 4: Consider changing the replenish O2 time. What happens when replenish O2 time is smaller? How about when it is larger?

Task 5: Considering all four sliders (temperature, pH, UV light, and replenish O2 time), try to modify the conditions so that you produce four different results in a model where water speed = 0: no removal of ECs, low removal of ECs, moderate removal of ECs, and high removal of ECs. For a model where water speed = 1, try to create conditions where there is low, moderate, and high numbers of ECs escaping.

Task 6: Determine the optimal water speed (1-5) for fungi survival and ensuring that few contaminants escape the filter. Why do you think this speed is the best? How does the idea of water speed apply in the real world? What engineering mechanisms could be added to ensure the optimal water speed?

Task 7: Using the different sliders and input boxes, find conditions that cause less fungi growth. Why do you think these conditions hurt fungi growth?

Final Task Part 1: Using any or all of the sliders, try to create a scenario where no species is too dominant or too weak, the reaction proceeds relatively quickly, and few contaminants escape.

Final Task Part 2: How many of sliders can you have at sub-optimal conditions while allowing minimal numbers of contaminants to escape. How far from the optimal condition can the sliders be while still maintaining low numbers of contaminants escaping.

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

The wolf-sheep predation model similarly shows competition between biological organisms. It also shows how the various agents/turtles rely on the patches for food. 

The enzyme kinetics model illustrates the relationship between Kc, Kd, and Kr. It also illustrates the lock and key model for how enzymes function as a catalyst in a reaction.

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

button-mushroom
true
0
Polygon -7500403 true true 148 226 166 225 173 222 176 211 177 193 176 175 166 141 149 135 149 226
Polygon -7500403 true true 151 226 133 225 126 222 123 211 122 193 123 175 133 141 150 135 150 226
Polygon -16777216 true false 151 147 133 151 112 155 90 157 73 152 77 128 100 111 124 108 138 109 151 110 152 148
Polygon -16777216 true false 151 148 169 152 190 156 212 158 229 153 225 129 202 112 178 109 164 110 151 111 150 149

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

complex
true
0
Polygon -1 true false 45 60 255 60 255 240 150 150 45 240 45 240 45 60
Polygon -955883 true false 45 240 45 240 255 240 150 150

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

enzyme
true
0
Polygon -1 true false 45 240 45 60 255 60 255 240 150 150

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

fungi-tendrils
true
0
Polygon -7500403 true true 113 244 105 212 107 183 106 124 98 96 110 71 123 76 113 95 119 166 136 129 154 73 178 68 166 99 158 123 152 142 166 173 171 193 186 226 178 233 169 221 155 192 144 164 132 168 119 181 142 228 157 269 147 272 126 211 112 200 121 263 114 266 112 231
Polygon -7500403 true true 149 137 175 149 180 120 190 84 199 84 190 103 189 119 183 151 187 173 202 211 192 214 174 172 168 157 151 157

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

petals
false
0
Circle -7500403 true true 117 12 66
Circle -7500403 true true 116 221 67
Circle -7500403 true true 41 41 67
Circle -7500403 true true 11 116 67
Circle -7500403 true true 41 191 67
Circle -7500403 true true 191 191 67
Circle -7500403 true true 221 116 67
Circle -7500403 true true 191 41 67
Circle -7500403 true true 60 60 180

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

substrate
true
0
Polygon -7500403 true true 45 240 150 150 255 240 75 240 60 240

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turkey-tail
true
0
Polygon -1 true false 102 180 72 185 63 175 59 155 59 128 69 109 89 91 112 75 146 73 166 73 207 84 230 106 237 132 233 157 231 172 213 185 196 190 177 187 161 178 158 179 100 179
Polygon -7500403 true true 94 187 80 174 78 157 79 130 92 110 129 95 154 93 192 96 214 119 215 138 209 160 200 176 191 190 146 196 105 195 89 185
Polygon -7500403 true true 103 171 89 161 75 151 60 149 51 164 48 190 56 223 85 237 114 233 132 227 144 215 137 188
Polygon -16777216 true false 99 192 96 181 99 149 112 131 149 116 170 126 188 146 189 175 174 194 132 198
Polygon -7500403 true true 197 177 224 165 235 171 241 184 233 213 190 239 169 237 154 232 140 213 151 186
Polygon -7500403 true true 152 187 132 193 139 207 160 202

turkey-tail-2
true
0
Circle -7500403 true true 105 105 90
Circle -7500403 true true 75 75 90
Circle -7500403 true true 45 105 90
Circle -7500403 true true 75 135 90
Circle -7500403 true true 135 75 90
Circle -7500403 true true 135 135 90
Circle -7500403 true true 165 105 90
Circle -16777216 true false 103 103 92

turkey-tail-3
true
0
Circle -7500403 true true 60 45 180
Circle -6459832 true false 75 75 150
Circle -16777216 true false 90 105 120
Circle -7500403 true true 105 135 90

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment_1" repetitions="35" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>(1 - ( count ECs / ec-added )) * 100</metric>
    <steppedValueSet variable="pH" first="3.6" step="0.1" last="4"/>
    <enumeratedValueSet variable="fungi-species">
      <value value="&quot;Phanerodontia Chrysosporium&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="temperature">
      <value value="30"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_PC_temp" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>(1 - ( count ECs / ec-added )) * 100</metric>
    <enumeratedValueSet variable="pH">
      <value value="3.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fungi-species">
      <value value="&quot;Phanerodontia Chrysosporium&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="temperature">
      <value value="29"/>
      <value value="29.5"/>
      <value value="30"/>
      <value value="30.5"/>
      <value value="31"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="PC_Interaction" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>(1 - ( count ECs / ec-added )) * 100</metric>
    <enumeratedValueSet variable="pH">
      <value value="3.6"/>
      <value value="3.7"/>
      <value value="3.8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fungi-species">
      <value value="&quot;Phanerodontia Chrysosporium&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="temperature">
      <value value="29.5"/>
      <value value="30"/>
      <value value="30.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="TV_Interaction" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>(1 - ( count ECs / ec-added )) * 100</metric>
    <enumeratedValueSet variable="pH">
      <value value="3.4"/>
      <value value="3.5"/>
      <value value="3.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fungi-species">
      <value value="&quot;Trametes Versicolor&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="temperature">
      <value value="23.5"/>
      <value value="24"/>
      <value value="24.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="PO_Interaction" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>(1 - ( count ECs / ec-added )) * 100</metric>
    <enumeratedValueSet variable="pH">
      <value value="4.7"/>
      <value value="4.8"/>
      <value value="4.9"/>
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fungi-species">
      <value value="&quot;Pleurotus Ostreatus&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="temperature">
      <value value="33.5"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
