breed [ antibiotics antibiotic ]
breed [ bacterias bacteria ]

bacterias-own [
  variation
  age
  my-target-patch
]

patches-own [
  my-bacteria
]

globals [
  antibiotic-flow-rate
  fever-factor?
  #-minutes
  #-hours
  #-days
  bacteria-counts
  given-doses
  wiped-out?
  tick-entered ; each antibiotics entering time
]

;;;;;;;;; initial setup procedures ;;;;;;;;;

to setup
  clear-all

  set-default-shape bacterias "bacteria"

  set fever-factor? false
  set antibiotic-flow-rate 0.8
  set #-hours 0
  set #-days 0
  set #-minutes 0
  set given-doses 0
  set wiped-out? false

  ask patches [
    set pcolor white
    set my-bacteria nobody
  ]

  setup-initial-bacteria

  reset-ticks
end

to setup-initial-bacteria
  let which-variations (list 3 4 5 6) ; four types of bacteria
  let #init-of-each-variation (list init#-3pores init#-4pores init#-5pores init#-6pores)
  (foreach #init-of-each-variation which-variations [ [n v] -> make-bacteria n v ])

  update-bacteria-counts
end

to make-bacteria [ number-of-bacteria which-variation ]
  create-bacterias number-of-bacteria [
    let this-bacteria self
    set variation which-variation
    set shape (word "bacteria-holes-" variation)
    set my-target-patch one-of patches with [ my-bacteria = nobody ]
    move-to my-target-patch
    ask my-target-patch [ set my-bacteria this-bacteria ]
    set age (reproduce-every * 60) ; initial bacteria are aged enough to be reproductive
    set size 1
    set color variation-color variation
  ]
end

;;;;;;;;;;;;;;;;;;;;;; runtime procedures ;;;;;;;;;;;;;;;;;;;;;;

to go
  reproduce-bacteria-into-available-space
  let count-of-bacteria count bacterias
  ask bacterias [
    move-bacteria  ; do age ++
    natural-death-bacteria
  ]
  move-antibiotics
  antibiotics-remove ; half of antibiotic molecules are removed at every half-life
  dose-admin
  update-bacteria-counts
  tick
  update-time
end

to dose-admin
  if not any? bacterias with [ not wiped-out? ] [
    set wiped-out? true
  ]
  auto-dose
end

to update-bacteria-counts
  ; count each type of bacteria. used for output plots.
  let cnt [ 0 0 0 0 ]
  ask bacterias [
    let i variation - 3
    set cnt replace-item i cnt (item i cnt + 1)
  ]
  set bacteria-counts cnt
end

to update-time
  ; one tick = 1 minute in simulated time
  set #-minutes ticks
  set #-hours floor (ticks / 60)
  set #-days (floor (#-hours / 2.4) / 10)
end

;;;;;;;;;;;;;;;;;;; antibiotic procedures ;;;;;;;;;;;;;;;;;;;

to auto-dose
  if give-dose? [
    dose-enter dosage
    set given-doses given-doses + 1
    set tick-entered ticks
  ]
end

to-report give-dose?
  report
    ticks mod (dose-every * 60) = 0 and
    not wiped-out? and (
      (auto-dose? = "yes, skip no doses")
    )
end

to dose-enter [ amount ]
  create-antibiotics amount[
    set shape "molecule"
    set size 1
    set color red
    setxy random-xcor random-ycor ; an antibiotic molecule is assigned to a random position.
  ]
end

to move-antibiotics
  ask antibiotics [
    forward antibiotic-flow-rate
    check-antibiotic-interaction

    set heading random-float 360 ; freely change its direction as it moves around the space.
  ]
end

to check-antibiotic-interaction
  let this-molecule self
  if any? bacterias-here [
    ask one-of bacterias-here [
      if kill-this-bacteria? [
        ask this-molecule [ die ]
        bacteria-death
      ]
    ]
    die
  ]
end

to antibiotics-remove ; every half-life
  if (ticks - tick-entered) mod (half-life * 60) = 0 [
    let count-of-antibiotics count antibiotics
    ask n-of (0.5 * count-of-antibiotics) antibiotics [ die ]
  ]
end

;;;;;;;;;;;;;;;;;;; bacteria  procedures ;;;;;;;;;;;;;;;;;;;;

to bacteria-death
  let this-bacteria self
  if is-patch-set? my-target-patch [
    ask my-target-patch [
      set my-bacteria nobody
    ]
  ]
  die
end

to natural-death-bacteria
  if random-float 1 < bacteria-death-prob [
    bacteria-death
  ]
end

to move-bacteria
  let my-distance-to-my-patch-center distance my-target-patch
  let speed-scalar 0.15 ; scales the speed of motion of the bacteria
  if my-distance-to-my-patch-center > 0.2 [
    set heading towards my-target-patch
    forward 0.1 * speed-scalar
    set age 0
  ]
  set age age + 1
end

to reproduce-bacteria-into-available-space
  let count-of-bacteria count bacterias
  let all-bacteria-that-can-reproduce bacterias with [ can-reproduce? count-of-bacteria ]
  let this-fever-factor 1

  if reproduce? [
    ask all-bacteria-that-can-reproduce [
      let available-patches neighbors with [ my-bacteria = nobody ]
      let this-bacteria self
      if any? available-patches [
        hatch 1 [
          set my-target-patch one-of available-patches
          ask my-target-patch [ set my-bacteria this-bacteria ]

          let new_variation variation
          if random-float 1 < mutation_p [ ; mutate offspring variation: +1 or -1
            ifelse random-float 1 < 0.5[
              set new_variation variation - 1 ; mutation decreasing susceptibility
            ]
            [
              set new_variation variation + 1 ; mutation increasing susceptibility
            ]
            ; if new number of holes excedes the min or max number of holes,
            ; keep its original value (i.e., no mutation occurs).
            if new_variation >= 3 and new_variation <= 6 [
              set variation new_variation
              set color variation-color variation
              set shape (word "bacteria-holes-" variation)
            ]
          ]
        ]
      ]
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;; reporters ;;;;;;;;;;;;;;;;;;;;;;;;

to-report kill-this-bacteria?
  ; variations go from 3 to 6, for the number of holes in the membrane,
  ; so the odds of getting into the bacteria are 30%, 40%, 50% and 60%
  report random 100 < (variation * 10)
end

to-report can-reproduce? [ total-bacteria-count ]
  let result false
  let this-fever-factor 1
  ; total-bacteria-count is used to simulate a slower growth rate for bacteria as their numbers increase in the body
  ; due to an immune system response of raising the temperature of the body as more an more bacteria above a certain
  ; threshold are detected in the body
  if total-bacteria-count > 50 and fever-factor? [
    set this-fever-factor (1 + 2 * ((total-bacteria-count - 50) / 75))
  ]
  if age >= (reproduce-every * 60 * this-fever-factor) [
    if not any? other bacterias in-radius 0.75 and my-target-patch = patch-here [
      set result true
    ]
  ]
  report result
end

;;;;;;;;;;;;;;;; visualization procedures ;;;;;;;;;;;;;;;;

to-report variation-color [ which-variation ]
  report item (which-variation - 3) [ violet green brown red orange ]
end

;;;;;;;;;;;;;;;;; plotting procedures ;;;;;;;;;;;;;;;;;;;;;

to make-variation-pens [ plot-pen-mode ]
  foreach [ 3 4 5 6 ] [ v ->
    create-temporary-plot-pen (word v)
    set-plot-pen-mode plot-pen-mode ; bar mode
    set-plot-pen-color variation-color v
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
502
10
906
415
-1
-1
12.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

SLIDER
22
36
194
69
init#-3pores
init#-3pores
0
20
4.0
1
1
NIL
HORIZONTAL

SLIDER
22
87
194
120
init#-4pores
init#-4pores
0
20
4.0
1
1
NIL
HORIZONTAL

SLIDER
21
137
193
170
init#-5pores
init#-5pores
0
20
4.0
1
1
NIL
HORIZONTAL

SLIDER
20
185
192
218
init#-6pores
init#-6pores
0
20
4.0
1
1
NIL
HORIZONTAL

SLIDER
137
321
323
354
reproduce-every
reproduce-every
0
10
5.0
0.5
1
hrs
HORIZONTAL

SWITCH
18
321
128
354
reproduce?
reproduce?
0
1
-1000

SLIDER
20
230
192
263
mutation_p
mutation_p
0
1
0.12
0.01
1
NIL
HORIZONTAL

SLIDER
224
138
396
171
dose-every
dose-every
0
24
12.0
1
1
hrs
HORIZONTAL

CHOOSER
227
34
398
79
auto-dose?
auto-dose?
"no, skip all dosing" "yes, skip no doses"
1

SLIDER
225
92
397
125
dosage
dosage
0
300
100.0
10
1
NIL
HORIZONTAL

BUTTON
417
15
483
48
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
416
59
483
92
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
503
435
593
480
NIL
given-doses
17
1
11

PLOT
10
368
252
541
Bacterial strain
# pores
# of bacteria
3.0
7.0
0.0
50.0
true
false
"make-variation-pens 1 ; use bar mode " ";; the `histogram` primitive can't make a multi-colored histogram,\n;; so instead we plot each bar individually\nforeach [ 3 4 5 6 ] [ v ->\n  set-current-plot-pen (word v)\n  plot-pen-reset\n  plotxy v item (v - 3) bacteria-counts\n]\nif (plot-y-max - floor plot-y-max) != 0 [\n  set-plot-y-range 0 floor plot-y-max + 1\n]"
PENS

PLOT
258
367
496
541
Bacteria population size
time (min.)
# of bacteria
0.0
100.0
0.0
16.0
true
false
"make-variation-pens 0 ; use line mode" "foreach [ 3 4 5 6 ] [ v ->\n  set-current-plot-pen (word v)\n  plotxy ticks item (v - 3) bacteria-counts\n]"
PENS

SLIDER
224
184
396
217
half-life
half-life
0
10
5.0
0.5
1
hrs
HORIZONTAL

TEXTBOX
25
13
175
31
Bacteria traits
13
0.0
1

TEXTBOX
230
10
380
28
Antibiotics traits
13
0.0
1

SLIDER
19
276
208
309
bacteria-death-prob
bacteria-death-prob
0
0.004
0.0018
0.00005
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

NOTE: This is a transformed model of the Bacterial Infection model in the NetLogo Models Library. Additional model information can be found in the Info tab in the original model. 
The model shows bacterial growth and antibiotic resistance through membrane permeability changes. Bacterial mutation rate is very high compared to that of human cells. The administration of antibiotics causes a selection pressure on bacteria and induces the evolution of antibiotic resistance. The model explains how bacteria evolve antibiotic resistance through the mutations which change membrane permeability. 

## HOW IT WORKS

A patient is infected with four different strains of bacteria, each with different membrane permeability level. The bacterial mutation changes membrane pereability, which is represented by the number of holes in the bacterial cell membrane. Also, the number of holes in the bacterial cell membrane indicates susceptibility to antibiotics. A random mutation can add one hole or remove one hole. Mutations to the number of holes exceeding  the range between three and six do not occur. 

Bacterial natural death is considered in the model.

A patient can take antibiotics repeatedly. Antibiotic molecules enter the patient's bloodstream and kill bacteria as they travel through the patient's body. The space is the patient's bloodstream. Antibiotics are given by placing moleucles at random positions in the space. The molecules move around the bloodstream until they decompose and exit the space. The half of exising antibiotic molecules decomposes every half-life.


## HOW TO USE IT

MUTATION_P: During the asexual reproduction, mutations can occur which change the number of holes in the bacterial cell membrane. MUTATION_P is a probability of a mutation that changes membrane permeability per reproduction.

BACTERIA-DEATH-PROB: Bacterial natural death occurs with a given probability, BACTERIA-DEATH-PROB at every time point. 

HALF-LIFE: When antibiotics enter the patient's bloodstream, the half of the existing antibiotic molecules decompose and are removed from the space at every HALF-LIFE.

## THINGS TO TRY

At first, try bacteria reproduction without mutations (set REPRODUCE? to "on" and set MUTATION_P to 0). Then, increase MUTATION_P and allow bacteria to change their membrane permeability and observe changes in bacterial growth and the distribution of different strains (permeability). 

Fix all bacteria traits to represent certain bacteria. Imagine developing new antibiotic treatment. Antibiotics have three parameters: dosage, frequency, and half-life. Change these three parameters to find successful combinations, i.e., successful treatments killing all bacteria.

## EXTENDING THE MODEL

Considering that bacteria can move with flagella, the model could be extended to allow bacteria to move in the space. 

## RELATED MODELS

Bacterial Infection model in the NetLogo Models Library.

## CREDITS AND REFERENCES

The original model is authored by Uri Wilensky (Copyright 2016 Uri Wilensky).

References:
Novak, M. and Quigley, D and Wilensky, U. (2016). NetLogo Bacterial Infection model. http://ccl.northwestern.edu/netlogo/models/BacterialInfection. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bacteria
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30

bacteria-0
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30

bacteria-1
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 90 270 105 285 195 285 210 270 216 254 84 254

bacteria-2
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 90 270 105 285 195 285 210 270 225 225 75 225

bacteria-2-membrane
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Rectangle -7500403 true true 75 135 90 180
Rectangle -7500403 false true 210 135 225 180
Rectangle -7500403 true true 210 135 225 180

bacteria-3
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 105 285 90 270 75 225 75 195 225 195 225 225 210 270 195 285

bacteria-4
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 105 285 90 270 75 225 75 165 225 165 225 225 210 270 195 285

bacteria-5
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 105 285 90 270 75 225 75 135 225 135 225 225 210 270 195 285

bacteria-6
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 105 285 90 270 75 225 75 105 225 105 225 225 210 270 195 285

bacteria-7
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 105 285 90 270 75 225 75 75 225 75 225 225 210 270 195 285

bacteria-8
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 105 286 90 271 75 226 75 61 83 44 120 45 182 46 219 46 225 61 225 226 210 271 195 286

bacteria-9
true
0
Polygon -7500403 false true 60 60 60 210 75 270 105 300 150 300 195 300 225 270 240 225 240 60 225 30 180 0 150 0 120 0 90 15
Polygon -7500403 false true 75 60 75 225 90 270 105 285 150 285 195 285 210 270 225 225 225 60 210 30 180 15 150 15 120 15 90 30
Polygon -7500403 true true 105 285 90 270 75 225 75 60 90 30 120 15 180 15 210 30 225 60 225 225 210 270 195 285

bacteria-holes-1
true
0
Line -7500403 true 240 75 225 45
Line -7500403 true 240 75 240 120
Line -7500403 true 240 180 240 225
Line -7500403 true 225 255 195 285
Line -7500403 true 225 45 195 15
Line -7500403 true 75 45 105 15
Line -7500403 true 60 75 75 45
Line -7500403 true 60 75 60 180
Line -7500403 true 60 180 60 225
Line -7500403 true 60 225 75 255
Line -7500403 true 105 15 195 15
Line -7500403 true 240 225 225 255
Line -7500403 true 75 255 105 285
Line -7500403 true 105 285 195 285

bacteria-holes-2
true
0
Line -7500403 true 240 75 225 45
Line -7500403 true 240 75 240 120
Line -7500403 true 240 180 240 225
Line -7500403 true 225 255 195 285
Line -7500403 true 225 45 195 15
Line -7500403 true 75 45 105 15
Line -7500403 true 60 75 75 45
Line -7500403 true 60 75 60 120
Line -7500403 true 60 180 60 225
Line -7500403 true 60 225 75 255
Line -7500403 true 105 15 195 15
Line -7500403 true 240 225 225 255
Line -7500403 true 75 255 105 285
Line -7500403 true 105 285 195 285

bacteria-holes-3
true
0
Line -7500403 true 240 75 225 45
Line -7500403 true 240 75 240 120
Line -7500403 true 240 180 240 225
Line -7500403 true 225 255 195 285
Line -7500403 true 225 45 195 15
Line -7500403 true 195 15 180 15
Line -7500403 true 105 15 120 15
Line -7500403 true 75 45 105 15
Line -7500403 true 60 75 75 45
Line -7500403 true 60 75 60 120
Line -7500403 true 60 180 60 225
Line -7500403 true 60 225 75 255
Line -7500403 true 105 285 195 285
Line -7500403 true 240 225 225 255
Line -7500403 true 75 255 105 285

bacteria-holes-4
true
0
Line -7500403 true 240 75 225 45
Line -7500403 true 240 75 240 120
Line -7500403 true 195 285 180 285
Line -7500403 true 240 180 240 225
Line -7500403 true 225 255 195 285
Line -7500403 true 225 45 195 15
Line -7500403 true 195 15 180 15
Line -7500403 true 105 15 120 15
Line -7500403 true 75 45 105 15
Line -7500403 true 60 75 75 45
Line -7500403 true 60 75 60 120
Line -7500403 true 60 180 60 225
Line -7500403 true 60 225 75 255
Line -7500403 true 105 285 120 285
Line -7500403 true 240 225 225 255
Line -7500403 true 75 255 105 285

bacteria-holes-5
true
0
Line -7500403 true 240 75 225 45
Line -7500403 true 240 75 240 120
Line -7500403 true 195 285 180 285
Line -7500403 true 240 180 240 225
Line -7500403 true 225 255 195 285
Line -7500403 true 60 75 75 45
Line -7500403 true 60 75 60 120
Line -7500403 true 60 180 60 225
Line -7500403 true 60 225 75 255
Line -7500403 true 105 285 120 285
Line -7500403 true 240 225 225 255
Line -7500403 true 75 255 105 285
Line -7500403 true 120 15 180 15

bacteria-holes-6
true
0
Line -7500403 true 240 75 225 45
Line -7500403 true 240 75 240 120
Line -7500403 true 240 180 240 225
Line -7500403 true 60 75 75 45
Line -7500403 true 60 75 60 120
Line -7500403 true 60 180 60 225
Line -7500403 true 60 225 75 255
Line -7500403 true 240 225 225 255
Line -7500403 true 120 285 180 285
Line -7500403 true 120 15 180 15

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

molecule
true
0
Circle -2674135 true false 99 129 42
Circle -13345367 true false 129 159 42
Circle -16777216 true false 129 129 42
Circle -13345367 true false 114 99 42
Circle -2674135 true false 144 84 42

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
