;; EECS 472 FINAL PROJECT 
;; Agent Model for examining theories of International Migration
;; Balasubramanian 'Subu' Kandaswamy


;;::::::::::::::::::::::::::::::::::::::::::::::GLOBAL DECLARATIONS::::::::::::::::::::::::::::::::::::::::::::::::::::
;;Patches
patches-own [ country ]
;;Turtles
turtles-own [ origin citizen prof employed life]
;;Global Variables
globals [
  ;;countries
  countries
  ;;professions
  profes
  ;;patches for countries
  patches-1 patches-2 patches-3 patches-4
  ;;wages  
  wage-1A wage-1B wage-1C wage-1D
  wage-2A wage-2B wage-2C wage-2D
  wage-3A wage-3B wage-3C wage-3D
  wage-4A wage-4B wage-4C wage-4D
  ;; gross national happiness index
  gnh-1 gnh-2 gnh-3 gnh-4
]

;;::::::::::::::::::::::::::::::::::::::::::::::SETUP AND GO::::::::::::::::::::::::::::::::::::::::::::::::::::
to setup
  set countries [ 1 2 3 4 ]
  set profes [ "A" "B" "C" "D" ]
  setup-patches
  setup-labels
  setup-globals
  setup-turtles
  calc-wage
  calc-gnh
  set-color-by-prof
  do-plot
end

;;wage-based-immigration
to go-wage
  immigrate-by-wage
  calc-wage
  set-color-by-prof
  tick
  if birth-rate-on?
  [ do-birth
    do-death
  ]
  do-plot
end

;;GNH based immigration
to go-gnh
  immigrate-by-gnh
  calc-gnh
  calc-wage
  set-color-by-prof
  tick
  if birth-rate-on?
  [ do-birth
    do-death
  ]
  do-plot
end

;; Employment based immigration
to go-employment
  immigrate-by-employment
  calc-wage
  set-color-by-prof
  tick
  if birth-rate-on?
  [ do-birth
    do-death
  ]
  do-plot
end

to setup-patches
  clear-all
  set-default-shape turtles "person"
  set patches-1 patches with [pycor > 0 and pxcor < 0]
  ask patches-1
  [ set pcolor green 
    set country 1
    ]
  set patches-2 patches with [pycor > 0 and pxcor > 0]
  ask patches-2
  [ set pcolor blue 
    set country 2]
  set patches-3 patches with [pycor < 0 and pxcor < 0]
  ask patches-3
  [ set pcolor orange 
    set country 3]
  set patches-4 patches with [pycor < 0 and pxcor > 0]
  ask patches-4
  [ set pcolor magenta 
    set country 4]
end

to setup-labels
  ask patch (min-pxcor / 2) (max-pycor / 2 ) [
    set plabel "Country 1"
  ]
  ask patch (max-pxcor / 2) (max-pycor / 2 ) [
    set plabel "Country 2"
  ]
  ask patch (min-pxcor / 2) (min-pycor / 2 ) [
    set plabel "Country 3"
  ]
  ask patch (max-pxcor / 2) (min-pycor / 2 ) [
    set plabel "Country 4"
  ]
end

to setup-globals
  set gnh-1 50
  set gnh-2 50
  set gnh-3 50
  set gnh-4 50
end

to setup-turtles
  crt population-1 [
    set citizen 1
    set origin 1
    set prof one-of [ "A" "B" "C" "D" ]
    set employed false
    set life life-expectancy-1
    move-to-empty-one-of patches-1
    ]
  crt population-2 [
    set citizen 2
    set origin 2
    set prof one-of [ "A" "B" "C" "D" ]
    set employed false
    set life life-expectancy-2
    move-to-empty-one-of patches-2
  ]
  crt population-3 [
    set citizen 3
    set origin 3
    set prof one-of [ "A" "B" "C" "D" ]
    set employed false
    set life life-expectancy-3
    move-to-empty-one-of patches-3
  ]
  crt population-4 [
    set citizen 4
    set origin 4
    set prof one-of [ "A" "B" "C" "D" ]
    set employed false
    set life life-expectancy-4
    move-to-empty-one-of patches-4
  ]
end

to set-color-by-prof
  ask turtles [
    ifelse employed = false
    [ set color red ]
    [ if prof = "A" [ set color green ]
      if prof = "B" [ set color blue ]
      if prof = "C" [ set color yellow ]
      if prof = "D" [ set color grey]
    ]
  ]
end

;;::::::::::::::::::::::::::::::::::::::::::::::IMMIGRATIUON PROCEDURES::::::::::::::::::::::::::::::::::::::::::::::::::::

to immigrate-by-wage
    ask turtles [
      let max-wage-desc wage-discrepancy
      let max-wage-country 0
      let max-wage-country-list (list 0)
      let curwage (get-wage citizen prof)
      ;; Find country which has better wage
      foreach [ 1 2 3 4 ]
      [
        if citizen != ? and (get-wage ? prof) > curwage [
          let twage-desc calc-wage-desc prof citizen ?
          if max-wage-desc <= twage-desc
            [
              set max-wage-desc twage-desc
              set max-wage-country-list lput ? max-wage-country-list
            ] 
        ]
      ]
      set max-wage-country-list but-first max-wage-country-list
      if length max-wage-country-list > 0 [
        set max-wage-country one-of max-wage-country-list
      ]
      ;; move to that country
      if max-wage-country != 0
      [
        move-to-empty-one-of patches with [ country = max-wage-country ]
        set citizen max-wage-country
        set employed false
        if wage-awareness? = true
        [ ifelse wage-empl-by-population? 
          [set-wage citizen prof (calc-wage-for citizen prof) ]
          [set-wage citizen prof (calc-wage-for-common citizen prof) ]       
        ]
      ]
    ]
end

to immigrate-by-gnh
    ask turtles [
      let max-gnh gnh-discrepancy
      let max-gnh-country 0
      let max-gnh-country-list (list 0)
      let curgnh (get-gnh citizen)
      ;; Find country which has better GNH
      foreach [ 1 2 3 4 ]
      [
        let tgnh (get-gnh ?)
        if citizen != ? and tgnh > curgnh [
          if max-gnh <= (tgnh - curgnh)
            [
              set max-gnh (tgnh - curgnh)
              set max-gnh-country-list lput ? max-gnh-country-list
            ] 
        ]
      ]
      set max-gnh-country-list but-first max-gnh-country-list
      if length max-gnh-country-list > 0 [
        set max-gnh-country one-of max-gnh-country-list
      ]
      ;;Move to that Country
      if max-gnh-country != 0
      [
        move-to-empty-one-of patches with [ country = max-gnh-country ]
        set citizen max-gnh-country
        set employed false
        if wage-awareness? = true
        [ ifelse wage-empl-by-population? 
          [set-wage citizen prof (calc-wage-for citizen prof) ]
          [set-wage citizen prof (calc-wage-for-common citizen prof) ]       
        ]
        if gnh-awareness? = true
        [ ifelse wage-empl-by-population? 
          [set-wage citizen prof (calc-wage-for citizen prof) ]
          [set-wage citizen prof (calc-wage-for-common citizen prof) ] 
          set-gnh citizen (calc-gnh-for citizen) ]
      ]
    ]
end


to immigrate-by-employment
    ask turtles with [employed = false ] [
      let max-job-avail (calc-job-avail prof citizen)
      let max-job-country 0
      let max-job-country-list (list 0)
      ;; Find country which has better Employment opportunities
      foreach [ 1 2 3 4 ]
      [
        let tjob-avail calc-job-avail prof ?
        if citizen != ? [
           if max-job-avail < tjob-avail
           [
             set max-job-avail tjob-avail
             set max-job-country-list lput ? max-job-country-list
           ] 
        ]
      ]
      set max-job-country-list but-first max-job-country-list
      if length max-job-country-list > 0 [
        set max-job-country one-of max-job-country-list
      ]
      ;; Move to that country
      if max-job-country != 0
      [
        move-to-empty-one-of patches with [ country = max-job-country ]
        set citizen max-job-country
        set employed false
        if wage-awareness? = true
        [ ifelse wage-empl-by-population? 
          [set-wage citizen prof (calc-wage-for citizen prof) ]
          [set-wage citizen prof (calc-wage-for-common citizen prof) ]       
        ]      
      ]
    ]
end

;;::::::::::::::::::::::::::::::::::::::::::::::CALCULATION UTILS::::::::::::::::::::::::::::::::::::::::::::::::::::
to calc-wage
  ifelse wage-empl-by-population? [
    foreach [ 1 2 3 4 ]
    [
      set-wage ? "A" (calc-wage-for ? "A")
      set-wage ? "B" (calc-wage-for ? "B")
      set-wage ? "C" (calc-wage-for ? "C")
      set-wage ? "D" (calc-wage-for ? "D")
    ]
  ]
  [
    foreach [ 1 2 3 4 ]
    [
      set-wage ? "A" (calc-wage-for-common ? "A")
      set-wage ? "B" (calc-wage-for-common ? "B")
      set-wage ? "C" (calc-wage-for-common ? "C")
      set-wage ? "D" (calc-wage-for-common ? "D")
    ]
  ]
end

to calc-gnh
      foreach [ 1 2 3 4 ]
    [ set-gnh ? (calc-gnh-for ?) ]
end

to-report calc-gnh-for [ state ]
  let le-desc 0
  let mp-desc 0
  let ai-desc 0
  let ue-desc 0
  set le-desc ((get-life-expectancy state) / (get-max-life-expectancy)) * 10
  set mp-desc 10 - (((get-population state) / (get-max-population)) * 10)
  set ai-desc ((get-average-income state) / (get-max-avg-income)) * 10
  set ue-desc 10 - (((get-unempl state) / (get-max-unempl)) * 10)
  report (le-desc + mp-desc + ai-desc + ue-desc)
end

to   do-death
  ask turtles [
    set life life - 1
    if life <= 0
    [ die ]
  ]
end

to-report calc-job-avail [profs state]
  let tot-profs count turtles with [ citizen = state and prof = profs ]
  let ideal-profs (get-ideal profs)
  ifelse ideal-profs > tot-profs
  [report  ideal-profs - tot-profs]
  [report 0]
end

to-report calc-wage-desc [ profs state nstate ]
let result 0
if (get-wage state profs) != 0 [
  set result (((get-wage nstate profs) - (get-wage state profs)) / (get-wage state profs)) * 100
]
report result
end

to-report calc-wage-for [ state profs ]
  let pop count turtles with [ citizen = state ]
  let prof-pop count turtles with [ citizen = state and prof = profs ]
  let ideal-prof floor (pop * (get-ideal profs)) / 100
  let totinv ideal-prof * 50
  ask turtles with [ citizen = state and prof = profs ]
    [ set employed false ]
  ifelse ideal-prof < prof-pop
  [ ask n-of ideal-prof turtles with [ citizen = state and prof = profs ]
    [ set employed true ]
  ]
  [ ask turtles with [ citizen = state and prof = profs ]
    [ set employed true ]
  ]
  ifelse prof-pop = 0
  [report totinv]
  [report totinv / prof-pop]
end

to-report calc-wage-for-common [ state profs ]
  let pop count turtles with [ citizen = state ]
  let prof-pop count turtles with [ citizen = state and prof = profs ]
  let ideal-prof floor (pop * (get-ideal profs)) / 100
  let totinv (get-ideal profs) * 50
  ask turtles with [ citizen = state and prof = profs ]
    [ set employed false ]
  ifelse ideal-prof < prof-pop
  [ ask n-of ideal-prof turtles with [ citizen = state and prof = profs ]
    [ set employed true ]
  ]
  [ ask turtles with [ citizen = state and prof = profs ]
    [ set employed true ]
  ]
  ifelse prof-pop = 0
  [report totinv]
  [report totinv / prof-pop]
end

to move-to-empty-one-of [locations]  ;; turtle procedure
  move-to one-of locations
  while [any? other turtles-here] [
    move-to one-of locations
  ]
end

to do-birth
  if ticks mod 5 = 0
  [
    crt  birth-rate-1[
      set citizen 1
      set origin 1
      set prof one-of [ "A" "B" "C" "D" ]
      set employed false
      set life life-expectancy-1
      move-to-empty-one-of patches-1
    ]
    crt birth-rate-2 [
      set citizen 2
      set origin 2
      set prof one-of [ "A" "B" "C" "D" ]
      set employed false
      set life life-expectancy-2
      move-to-empty-one-of patches-2
    ]
    crt birth-rate-3 [
      set citizen 3
      set origin 3
      set prof one-of [ "A" "B" "C" "D" ]
      set employed false
      set life life-expectancy-3
      move-to-empty-one-of patches-3
    ]
    crt birth-rate-4 [
      set citizen 4
      set origin 4
      set prof one-of [ "A" "B" "C" "D" ]
      set employed false
      set life life-expectancy-4
      move-to-empty-one-of patches-4
    ]]
  
end

;;::::::::::::::::::::::::::::::::::::::::::::::PLOTS::::::::::::::::::::::::::::::::::::::::::::::::::::

to do-plot
  do-plot-population
  do-plot-gnh
  do-plot-wage
end

to do-plot-Population
  set-current-plot "Population"
  set-current-plot-pen "country-1"
  plot count turtles with [ citizen = 1 ]      
  set-current-plot-pen "country-2"
  plot count turtles with [ citizen = 2 ]
  set-current-plot-pen "country-3"
  plot count turtles with [ citizen = 3 ]      
  set-current-plot-pen "country-4"
  plot count turtles with [ citizen = 4 ]            
end

to do-plot-gnh
  set-current-plot "Gross National Happiness"
  set-current-plot-pen "gnh-1"
  plot gnh-1   
  set-current-plot-pen "gnh-2"
  plot gnh-2
  set-current-plot-pen "gnh-3"
  plot gnh-3     
  set-current-plot-pen "gnh-4"
  plot gnh-4           
end

to do-plot-wage
  set-current-plot "wage-A"
  set-current-plot-pen "wage-1A"
  plot wage-1A 
  set-current-plot-pen "wage-2A"
  plot wage-2A
  set-current-plot-pen "wage-3A"
  plot wage-3A     
  set-current-plot-pen "wage-4A"
  plot wage-4A          
end

;;::::::::::::::::::::::::::::::::::::::::::::::GETTER/SETTERS AND UTILS::::::::::::::::::::::::::::::::::::::::::::::::::::

to-report get-ideal [ profs ]
  if profs = "A"
  [ report ideal-A-perc ]
  if profs = "B"
  [ report ideal-B-perc ]
  if profs = "C"
  [ report ideal-C-perc ]
  if profs = "D"
  [ report ideal-D-perc ]
end

to-report get-unempl [state]
  report count turtles with [ citizen = state and employed = false ]
end

to-report get-gnh [ state ]
  if state = 1 [ report gnh-1 ]
  if state = 2 [ report gnh-2 ]
  if state = 3 [ report gnh-3 ]
  if state = 4 [ report gnh-4 ]
end

to set-gnh [ state new-gnh ]
  if state = 1 [ set gnh-1 new-gnh ]
  if state = 2 [ set gnh-2 new-gnh ]
  if state = 3 [ set gnh-3 new-gnh ]
  if state = 4 [ set gnh-4 new-gnh ]
end

to-report get-life-expectancy [ state ]
  if state = 1 [ report life-expectancy-1 ]
  if state = 2 [ report life-expectancy-2 ]
  if state = 3 [ report life-expectancy-3 ]
  if state = 4 [ report life-expectancy-4 ]
end

to-report get-population [ state ]
  report count turtles with [citizen = state]
end

to-report get-average-income [ state ]
  let tot-pop get-population state
  let income-A (count turtles with [ citizen = state and prof = "A" ]) * (get-wage state "A")
  let income-B (count turtles with [ citizen = state and prof = "B" ]) * (get-wage state "B")
  let income-C (count turtles with [ citizen = state and prof = "C" ]) * (get-wage state "C")
  let income-D (count turtles with [ citizen = state and prof = "D" ]) * (get-wage state "D")
  ifelse tot-pop != 0
  [ report ((income-A + income-B + income-C + income-D) / tot-pop) ]
  [ report 0 ]
end

to-report get-max-life-expectancy
  let max-exp 0
  foreach [ 1 2 3 4]
  [
    if (get-life-expectancy ?) > max-exp
    [ set max-exp (get-life-expectancy ?) ]
  ]
  report max-exp
end

to-report get-max-population
  let max-pop 0
  foreach [ 1 2 3 4]
  [
    if (get-population ?) > max-pop
    [ set max-pop (get-population ?) ]
  ]
  report max-pop
end

to-report get-max-avg-income
  let max-avg-income 0
  foreach [ 1 2 3 4]
  [
    if (get-average-income ?) > max-avg-income
    [ set max-avg-income (get-average-income ?) ]
  ]
  report max-avg-income
end

to-report get-max-unempl
  let max-unempl 0
  foreach [ 1 2 3 4]
  [
    if (get-unempl ?) > max-unempl
    [ set max-unempl (get-unempl ?) ]
  ]
  report max-unempl
end

to-report get-wage [ state profs ]
  if state = 1 [
    if profs = "A" [ report wage-1A ]
    if profs = "B" [ report wage-1B ]
    if profs = "C" [ report wage-1C ]
    if profs = "D" [ report wage-1D ]
  ]
  if state = 2 [
    if profs = "A" [ report wage-2A ]
    if profs = "B" [ report wage-2B ]
    if profs = "C" [ report wage-2C ]
    if profs = "D" [ report wage-2D ]
  ]
  if state = 3 [
    if profs = "A" [ report wage-3A ]
    if profs = "B" [ report wage-3B ]
    if profs = "C" [ report wage-3C ]
    if profs = "D" [ report wage-3D ]
  ]
  if state = 4 [
    if profs = "A" [ report wage-4A ]
    if profs = "B" [ report wage-4B ]
    if profs = "C" [ report wage-4C ]
    if profs = "D" [ report wage-4D ]
  ]
end

to set-wage [ state profs new-wage]
  if state = 1 [
    if profs = "A" [ set wage-1A new-wage]
    if profs = "B" [ set wage-1B new-wage]
    if profs = "C" [ set wage-1C new-wage]
    if profs = "D" [ set wage-1D new-wage]
  ]
  if state = 2 [
    if profs = "A" [ set wage-2A new-wage]
    if profs = "B" [ set wage-2B new-wage]
    if profs = "C" [ set wage-2C new-wage]
    if profs = "D" [ set wage-2D new-wage]
  ]
  if state = 3 [
    if profs = "A" [ set wage-3A new-wage]
    if profs = "B" [ set wage-3B new-wage]
    if profs = "C" [ set wage-3C new-wage]
    if profs = "D" [ set wage-3D new-wage]
  ]
  if state = 4 [
    if profs = "A" [ set wage-4A new-wage]
    if profs = "B" [ set wage-4B new-wage]
    if profs = "C" [ set wage-4C new-wage]
    if profs = "D" [ set wage-4D new-wage]
  ]
end
  
@#$#@#$#@
GRAPHICS-WINDOW
210
10
675
496
17
17
13.0
1
10
1
1
1
0
1
1
1
-17
17
-17
17
1
1
1
ticks

BUTTON
16
20
82
53
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
12
104
184
137
population-1
population-1
0
100
20
1
1
NIL
HORIZONTAL

SLIDER
11
234
183
267
population-2
population-2
0
100
50
1
1
NIL
HORIZONTAL

SLIDER
9
364
181
397
population-3
population-3
0
100
50
1
1
NIL
HORIZONTAL

SLIDER
8
489
180
522
population-4
population-4
0
100
50
1
1
NIL
HORIZONTAL

SLIDER
680
57
852
90
ideal-A-perc
ideal-A-perc
0
100
25
1
1
NIL
HORIZONTAL

SLIDER
681
95
853
128
ideal-B-perc
ideal-B-perc
0
100
25
1
1
NIL
HORIZONTAL

SLIDER
681
135
853
168
ideal-C-perc
ideal-C-perc
0
100
25
1
1
NIL
HORIZONTAL

SLIDER
683
177
855
210
ideal-D-perc
ideal-D-perc
0
100
25
1
1
NIL
HORIZONTAL

MONITOR
709
272
766
317
1-A%
100 * (count turtles with [ citizen = 1 and prof = \\"A\\" ]) / (count turtles with [ citizen = 1 ])
1
1
11

MONITOR
710
321
767
366
1-B%
100 * (count turtles with [ citizen = 1 and prof = \\"B\\" ]) / (count turtles with [ citizen = 1 ])
1
1
11

MONITOR
711
370
768
415
1-C%
100 * (count turtles with [ citizen = 1 and prof = \\"C\\" ]) / (count turtles with [ citizen = 1 ])
1
1
11

MONITOR
712
421
769
466
1-D%
100 * (count turtles with [ citizen = 1 and prof = \\"D\\" ]) / (count turtles with [ citizen = 1 ])
1
1
11

MONITOR
778
272
835
317
2-A%
100 * (count turtles with [ citizen = 2 and prof = \\"A\\" ]) / (count turtles with [ citizen = 2 ])
1
1
11

MONITOR
779
321
836
366
2-B%
100 * (count turtles with [ citizen = 2 and prof = \\"B\\" ]) / (count turtles with [ citizen = 2 ])
1
1
11

MONITOR
783
370
833
415
2-C%
100 * (count turtles with [ citizen = 2 and prof = \\"C\\" ]) / (count turtles with [ citizen = 2 ])
1
1
11

MONITOR
784
418
841
463
2-D%
100 * (count turtles with [ citizen = 2 and prof = \\"A\\" ]) / (count turtles with [ citizen = 2 ])
1
1
11

MONITOR
849
270
906
315
3-A%
100 * (count turtles with [ citizen = 3 and prof = \\"A\\" ]) / (count turtles with [ citizen = 3 ])
1
1
11

MONITOR
849
318
906
363
3-B%
100 * (count turtles with [ citizen = 3 and prof = \\"B\\" ]) / (count turtles with [ citizen = 3 ])
1
1
11

MONITOR
850
368
907
413
3-C%
100 * (count turtles with [ citizen = 3 and prof = \\"C\\" ]) / (count turtles with [ citizen = 3 ])
1
1
11

MONITOR
853
418
910
463
3-D%
100 * (count turtles with [ citizen = 3 and prof = \\"D\\" ]) / (count turtles with [ citizen = 3 ])
1
1
11

MONITOR
916
269
973
314
4-A%
100 * (count turtles with [ citizen = 4 and prof = \\"A\\" ]) / (count turtles with [ citizen = 4 ])
1
1
11

MONITOR
918
319
975
364
4-B%
100 * (count turtles with [ citizen = 4 and prof = \\"B\\" ]) / (count turtles with [ citizen = 4 ])
1
1
11

MONITOR
919
368
976
413
4-C%
100 * (count turtles with [ citizen = 4 and prof = \\"C\\" ]) / (count turtles with [ citizen = 4 ])
1
1
11

MONITOR
921
418
978
463
4-D%
100 * (count turtles with [ citizen = 4 and prof = \\"D\\" ]) / (count turtles with [ citizen = 4 ])
1
1
11

SLIDER
367
510
507
543
wage-discrepancy
wage-discrepancy
0
100
5
1
1
NIL
HORIZONTAL

BUTTON
10
64
96
97
NIL
go-wage
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
883
22
952
67
NIL
wage-1A
17
1
11

MONITOR
884
73
953
118
NIL
wage-2A
17
1
11

MONITOR
884
123
953
168
NIL
wage-3A
17
1
11

MONITOR
885
176
954
221
NIL
wage-4A
17
1
11

BUTTON
745
502
841
535
step-wage
go-wage
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

PLOT
1032
305
1378
523
Population
ticks
population
0.0
100.0
0.0
100.0
true
false
PENS
"country-1" 1.0 0 -10899396 true
"country-2" 1.0 0 -13345367 true
"country-3" 1.0 0 -955883 true
"country-4" 1.0 0 -5825686 true

SWITCH
215
550
360
583
wage-awareness?
wage-awareness?
0
1
-1000

BUTTON
107
63
185
96
go-empl
go-employment
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
515
549
710
582
wage-empl-by-population?
wage-empl-by-population?
1
1
-1000

SLIDER
11
143
183
176
life-expectancy-1
life-expectancy-1
0
100
75
1
1
NIL
HORIZONTAL

SLIDER
11
182
183
215
birth-rate-1
birth-rate-1
0
10
5
1
1
NIL
HORIZONTAL

SLIDER
10
270
182
303
life-expectancy-2
life-expectancy-2
0
100
75
1
1
NIL
HORIZONTAL

SLIDER
9
306
181
339
birth-rate-2
birth-rate-2
0
10
5
1
1
NIL
HORIZONTAL

SLIDER
9
402
181
435
life-expectancy-3
life-expectancy-3
0
100
75
1
1
NIL
HORIZONTAL

SLIDER
8
439
180
472
birth-rate-3
birth-rate-3
0
10
5
1
1
NIL
HORIZONTAL

SLIDER
7
528
179
561
life-expectancy-4
life-expectancy-4
0
100
75
1
1
NIL
HORIZONTAL

SLIDER
7
566
179
599
birth-rate-4
birth-rate-4
0
10
5
1
1
NIL
HORIZONTAL

SWITCH
214
509
361
542
birth-rate-on?
birth-rate-on?
0
1
-1000

MONITOR
967
27
1024
72
NIL
gnh-1
17
1
11

MONITOR
967
75
1024
120
NIL
gnh-2
17
1
11

MONITOR
967
124
1024
169
NIL
gnh-3
17
1
11

MONITOR
967
173
1024
218
NIL
gnh-4
17
1
11

SLIDER
513
511
685
544
gnh-discrepancy
gnh-discrepancy
0
10
2
0.5
1
NIL
HORIZONTAL

BUTTON
110
20
189
53
NIL
go-gnh
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
852
501
927
534
step-gnh
go-gnh
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
366
548
507
581
gnh-awareness?
gnh-awareness?
0
1
-1000

TEXTBOX
705
20
855
54
The Total should exactly be 100
14
15.0
1

PLOT
1030
175
1370
295
Gross National Happiness
ticks
GNH
0.0
100.0
0.0
40.0
true
false
PENS
"gnh-1" 1.0 0 -10899396 true
"gnh-2" 1.0 0 -13345367 true
"gnh-3" 1.0 0 -955883 true
"gnh-4" 1.0 0 -5825686 true

PLOT
1030
10
1375
165
wage-A
ticks
wage
0.0
100.0
0.0
100.0
true
false
PENS
"wage-1A" 1.0 0 -10899396 true
"wage-2A" 1.0 0 -13345367 true
"wage-3A" 1.0 0 -955883 true
"wage-4A" 1.0 0 -5825686 true

@#$#@#$#@
WHAT IS IT?
-----------
 The Immigration agent model is designed to examine the various factors responsible for triggering migration. The model enables us to compare the explanations given by two of the famous economic theories namely Neo Classical economic theory and Keynesian theory . In addition it compares these two established theories with a novel approach of utilizing people_ happiness, measured by Gross National Happiness index as a causal reason for migration. The model can be used to examine the following.
_	rate of immigration based on economical situation, employment rate and Gross National Happiness.
_	Impact of immigration on wage rates and unemployment rate.
_	Impact of population growth on migration rate and wage rate.



HOW IT WORKS
------------
You can test the three economic theories supported by the model in isolation. Set the initial population and ideal percentages.
_ Go-wage will set the mode of the model to simulate wage based immigration.
_ Go-empl will set the mode of the model to simulate employment opportunity based immigration.
_ Go-gnh will set the mode of the model to simulate Gross National Happiness (GNH) based immigration.

The turtles will migrate based on the mode. In wage-based immigration mode, a turtle will decide to migrate if it finds out about better wage in a different country. In employment mode, unemployed turtles migrate to places that have jobs. In GNH mode the turtle will go to a country which has better GNH index than its home.

you can also control factors such as birth rate, awareness, kind of profession ( population dependent / independent),etc.

HOW TO USE IT
-------------
Set the following sliders before starting the simulation

Population-x - Initial population of the four countries ( x is 1,2,3 and 4)

Ideal-X-perc - Indicates Ideal percentage of professionals with skill set X for a utopian equilibrium. ( X can take A, B, C and D )

wage-discrepancy - use this slider to set the tolderance of turtle for discrepancy in wage-rates. The turtle will migrate only of the wage difference is higher than the tolerance level.

wage-empl-by-population? - ON implies that the wages are proportional to population size. OFF indicates otherwise.

wage-awareness? - Set to 'on' if you want the turtle to be aware of the other immigrants competing for the same job. Set to 'off' to simulate Gold Rush mode. The turtle would pay no attention to the competition.

gnh-discrepancy - similar to wage-discrepancy but for GNH.

gnh-awareness? - similar to wage-awareness? but for GNH.

birth-rate-on? - Set this if you wanna enable birth and death of turtles based on life-expectancy and birth-rate sliders.

life-expectancy-X - sets the life expectancy of a turtle which was born on Country X ( X can take 1,2,3 and 4 )

birth-rate-X - sets the birth rate of Country X based on which new turtles will be created for every ticks ( X can take 1,2,3 and 4 )

Once setting up all the above. Do the following

SETUP - sets the countries ready with initial population

go-empl - unemployment induced immigration ( Keynesian Theory)

go-wage - Wage based immigration ( Neo classical theory )

go-GNH - Gross National Happiness based immigration

THINGS TO NOTICE
----------------
_ Observe how the turtles migrate over time.

_ Observe the Wage-1A plot to undrestand the change in wage-rate for various settings.

_ Observe the Population plot to see how a given immigration strategy would affect the population.

_ Observe the GNH plot to see how GNH value changes over time.

_ Observe the monitors to see how every country strives to achieve equilibrium ( ideal percentages)


THINGS TO TRY
-------------
_ Modify various settings to see what kind of immigration strategy would help achieve economic equilibrium faster.


EXTENDING THE MODEL
-------------------
_ Add immigration control. Every country should allow only a certain number of immigrants.
_ Add technological advancement factor which would reduce the necessity for manual labor thereby increasing unemployment.


NETLOGO FEATURES
----------------
This section could point out any especially interesting or unusual features of NetLogo that the model makes use of, particularly in the Procedures tab.  It might also point out places where workarounds were needed because of missing features.


RELATED MODELS
--------------
This section could give the names of models in the NetLogo Models Library or elsewhere which are of related interest.


CREDITS AND REFERENCES
----------------------
This section could contain a reference to the model's URL on the web if it has one, as well as any other necessary credits or references.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="wage-empl-by-pop-1" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go-wage</go>
    <timeLimit steps="500"/>
    <metric>count turtles with [ citizen = 1 ]</metric>
    <metric>count turtles with [ citizen = 2 ]</metric>
    <metric>count turtles with [ citizen = 3 ]</metric>
    <metric>count turtles with [ citizen = 4 ]</metric>
    <metric>wage-1A</metric>
    <metric>wage-2A</metric>
    <metric>wage-3A</metric>
    <metric>wage-4A</metric>
    <enumeratedValueSet variable="birth-rate-1">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-2">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-3">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-4">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-1">
      <value value="20"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-2">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-3">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-4">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-2">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-3">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-4">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-A-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-B-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-C-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-D-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-discrepancy">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-discrepancy">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-empl-by-population?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-on?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-awareness?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="wage-empl-by-pop-1-t" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go-wage</go>
    <timeLimit steps="500"/>
    <metric>count turtles with [ citizen = 1 ]</metric>
    <metric>count turtles with [ citizen = 2 ]</metric>
    <metric>count turtles with [ citizen = 3 ]</metric>
    <metric>count turtles with [ citizen = 4 ]</metric>
    <metric>wage-1A</metric>
    <metric>wage-2A</metric>
    <metric>wage-3A</metric>
    <metric>wage-4A</metric>
    <enumeratedValueSet variable="birth-rate-1">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-2">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-3">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-4">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-1">
      <value value="20"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-2">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-3">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-4">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-2">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-3">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-4">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-A-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-B-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-C-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-D-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-discrepancy">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-discrepancy">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-empl-by-population?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-on?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="wage-empl-by-pop-1-o" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go-wage</go>
    <timeLimit steps="500"/>
    <metric>count turtles with [ citizen = 1 ]</metric>
    <metric>count turtles with [ citizen = 2 ]</metric>
    <metric>count turtles with [ citizen = 3 ]</metric>
    <metric>count turtles with [ citizen = 4 ]</metric>
    <metric>wage-1A</metric>
    <metric>wage-2A</metric>
    <metric>wage-3A</metric>
    <metric>wage-4A</metric>
    <enumeratedValueSet variable="birth-rate-1">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-2">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-3">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-4">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-1">
      <value value="10"/>
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-2">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-3">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-4">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-2">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-3">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-4">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-A-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-B-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-C-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-D-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-discrepancy">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-discrepancy">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-empl-by-population?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-on?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="wage-empl-by-pop-1-o-wd" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go-wage</go>
    <timeLimit steps="500"/>
    <metric>count turtles with [ citizen = 1 ]</metric>
    <metric>count turtles with [ citizen = 2 ]</metric>
    <metric>count turtles with [ citizen = 3 ]</metric>
    <metric>count turtles with [ citizen = 4 ]</metric>
    <metric>wage-1A</metric>
    <metric>wage-2A</metric>
    <metric>wage-3A</metric>
    <metric>wage-4A</metric>
    <enumeratedValueSet variable="birth-rate-1">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-2">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-3">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-4">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-1">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-2">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-3">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-4">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-2">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-3">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-4">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-A-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-B-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-C-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-D-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-discrepancy">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-discrepancy">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-empl-by-population?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-on?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="go-empl-by-pop-1" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go-employment</go>
    <timeLimit steps="500"/>
    <metric>count turtles with [ citizen = 1 ]</metric>
    <metric>count turtles with [ citizen = 2 ]</metric>
    <metric>count turtles with [ citizen = 3 ]</metric>
    <metric>count turtles with [ citizen = 4 ]</metric>
    <metric>wage-1A</metric>
    <metric>wage-2A</metric>
    <metric>wage-3A</metric>
    <metric>wage-4A</metric>
    <enumeratedValueSet variable="birth-rate-1">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-2">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-3">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-4">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-1">
      <value value="10"/>
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-2">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-3">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-4">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-2">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-3">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-4">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-A-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-B-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-C-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-D-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-discrepancy">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-discrepancy">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-empl-by-population?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-on?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="go-empl-by-pop-2-brate" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go-employment</go>
    <timeLimit steps="500"/>
    <metric>count turtles with [ citizen = 1 ]</metric>
    <metric>count turtles with [ citizen = 2 ]</metric>
    <metric>count turtles with [ citizen = 3 ]</metric>
    <metric>count turtles with [ citizen = 4 ]</metric>
    <metric>wage-1A</metric>
    <metric>wage-2A</metric>
    <metric>wage-3A</metric>
    <metric>wage-4A</metric>
    <metric>count turtles with [ citizen = 1  and employed = false]</metric>
    <metric>count turtles with [ citizen = 2  and employed = false]</metric>
    <metric>count turtles with [ citizen = 3  and employed = false]</metric>
    <metric>count turtles with [ citizen = 4  and employed = false]</metric>
    <enumeratedValueSet variable="birth-rate-1">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-2">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-3">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-4">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-1">
      <value value="25"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-2">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-3">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-4">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-2">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-3">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-4">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-A-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-B-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-C-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-D-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-discrepancy">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-discrepancy">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-empl-by-population?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-on?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="go-gnh-by-pop-1" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go-gnh</go>
    <timeLimit steps="500"/>
    <metric>count turtles with [ citizen = 1 ]</metric>
    <metric>count turtles with [ citizen = 2 ]</metric>
    <metric>count turtles with [ citizen = 3 ]</metric>
    <metric>count turtles with [ citizen = 4 ]</metric>
    <metric>wage-1A</metric>
    <metric>wage-2A</metric>
    <metric>wage-3A</metric>
    <metric>wage-4A</metric>
    <metric>gnh-1</metric>
    <metric>gnh-2</metric>
    <metric>gnh-3</metric>
    <metric>gnh-4</metric>
    <enumeratedValueSet variable="birth-rate-1">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-2">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-3">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-4">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-1">
      <value value="10"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-2">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-3">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population-4">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-2">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-3">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="life-expectancy-4">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-A-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-B-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-C-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ideal-D-perc">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-discrepancy">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-discrepancy">
      <value value="0.5"/>
      <value value="1"/>
      <value value="1.5"/>
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gnh-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-empl-by-population?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="birth-rate-on?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="wage-awareness?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
