extensions [table]

breed [people person]
breed [expectations expectation]
breed [technologies technology]


people-own [usage understanding influencer persistence]
expectations-own [feature strength]
technologies-own [affordances]


directed-link-breed [holds hold]


to setup
  ca  
  

  ;; If auto-total-people is on, make sure the number of people expecting a, b and both doesn't exceed
  ;; the total number set. 
  ;; Populate the rest of the world (up to the total number set) with people with no expectations.
  
  ifelse auto-total-people != "off" and #-expecting-a + #-expecting-b + #-expecting-both > auto-total-people
  [ user-message (word "There are too many people. Turn auto-total off, or reduce the number of people expecting a, b or both.")]
  [ if auto-total-people != "off"
    [ set #-no-expectations auto-total-people - (#-expecting-a + #-expecting-b + #-expecting-both) ]
  
  
  ;; create the total number of people who will be in the model
  ;; distribute them randomly around the world
  ;; format their size and shape
  ;; set their usage blank
  ;; format their color to reflect their usage
  ;; give each of them a blank table that will hold their expectations about the technology
  
  create-people total-# 
  [ setxy random-pxcor random-pycor
    set shape "person"
    set size 1.6
    set usage []
    color-code usage .8 
    set understanding table:make ]  
  
  
  ;; for the number of people expecting only a, pick people with blank tables, and set their understanding of feature a to 1 
  ask n-of #-expecting-a people with [ table:length understanding < 1 ]
  [ table:put understanding "a" 1 ]
  
  ;; for the number of people expecting only b, pick people with blank tables, and set their understanding of feature b to 1 
  ask n-of #-expecting-b people with [ table:length understanding < 1 ]
  [ table:put understanding "b" 1 ]
  
  ;; for the number of people expecting a and b, pick people with blank tables, and set their understanding of 
  ;; features a and b to 1   
  ask n-of #-expecting-both people with [ table:length understanding < 1 ]
  [ table:put understanding "a" 1
    table:put understanding "b" 1 ] 
  
  
  ;;  build a network of expectations to provide a visual representation of each person's table of understanding
  ask people
  [ update-expectations 
    
    ;; if limited persistence is on, give people a persistence of 3
    ;; if it's off, give people a persistence that is extremely high   
    ifelse limited-persistence
    [ set persistence 3 ]
    [ set persistence 1000 ] ]

  ;; format the network of expectations
  ask expectations
  [ format-expectations ]
  


  ;; set up the technology 
  adopt-technology  

  reset-ticks ]
  
  

end


to update-expectations
  ;; ask people whose table of understandings doesn't match the visual representation of their expectations
  ask people with [table:length understanding != count out-hold-neighbors]

  ;; create a blank slate 
  [ ask out-hold-neighbors 
    [ die ]
    
  ;; build the network of expectations up from the understanding table  
  let instructions table:keys understanding
  while [instructions != []]
  [ hatch-expectations 1
    [ create-hold-from myself
      ask my-in-holds [hide-link] 
      set feature first instructions
      set strength [table:get understanding first instructions] of myself 
      set instructions but-first instructions  ] ] ]

  ask expectations
  [ format-expectations ] 
  
end



to format-expectations
  
  ;; make expectations color, shape and placement reflect what they indicate

  if strength > 0
  [ show-turtle 
    set shape "circle"
    set size .5 ]

  if strength < 0
  [ show-turtle
    set shape "x"
    set size .6 ]

  if strength = 0
  [hide-turtle]
  
  
  if feature = "a"
  [ setxy ( [ xcor ] of one-of in-hold-neighbors - .5) ( [ ycor ] of one-of in-hold-neighbors + .5)  ] 
  if feature = "b"
  [ setxy ( [ xcor ] of one-of in-hold-neighbors + .5) ( [ ycor ] of one-of in-hold-neighbors + .5) ]
  
  color-code feature 1.5
  
end




to adopt-technology
 
create-technologies 1
  [ set shape "box"
    set size 3
    
    ;; create a blank table that will hold affordances
    set affordances table:make
    
    ;; fill in the table from the features selected by the user
    let instructions (sentence technology-affordances)
    while [instructions != [] ]
  [ table:put affordances first instructions 1
    set instructions but-first instructions ] 
  
  color-code (table:keys affordances) -2.5]
    

end


;; color code expectations, usage and affordances to  provide a visual indication of whether they match.  
to color-code [thing number]

;; I tried to let the number (i.e. the amount of color change) be set within this procedure, but couldn't get it to work.
;; When the procedure was called by something that didn't hold one of the possible "things" it would
;; stop during the if statements and return an error message. Not sure why it wouldn't just return false and keep going...
;; But anyway, I relented and made the number I want each element's color adjusted by its own input, that I just list
;; when I call the procedure.
;; It's still much more parsimonious this way than having entirely separate procedures to color code expectations,
;; usage, and affordances which is what I used to have...

;let number 1
;if thing = usage 
;[set number .8]
;if thing = (table:keys affordances)
;[set number -2.5]
;if thing = feature
;[set number 1.5] 

  
  if member? "a" thing and not member? "b" thing
    [ set color blue + number ] 
  if member? "b" thing and not member? "a" thing
    [ set color yellow + number ] 
  if member? "a" thing and member? "b" thing
    [ set color green + number] 
  if not member? "a" thing and not member? "b" thing
    [ set color gray + number ]
  
end




to go
  
  ;; stop conditions
  if all? people [persistence = 0] 
    [ stop ]
  if ticks > 99 and (all? people [usage = []] 
    or all? people [usage = ["a"]] 
    or all? people [usage = ["b"]] 
    or all? people [usage = ["a" "b"] or usage = ["b" "a"]]) 
    [ stop ]
  
  ;; running the model  
  ask people
  [ move
    interact
    update-expectations ] 
  
  tick  
  
end  
  
  
  
to move
let path random-normal 0 30
    rt path
    fd 1 
    ask out-hold-neighbors
    [ rt path
      fd 1 ]     
end  
  
  
to interact
  
  set influencer nobody
  
  if persistence > 0
  
  ;; if I am near the technology or a person, pick one of those entities to influence me  
  [ let potential-influencers (turtle-set technologies in-radius 3 other people in-radius 1)
    ;show potential-influencers
    set influencer one-of potential-influencers
    ;show influencer
    
    ;; if I am being influenced by a person (social interaction), then learn from their expectations
    if is-person? influencer 
    [ learn-from ([understanding] of influencer) ]
    
    ;; if I am being influenced by the technology (material interaction), then if I have some understanding of
    ;; what the technology is for I will try to use it for that.
    ;; otherwise I will learn from the technology
    if is-technology? influencer 
    [ ifelse table:length understanding > 0
      [ use-technology ] 
      [ learn-from ([affordances] of influencer) ]  
      ;; increment my persistence down 
      set persistence persistence - 1 ] ]
  
  
end 
   
  

 
to learn-from [source]
  
  ;; if the influencer has any features to learn from,
  ;; pick one to be the insight I learn
  if table:length source > 0
  [ let insight one-of table:keys source
    
    ;; if that feature is new to me, put the insight into my table of understandings with the same value
    ;; that my influencer has for that feature
    if not table:has-key? understanding insight
    [ table:put understanding insight table:get source insight ] ]
  
end 
 
 

to use-technology

;; if I have any positive expectations, pick one of those to be the way I try to use the technology 
if any? out-hold-neighbors with [strength > 0] 
[ let use [feature] of one-of out-hold-neighbors with [ strength > 0 ]
  
;; Note: this is the only time I use the expectations breed for more than just visualization. I tried 
;; several ways of doing this without appealing to the expectations, but none of them worked. Can't ask 
;; about values in a table directly (only keys), can't flatten a table to a non-nested list using sentence 
;; (not sure why)... I couldn't think of any other reasonably parsimonious way, using just the tables, 
;; that I could pick a feature with a positive value. I wanted to be consistent about either using the 
;; expectations turtles for functionality or not. I found them awkward to use for some of the more central 
;; interactional procedures--in particular I wanted to create learning procedures that would be "symmetric"
;; (ie work the same) with both people and technology, so I wanted a way to store expectations and 
;; affordances the same way for both people and technology...plus it felt strange for expectations to be 
;; "held" by people, but not be a directly accessible property of those people, so on balance I decided to 
;; switch to using tables. But in this case, I just couldn't figure out a way to do what I needed
;; with tables. I think the functionality should be the same, just a bit less elegant to switch.

;; if the feature I'm trying to use is one of the technology's affordances, then put that feature into my usage
;; if not, set my expectation for that feature negative (and ask my expectation network to adjust accordingly) 
;; and remove it from my usage 
  ifelse table:has-key? [affordances] of influencer use
[ set usage lput use usage
  set usage remove-duplicates usage
  color-code usage .8 ]
[ table:put understanding use -1 
  ask out-hold-neighbors with [feature = use ]
  [ set strength [table:get understanding use] of myself ] 
  set usage remove use usage ] ]

  
end 



to-report total-#
report #-expecting-a + #-expecting-b + #-expecting-both + #-no-expectations
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
649
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
68
26
134
59
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
19
196
191
229
#-expecting-a
#-expecting-a
0
100
1
1
1
NIL
HORIZONTAL

SLIDER
19
242
191
275
#-expecting-b
#-expecting-b
0
100
49
1
1
NIL
HORIZONTAL

SLIDER
19
287
191
320
#-expecting-both
#-expecting-both
0
100
0
1
1
NIL
HORIZONTAL

SLIDER
19
332
190
365
#-no-expectations
#-no-expectations
0
100
50
1
1
NIL
HORIZONTAL

CHOOSER
19
120
193
165
technology-affordances
technology-affordances
["a"] ["b"] ["a" "b"] []
0

BUTTON
26
73
107
106
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
122
72
185
105
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
679
295
879
445
usages
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"using for a" 1.0 0 -13345367 true "" "plot count people with [ usage = [\"a\"] ]"
"using for b" 1.0 0 -4079321 true "" "plot count people with [ usage = [\"b\"] ]"
"not using" 1.0 0 -7500403 true "" "plot count people with [ usage = [] ]"

PLOT
676
89
876
239
expectations
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"a" 1.0 0 -13345367 true "" "plot count expectations with [feature = \"a\" and strength > 0]"
"not a" 1.0 0 -11053225 true "" "plot count expectations with [feature = \"a\" and strength < 0]"
"b" 1.0 0 -4079321 true "" "plot count expectations with [feature = \"b\" and strength > 0]"
"not b" 1.0 0 -4539718 true "" "plot count expectations with [feature = \"b\" and strength < 0]"

SWITCH
680
17
857
50
limited-persistence
limited-persistence
1
1
-1000

MONITOR
71
389
128
434
NIL
total-#
17
1
11

CHOOSER
34
451
172
496
auto-total-people
auto-total-people
"off" 100
1

@#$#@#$#@
## WHAT IS IT?

This model is based on Leonardi, P. M. (2012). Car Crashes Without Cars: Lessons About Simulation Technology and Organizational Change from Automotive Design. MIT Press (MA).

It is intended to represent what happens when a new technology is introduced into an organization and people are trying to figure out what it is good for. It models a cyclical set of dynamics between people's interactions with one another ("social interactions"), and interactions with the technology itself ("material interactions") working together to determine whether the tool is actually taken up and used. 

The cycle works as follows: Social interactions (i.e. people talking to one another about the new tool) shape people’s expectations for the technology. These expectations turn shape their material interactions with the tool, as people tend to find only the affordances they know to look for. Information from these material interactions (i.e. whether the technology worked as a person expected or not) is then incorporated into future social interactions.

In Leonardi's ethnographic work, he found that these dynamics led to two different patterns:

1. People begin with a widespread expectation that the technology affords something it does not. In the end, most people abandon the technology.
2. People begin without clear expectations of what the technology affords. In the end most people use the technology.

This model is the first and simplest in a series designed to model these dynamics.

## HOW IT WORKS

The agents in this model are people and technologies. People have three important attributes: 

1. A set of expectations about what the technology affords, including both a set of features (“a”, “b”, both, or neither) and a valence for each feature present (1 or -1). A valence of 1 represents a belief that the technology does afford that feature. A valence of -1 represents a belief that the technology does not afford that feature. People can also lack any expectation for a particular feature. 
2. A usage, which refers to the features of the technology that they use. Usage can also take values of “a,” “b,” both or neither, however it does not hold valence. 
3. A level of persistence. This refers to how many times people will interact with the technology before their expectations and usage become permanently fixed. 

The technology (there is never more than one technology in a given simulation) holds one attribute: its affordances. The technology’s set of affordances can include “a”, “b”, both, or neither. If a particular feature is present its valence is always positive. 


When the model begins, each person follows the following rules at each tick:

1. Turn a little and take a step forward 
2. If I am close to any other agents (people or technology) and my persistence has not run out, pick a nearby agent to interact with
    1. If I am interacting with another person who has an expectation about a feature I didn’t know about, take on that person’s expectation
    2. If I am interacting with the technology, try to use it for one thing I expect it will be able to do
        1. If that feature is among the technology’s affordances, add it to my usage. If not, change my expectations to not believing the technology affords that feature
        3. If I have no expectations, learn one feature from the technology itself (the same way I would from a person)
        4. Increment my persistence down
        5. (If I have only negative expectations do nothing)


## HOW TO USE IT

Select a set of affordances for the technology. 

Select the number of people who will start with expectations of “a,” “b,” both, or neither. All starting expectations are positive. 

If auto-total people is set to a number, the model will automatically add people with no expectations to the model until the number of people equals auto-total number. If the user has already selected more than that number of people with starting expectations of "a", "b" or both, an error message will pop up.

Optionally “limited-persistence” can be turned on, which sets starting persistence at 3 for all people. When “limited persistence” is off, people start with a persistence of 1000—-an arbitrarily high number.

## THINGS TO NOTICE

People's expectations are represented by little symbols next to their head. Blue represents "a", and yellow represents "b". Dots indicate a positive expectation. Xs indicate a negative expectation.

People's usage is represented by the color of their body. Gray represents no usage, blue represents "a," yellow represents "b" and green represents both "a" and "b."

## THINGS TO TRY

Try to match the reference patterns Leonardi found in his ethnographic work!

## EXTENDING THE MODEL

Try making "persistence" a slider instead of a switch.

## NETLOGO FEATURES

Note that both expectations and affordances are coded here using the Tables extension.

## RELATED MODELS

Leonardi Model 2: http://modelingcommons.org/browse/one_model/4074

Leonardi Model 3: http://modelingcommons.org/browse/one_model/4075

## CREDITS AND REFERENCES

http://modelingcommons.org/browse/one_model/4072

Leonardi, P. M. (2012). Car Crashes Without Cars: Lessons About Simulation Technology and Organizational Change from Automotive Design. MIT Press (MA).
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   