
globals [
  shape-options
  social-gain
]

turtles-own [
  ;;; internal setting variables
  user-id
  my-shape
  social-currency
  
  ;;; turn variables
  take-action
  my-neighbors
  got-away?
  already-moved?
  action-selected?
  next-xcor
  next-ycor
  
  ;;; lifetime variables
  my-thefts
  my-placids
  my-protects
  how-many-times-I-got-away
  
  ;;; helper variables
  my-count ;; this is just a placeholder variable; not at all necessary for logic but is handy to avoid retyping code bloacks
  ]

to startup
  hubnet-reset
  setup
end

to setup
  ca
  
  ask patches [ set pcolor grey ]
  set social-gain 10 ;; the one global; might want to make the increment greater or smaller

  reset-ticks
end

to play
  update-active-turtles
  display
  if (any? turtles) [
  
    ;;; first code block resets internals, moves the turtles, and picks what to do
    ask turtles [
      ;;; reset some information at each tick
      send-player-info
      set got-away? false 
      set already-moved? false
      set action-selected? false
      set my-neighbors other turtles in-radius 10 ;; this is a smaller radius than the baseline netlogo model since we want to give turtles a chance to avoid thieves
    ]
    
    if any? turtles with [color = red] [
      hubnet-broadcast-message "thieves who were caught stealing last turn (red turtles) move (10 seconds)"
      move-thieves ;;; thieves move first so that others can avoid them
      ask turtles with [color = red] [
        setxy next-xcor next-ycor
      ]
      display ;; have to tick multiple times per round; in this function we also update user info
    ]
    hubnet-broadcast-message "everyone else moves (10 seconds)"
    move-others ;;; then everyone else
    ask turtles with [not (color = red)] [
      setxy next-xcor next-ycor
    ]
    display
    
    hubnet-broadcast-message "Everyone: pick what you're going to do next. You have 10 seconds"
    get-actions ;;; after everyone has moved, they choose what to do
    
    resolve ;;; and they resolve the consequences of their actions, including color change
    
    tick
  ]
  
end

to resolve
  ;;; this is only its own function out of convenience; could be in-line with go
  ask turtles [
    ifelse (take-action = "steal")
    [ steal 
      if (random 100 < probability-of-getting-away) 
      [ set got-away? true
        set how-many-times-I-got-away how-many-times-I-got-away + 1]
      set my-thefts my-thefts + 1] ;;; thieves have a chance of getting away with it
    [ ifelse (take-action = "protect") 
      [ protect 
        set my-protects my-protects + 1]
      [ do-nothing 
        set my-placids my-placids + 1]
    ]
  ]
end

to steal
  ;;; note that steal actually handles MOST of what happens to turtles, i.e., both thieves and protectors  
 
  ;;; in this implementation, a successful thief takes TOTAL social-gain divided amongst the victims. another possibility would be to have it take a social-gain from EACH victim
  ifelse (got-away? or not any? my-neighbors with [take-action = "protect"]) ;; if no one is watching out for thieves or you get away with it
  [ set my-count (count my-neighbors)
    if (my-count > 0) ;; provided you have neighbors
    [ set social-currency (social-currency + social-gain) ;; get social-gain from chumpy neighbors
      ask my-neighbors [set social-currency (social-currency - (social-gain / [my-count] of myself))] ];; each neighbor loses social-gain/count neighbors                                                                                                    ;;; if you have no neighbors, don't sweat it, you do nothing.
    set color green ;;; it looks like you did nothing
  ] 
  
  ;;; otherwise, someone is watching out for thieves AND you got caught
  [ set my-count (count my-neighbors with [take-action = "protect"])
    set social-currency (social-currency - social-gain) ;; lose social-gain
    ask my-neighbors with [take-action = "protect"]
      [ set social-currency (social-currency + (social-gain / [my-count] of myself))]  ;;; protectors don't each get social-gain, since the more protectors you have, the less each protection matters
   
    set color red ;;; labeled a thief
  ]
end

to protect
  set my-count (count my-neighbors with [take-action = "steal" and not got-away?])
  if (my-count = 0) ;; if none of your neighbors tried to steal and did not get away with it, then you are being a busybody
    [ set social-currency (social-currency - social-gain) ;; lose social-gain 
      ask my-neighbors [set social-currency (social-currency + (social-gain / [count my-neighbors] of myself))]] ;;; all your neighbors are smugly satisfied with themselves and each other
  set color blue ;;; protectors turn blue regardless of whether they are nosy or not
end

to do-nothing
  ;;; the only thing we need to do within this function is change the color a bit
  set color green
end

to update-active-turtles
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-member ]
    [
      if hubnet-exit-message?
      [ remove-member ]
    ]
  ]
end

;; Create a turtle, set its shape, color, and position
;; and tell the node what its turtle looks like and where it is
to create-new-member
  crt 1
  [
    set user-id hubnet-message-source
    set social-currency 20 ;; they should start with something so they have something to lose.
    set my-shape one-of shapes
    set shape my-shape
    set size 3
    set color black
    hubnet-send-message user-id (word "You are a: " my-shape)
    send-player-info
  ]
end

to move-thieves
  ask turtles with [(color = red)] [ ;;; just the people who stole last turn
    hubnet-send-message user-id "Click where you want to move next. You have 10 seconds."
  ]
  reset-timer
  while [timer < 10]
  [
    while [hubnet-message-waiting?]
    [ 
      hubnet-fetch-message    
      ifelse hubnet-enter-message?
      [ create-new-member ]
      [
        ifelse hubnet-exit-message?
        [ remove-member ]
        [
          ask turtles with [user-id = hubnet-message-source and color = red and not already-moved?]
          [
            ifelse hubnet-message-tag = "View"
            [ set next-xcor (item 0 hubnet-message)
              set next-ycor (item 1 hubnet-message)
              set already-moved? true]
            [hubnet-send-message user-id "Please click within the view screen."]
          ]
        ]  
      ]
    ]
  ]
end
  
to move-others
  ask turtles with [ not (color = red)] [ ;;; just people who did not steal
    hubnet-send-message user-id "Click where you want to move next. You have 10 seconds."
  ]
  
  reset-timer
  while [timer < 10]
  [
    while [hubnet-message-waiting?]
    [ 
      hubnet-fetch-message    
      ifelse hubnet-enter-message?
      [ create-new-member ]
      [
        ifelse hubnet-exit-message?
        [ remove-member ]
        [
          ask turtles with [user-id = hubnet-message-source and not (color = red) and not already-moved?]
          [
            ifelse hubnet-message-tag = "View"
            [ set next-xcor (item 0 hubnet-message)
              set next-ycor (item 1 hubnet-message)
              set already-moved? true]
            [hubnet-send-message user-id "Please click within the view screen."]
          ] 
        ] 
      ]
    ]
  ]
end

to get-actions
  reset-timer
  while [timer < 15]
  [
    while [hubnet-message-waiting?]
    [
      hubnet-fetch-message
      ifelse hubnet-enter-message?
      [ create-new-member ]
      [
        ifelse hubnet-exit-message?
        [ remove-member ]
        [
          ask turtles with [user-id = hubnet-message-source and not action-selected?] ;; just to stop people from changing their actions a lot and slowing things down
          [ 
            ifelse hubnet-message-tag = "View"
            [hubnet-send-message user-id "Please select an action using one of the buttons."]
            [ set take-action hubnet-message-tag 
              set action-selected? true]
          ]
        ]
      ]
    ]
  ]
end

to remove-member
  ask turtles with [user-id = hubnet-message-source] [die]
end

to send-player-info ;; player procedure
  ;;; global info
  hubnet-send user-id "Percent Chance of getting away with stealing" probability-of-getting-away
  hubnet-send user-id "Your social currency" social-currency
  hubnet-send user-id "How many times you've stolen" my-thefts
  hubnet-send user-id "How many times you've protected" my-protects
  hubnet-send user-id "How many times you've done nothing"  my-placids
  hubnet-send user-id "How many times you got away with stealing" how-many-times-I-got-away
  hubnet-send user-id "Richest Turtle" highest-score
  hubnet-send user-id "Poorest Turtle" lowest-score
end

to-report highest-score
  report max [social-currency] of turtles
end

to-report lowest-score
  report min [social-currency] of turtles
end

to-report proportion-turtles-who-just-stole
  ifelse (count turtles = 0)
  [report 0]
  [ report turtles-who-just-stole / count turtles]
end

to-report proportion-turtles-who-just-protected
  ifelse (count turtles = 0)
  [report 0]
  [ report turtles-who-just-protected / count turtles]
end

to-report proportion-turtles-who-just-did-nothing
  ifelse (count turtles = 0)
  [report 0]
  [ report turtles-who-just-did-nothing / count turtles]
end

to-report turtles-who-just-stole
  report count turtles with [take-action = "steal"]
end

to-report turtles-who-just-protected
  report count turtles with [take-action = "protect"]
end

to-report turtles-who-just-did-nothing
  report count turtles with [take-action = "do-nothing"]
end

to-report num-thefts ;; turtle reporter
  report my-thefts
end

to-report num-placidities
  report my-placids
end

to-report num-protections
  report my-protects
end

to-report times-gotten-away
  report how-many-times-I-got-away
end
@#$#@#$#@
GRAPHICS-WINDOW
417
11
979
594
50
50
5.47
1
10
1
1
1
0
1
1
1
-50
50
-50
50
1
1
1
ticks
30.0

BUTTON
31
13
94
46
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
24
65
226
98
probability-of-getting-away
probability-of-getting-away
0
100
30
5
1
NIL
HORIZONTAL

BUTTON
115
14
178
47
NIL
play
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
13
112
360
327
Actions per turn
Time
Proportion
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Thefts" 1.0 0 -2674135 true "" "plot proportion-turtles-who-just-stole"
"Protects" 1.0 0 -13345367 true "" "plot proportion-turtles-who-just-protected"
"Placids" 1.0 0 -13840069 true "" "plot proportion-turtles-who-just-did-nothing"

@#$#@#$#@
## WHAT IS IT?

This model is a way to explore how people select whether to exploit, protect, or ignore their neighbors when their neighbors can see what they have previously done.  Turtles have social currency, which tracks how well they are doing.

## HOW IT WORKS

Every turn, users who were caught stealing last turn (red turtles) move somewhere on the board. Then everyone else (including new turtles) move.  Finally, everyone selects their next action.

There is a global probability of not getting caught which indicates how likely a thieving turtle is to go undetected.

If turtles steal and no one nearby protects or they get away with the theft, they get a certain amount of social currency from their neighbors.  If someone nearby protects and the thief is not sufficiently sneaky, the protectors get social currency from the thieves.  If someone protects and fails to catch a thief, they give social currency to all their neighbors.

## HOW TO USE IT

If you were caught stealing last turn (you are red), choose where to move by clicking on the screen.  If you did not steal or were not caught, you move after the thieves.  Finally, everyone chooses what to do next.

## THINGS TO NOTICE

Pay close attention to the probability of getting away with theft.  It defines how likely you are to be caught if you steal and someone near you protects.

## CREDITS AND REFERENCES

This model was developed by Joseph Blass as part of a final project for Uri Wilensky's Multi-Agent Modeling class at Northwestern University, Spring 2015.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Vary-p-Getaway-&amp;-initCurrency-&amp;-mateThreshold" repetitions="4" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>proportion-turtles-who-just-stole</metric>
    <metric>proportion-turtles-who-just-did-nothing</metric>
    <metric>proportion-turtles-who-just-protected</metric>
    <metric>proportion-of-thieving-turtles</metric>
    <metric>proportion-of-protector-turtles</metric>
    <metric>proportion-of-placid-turtles</metric>
    <metric>proportion-of-mixed-turtles</metric>
    <enumeratedValueSet variable="theft-threshold">
      <value value="34"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stop-at-5000-turtles-or-1000-ticks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-social-currency">
      <value value="11"/>
      <value value="10"/>
      <value value="61"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mate-threshold">
      <value value="40"/>
      <value value="10"/>
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="protect-threshold">
      <value value="66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-of-getting-away">
      <value value="0"/>
      <value value="10"/>
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="vary-initThresholds-&amp;-p-getaway" repetitions="4" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>proportion-turtles-who-just-stole</metric>
    <metric>proportion-turtles-who-just-did-nothing</metric>
    <metric>proportion-turtles-who-just-protected</metric>
    <metric>proportion-of-thieving-turtles</metric>
    <metric>proportion-of-protector-turtles</metric>
    <metric>proportion-of-placid-turtles</metric>
    <metric>proportion-of-mixed-turtles</metric>
    <enumeratedValueSet variable="theft-threshold">
      <value value="20"/>
      <value value="10"/>
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stop-at-5000-turtles-or-1000-ticks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-social-currency">
      <value value="21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mate-threshold">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="protect-threshold">
      <value value="20"/>
      <value value="10"/>
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-of-getting-away">
      <value value="0"/>
      <value value="10"/>
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
VIEW
551
13
1101
563
0
0
0
1
1
1
1
1
0
1
1
1
-50
50
-50
50

BUTTON
14
22
107
82
steal
NIL
NIL
1
T
OBSERVER
NIL
NIL

BUTTON
229
23
351
82
do-nothing
NIL
NIL
1
T
OBSERVER
NIL
NIL

BUTTON
116
22
220
81
protect
NIL
NIL
1
T
OBSERVER
NIL
NIL

MONITOR
44
168
168
217
Your social currency
NIL
3
1

MONITOR
373
29
513
78
Richest Turtle
NIL
3
1

MONITOR
374
94
512
143
Poorest Turtle
NIL
3
1

MONITOR
28
225
188
274
How many times you've stolen
NIL
0
1

MONITOR
21
279
198
328
How many times you've protected
NIL
3
1

MONITOR
15
332
209
381
How many times you've done nothing
NIL
0
1

TEXTBOX
232
151
602
272
At each turn:\n  * turtles who stole last turn move\n  * other turtles move\n  * turtles decide what to do
15
0.0
1

MONITOR
5
389
230
438
How many times you got away with stealing
NIL
3
1

TEXTBOX
233
244
543
531
Turtles who steal:\n  * gain social currency from their neighbors if they are not caught\n  * lose social currency to any neighbors who catch them stealing\nTurtles who protect:\n  * gain social currency from turtles they catch stealing\n  * lose social currency to all their neighbors if they don't catch anyone stealing\nTurtles who do nothing:\n  * can get social currency if they have a nosy neighbor
12
0.0
1

MONITOR
41
94
308
143
Percent Chance of getting away with stealing
NIL
3
1

PLOT
7
453
404
670
Actions per turn
Time
Proportion
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Thefts" 1.0 0 -2674135 true
"Protects" 1.0 0 -13345367 true
"Placids" 1.0 0 -13840069 true

@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
