;;; originally all the globals were able to be set by the user; too many damn parameters! 
;;; Now they are set in setup; the indicated settings provide some nice behavior
globals [
  number-of-turtles
  social-radius
  exile-distance
  social-gain  
  life-expectancy
]

turtles-own [
  ;;; internal setting variables
  social-currency
  p-theft
  p-protect
  age
  my-life-expectancy
  
  ;;; color variables
  r
  g
  b
  
  ;;; turn variables
  take-action
  my-neighbors
  got-away?
  
  ;; reproduction variables
  can-breed?
  my-mateworthy-neighbors
  my-mate
    
  ;;; helper variables
  random-choice-num ;;; every turn a turtle uses random to pick what to do; this is so we can pick one random number per turn in the ifelse block
  my-count ;; this is just a placeholder variable; not at all necessary for logic but is handy to avoid retyping code bloacks
  ]

to setup
  ca
  
  if (verbose? and protect-threshold <= theft-threshold) [ ;;; verbose is just there to be able to turn this message off for behavior-space experiments
    user-message "Oops: the protection threshold must be larger than the theft threshold"
    stop
  ]
  
  ;;; set globals to some nice defaults
  set number-of-turtles 150 ;;; This mostly affects how fast things change; doesnt majorly affect the outcome of the model
  set social-radius 8 ;;; if this is smaller turtles tend to die out quickly
  set exile-distance 8
  set life-expectancy 50 ;;; much less than 50 and turtles die out before reproducing; much more and everything just slows down. Might as well specify.
  set social-gain 10 ;;; NOTE: in a different model, social-gain would be a crucial parameter to vary! 
  ;;; However, in this model it doesn't make things that interesting, just move faster.
  ;;; I think this is because this model is set up such that it's a constant increment, that is, turtles only ever gain or lose
  ;;; a maximum of social-gain divided amongst neighbors, rather than from each neighbor. Also, social-currency is not a zero sum game 
  ;;; (since turtles are born with social-currency). I thought it made sense to hide it from the user for simplicity
   
  ask patches [ set pcolor grey ] ;;; turtles are initialized black and can end up white, so grey background is best
  
  crt number-of-turtles [
    ;;; initialize color and location. we are going to do some color math so we have to do this a little hamfistedly
    set r 0
    set g 0
    set b 0
    set color rgb r g b ;; everyone starts off green, randomly placed in the world, with equal social currency
    set size 2
    setxy random-xcor random-ycor
    
    ;; internal variables
    set can-breed? (social-currency > mate-threshold)
    set social-currency initial-social-currency
    set p-theft (theft-threshold - 5 + random 11) ;; everyone has their own theft threshold within 5 of the set threshold
    set p-protect (protect-threshold - 5 + random 11) ;; ditto protection threshold
    if p-theft >= p-protect [set p-theft (p-protect - 1)] ;; make sure p-theft < p-protect; this gives preference to protection rather than theft. call me old-fashioned.
    set age 0
    set my-life-expectancy (life-expectancy - 5 + random 11)
    set got-away? false
      
    
  ]
  
  reset-ticks
end

to go
  if (count turtles = 0) [stop]
  if stop-at-5000-turtles-or-1000-ticks? ;; do we want the model to eventually halt?
  [ if (count turtles > 5000 or ticks > 1000) [stop]] 
  ;;; these population/tick limits were decided upon basically by seeing where the model starts to slow down enormously and by which point clear trends have emerged.
  
  ;;; first code block resets internals, moves the turtles, and picks what to do
  ask turtles [
    
    turn-reset
    ;;; move
    ifelse any? my-neighbors
    [ face max-one-of my-neighbors [social-currency]] ;; face your most popular neighbor
    [ rt random 360 ] ;; or wiggle if you have no neighbors
    fd 1 ;; take a step towards them
    
    ;;; pick an action. Actions will actually happen during the resolution phase
    ;;; this does NOT have to happen after all the turtles have moved because they are just picking what
    ;;; they WILL do, they are not actually doing it yet
    set random-choice-num random 100
    ifelse (random-choice-num < p-theft) 
    [ set take-action "steal" 
      ifelse (random 100 < probability-of-getting-away) ;; if you steal, figure out whether you get away with it (independent of who's watching)
      [ set got-away? true]
      [ set got-away? false]]
    [ ifelse (random-choice-num < p-protect) ;; if we got here that means p-theft < choice-num < p-protect
      [ set take-action "do-nothing" ] ;; and we do nothing
      [ set take-action "protect" ]]
  ]  ;; otherwise we protect
  
  ;;; resolve performs the actions and resolves the consequences
  ;;; has to happen after all turtles have moved and chosen their actions
  ask turtles [resolve] ;; within resolve, functions to actually perform actions are called
  
  ask turtles [
    if (social-currency > 100) [ set social-currency 100 ]
    if (social-currency <= 0) [ die ] ;; turtles at zero die
    if (age > my-life-expectancy) [ die ]
    set age (age + 1)
  ]
  
  ask turtles [ if can-breed? [ reproduce-if-possible ]] ;;; after all actions are resolved, turtles reproduce if they can
  
  tick
end

to turn-reset
    ;;; reset some information at each tick
    set my-mate false ;; reset mate info
    set got-away? false 
    set color rgb r g b ;;; reset color
    set can-breed? (social-currency > mate-threshold) ;;; reset breeding status
    set my-neighbors other turtles in-radius social-radius ;; figure out who your neighbors are
    if (social-currency <= 0) [ die ] ;; turtles at zero die
    if (age > my-life-expectancy) [ die ]
end

to resolve
  ;;; this is only its own function out of convenience; could be in-line with go
  ifelse (take-action = "steal")
  [ steal ]
  [ ifelse (take-action = "protect") 
    [ protect ]
    [ do-nothing ]
  ]
  
end

to steal
  ;;; note that steal actually handles MOST of what happens to turtles, i.e., both thieves and protectors  
 
  ;;; in this implementation, a successful thief takes TOTAL social-gain divided amongst the victims. another possibility would be to have it take a social-gain from EACH victim
  ifelse (got-away? or not any? my-neighbors with [take-action = "protect"]) ;; if no one is watching out for thieves or you get away with it
  [ set my-count (count my-neighbors)
    if (my-count > 0) ;; provided you have neighbors
    [ set social-currency (social-currency + social-gain) ;; get social-gain from chumpy neighbors
      ask my-neighbors [set social-currency (social-currency - (social-gain / [my-count] of myself))] ];; each neighbor loses social-gain/count neighbors
                                                                                                       ;;; if you have no neighbors, don't sweat it, you do nothing.
  ] 
  
  ;;; otherwise, someone is watching out for thieves AND you got caught
  [ set my-count (count my-neighbors with [take-action = "protect"])
    set social-currency (social-currency - social-gain) ;; lose social-gain
    ask my-neighbors with [take-action = "protect"]
      [ set social-currency (social-currency + (social-gain / [my-count] of myself))]  ;;; protectors don't each get social-gain, since the more protectors you have, the less each protection matters
    
    ;;; and deal with the consequences of getting caught
    rt random 360
    fd exile-distance ;;; if you get caught stealing, turn and walk away
  ]
  
  ifelse (r < 230) ;;; thieves turn redder
    [ set r (r + 25) ]
    [ set r 255 ]
  
end

to protect
  set my-count (count my-neighbors with [take-action = "steal" and not got-away?])
  if (my-count = 0) ;; if none of your neighbors tried to steal and did not get away with it, then you are being a busybody
    [ set social-currency (social-currency - social-gain) ;; lose social-gain 
      ask my-neighbors [set social-currency (social-currency + (social-gain / [count my-neighbors] of myself))]] ;;; all your neighbors are smugly satisfied with themselves and each other
  
  ;;; otherwise, someone has tried to steal
  ;;; but we don't have to do anything because it is handled by the steal function
  
  ;;; protectors turn bluer
  ifelse (b < 230) ;;; turtles who are protectors turn bluer
  [ set b (b + 25) ]
  [ set b 255 ]
  
end

to do-nothing
  ;;; the only thing we need to do within this function is change the color a bit
  ifelse (g < 230) ;;; show the turtle's guilt by turning it redder
  [ set g (g + 25) ]
  [ set g 255 ]

end

to reproduce-if-possible
  ;; only called with turtles who can reproduce
  set my-mateworthy-neighbors my-neighbors with [can-breed? and (my-mate = false)] ;; my-mate is set to false at the beginning of each tick
  if any? my-mateworthy-neighbors
  [ set my-mate max-one-of my-mateworthy-neighbors [social-currency]
    ask my-mate [ set my-mate true ] ;;; I know it's a bit odd to have the my-mate variable be a combination of booleans and agents, but this should still work; the point is to have my-mate bound or not
    ;; we do it this way so that turtles don't mate twice per turn
    hatch 1 [
      set social-currency initial-social-currency
      set p-protect ((([p-protect] of myself + [p-protect] of [my-mate] of myself) / 2) - 5 + random 11) ;; average p-protect of parents, plus some randomness
      set p-theft ((([p-theft] of myself + [p-theft] of [my-mate] of myself) / 2) - 5 + random 11) ;; average p-theft of parents, plus some randomness
      if p-theft >= p-protect [set p-theft (p-protect - 1)] ;; make sure p-theft < p-protect; this gives preference to protection rather than theft. call me old-fashioned.
      set age 0
      set my-life-expectancy ((([my-life-expectancy] of myself + [my-life-expectancy] of [my-mate] of myself) / 2) - 5 + random 11)
      set got-away? false
      
      ;;; placement and color
      set r 0
      set g 0
      set b 0
      set color rgb r g b ;; no matter who your parents are, you start off innocent
      rt random 360
      fd (exile-distance / 2)
    ]
  ]
  
end

;;; reporters for charts
to-report num-turtles-that-can-breed
  ifelse (count turtles = 0)
  [report 0]
  [ report count turtles with [can-breed?]]
end

to-report proportion-of-thieving-turtles
  ifelse (count turtles = 0)
  [report 0]
  [ report count turtles with [r > 250] / count turtles]
end

to-report proportion-of-protector-turtles
  ifelse (count turtles = 0)
  [report 0]
  [ report count turtles with [b > 250] / count turtles]
end

to-report proportion-of-placid-turtles
  ifelse (count turtles = 0)
  [report 0]
  [ report count turtles with [g > 250] / count turtles]
end

to-report proportion-of-mixed-turtles
  ifelse (count turtles = 0)
  [report 0]
  [ report count turtles with [(r > 250) and (b > 250)] / count turtles]
end

to-report proportion-turtles-who-just-stole
  ifelse (count turtles = 0)
  [report 0]
  [ report turtles-who-just-stole / count turtles]
end

to-report proportion-turtles-who-just-protected
  ifelse (count turtles = 0)
  [report 0]
  [ report turtles-who-just-protected / count turtles]
end

to-report proportion-turtles-who-just-did-nothing
  ifelse (count turtles = 0)
  [report 0]
  [ report turtles-who-just-did-nothing / count turtles]
end

to-report turtles-who-just-stole
  report count turtles with [take-action = "steal"]
end

to-report turtles-who-just-protected
  report count turtles with [take-action = "protect"]
end

to-report turtles-who-just-did-nothing
  report count turtles with [take-action = "do-nothing"]
end

to-report proportion-breeding-turtles
  ifelse (count turtles with [can-breed?] = 0)
  [report 0]
  [report ((count turtles with [can-breed?]) / count turtles)]
end

to-report proportion-thieving-breeders
  ifelse (count turtles with [can-breed?] = 0)
    [report 0]
    [report ((count turtles with [can-breed? and (take-action = "steal")]) / (count turtles with [can-breed?]))]
end

to-report proportion-protecting-breeders
  ifelse (count turtles with [can-breed?] = 0)
    [report 0]
    [report ((count turtles with [can-breed? and (take-action = "protect")]) / (count turtles with [can-breed?]))]
end

to-report proportion-complacent-breeders
  ifelse (count turtles with [can-breed?] = 0)
    [report 0]
    [report ((count turtles with [can-breed? and (take-action = "do-nothing")]) / (count turtles with [can-breed?]))]
end
@#$#@#$#@
GRAPHICS-WINDOW
15
395
260
580
50
50
1.525
1
10
1
1
1
0
1
1
1
-50
50
-50
50
1
1
1
ticks
30.0

SLIDER
21
97
193
130
theft-threshold
theft-threshold
0
100
34
1
1
NIL
HORIZONTAL

SLIDER
21
139
193
172
protect-threshold
protect-threshold
0
100
66
1
1
NIL
HORIZONTAL

BUTTON
31
13
94
46
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
113
13
176
46
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
21
182
198
215
mate-threshold
mate-threshold
0
100
55
1
1
NIL
HORIZONTAL

SLIDER
21
225
193
258
initial-social-currency
initial-social-currency
0
100
20
1
1
NIL
HORIZONTAL

PLOT
220
71
550
289
Prop  Turtles by Type
Time
Proportion
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Thieves" 1.0 0 -2674135 true "" "plot proportion-of-thieving-turtles"
"Protectors" 1.0 0 -13345367 true "" "plot proportion-of-protector-turtles"
"Placids" 1.0 0 -13840069 true "" "plot proportion-of-placid-turtles"

BUTTON
67
55
144
88
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
299
10
444
55
NIL
turtles-who-just-protected
17
1
11

MONITOR
458
10
574
55
NIL
turtles-who-just-stole
17
1
11

MONITOR
588
10
738
55
NIL
turtles-who-just-did-nothing
17
1
11

MONITOR
221
11
288
56
population
count turtles
17
1
11

PLOT
253
299
697
551
Breeder Types
Time
Breeder Types
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Thieves" 1.0 0 -2674135 true "" "plot proportion-thieving-breeders"
"Protectors" 1.0 0 -14070903 true "" "plot proportion-protecting-breeders"
"Placids" 1.0 0 -13840069 true "" "plot proportion-complacent-breeders"

SWITCH
20
312
243
345
stop-at-5000-turtles-or-1000-ticks?
stop-at-5000-turtles-or-1000-ticks?
0
1
-1000

SLIDER
21
268
196
301
probability-of-getting-away
probability-of-getting-away
0
100
100
5
1
NIL
HORIZONTAL

PLOT
567
71
922
289
Proportion of Actions  by Turn
Time
Proportion
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Thefts" 1.0 0 -2674135 true "" "plot proportion-turtles-who-just-stole"
"Placids" 1.0 0 -13840069 true "" "plot proportion-turtles-who-just-did-nothing"
"Protections" 1.0 0 -13345367 true "" "plot proportion-turtles-who-just-protected"

SWITCH
22
350
128
383
verbose?
verbose?
1
1
-1000

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Vary-p-Getaway-&amp;-initCurrency-&amp;-mateThreshold" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt;= 300</exitCondition>
    <metric>proportion-turtles-who-just-stole</metric>
    <metric>proportion-turtles-who-just-did-nothing</metric>
    <metric>proportion-turtles-who-just-protected</metric>
    <metric>proportion-of-thieving-turtles</metric>
    <metric>proportion-of-protector-turtles</metric>
    <metric>proportion-of-placid-turtles</metric>
    <metric>proportion-of-mixed-turtles</metric>
    <enumeratedValueSet variable="theft-threshold">
      <value value="20"/>
      <value value="33"/>
      <value value="50"/>
      <value value="66"/>
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stop-at-5000-turtles-or-1000-ticks?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="initial-social-currency" first="20" step="15" last="65"/>
    <steppedValueSet variable="mate-threshold" first="40" step="15" last="85"/>
    <enumeratedValueSet variable="protect-threshold">
      <value value="20"/>
      <value value="33"/>
      <value value="50"/>
      <value value="66"/>
      <value value="80"/>
    </enumeratedValueSet>
    <steppedValueSet variable="probability-of-getting-away" first="0" step="20" last="100"/>
  </experiment>
  <experiment name="vary-initThresholds-&amp;-p-getaway" repetitions="4" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>proportion-turtles-who-just-stole</metric>
    <metric>proportion-turtles-who-just-did-nothing</metric>
    <metric>proportion-turtles-who-just-protected</metric>
    <metric>proportion-of-thieving-turtles</metric>
    <metric>proportion-of-protector-turtles</metric>
    <metric>proportion-of-placid-turtles</metric>
    <metric>proportion-of-mixed-turtles</metric>
    <enumeratedValueSet variable="theft-threshold">
      <value value="20"/>
      <value value="10"/>
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stop-at-5000-turtles-or-1000-ticks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-social-currency">
      <value value="21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mate-threshold">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="protect-threshold">
      <value value="20"/>
      <value value="10"/>
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-of-getting-away">
      <value value="0"/>
      <value value="10"/>
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="vary-initThresholds-&amp;-p-getaway" repetitions="3" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="500"/>
    <exitCondition>count turtles &gt;= 2500</exitCondition>
    <metric>proportion-turtles-who-just-stole</metric>
    <metric>proportion-turtles-who-just-did-nothing</metric>
    <metric>proportion-turtles-who-just-protected</metric>
    <metric>proportion-of-thieving-turtles</metric>
    <metric>proportion-of-protector-turtles</metric>
    <metric>proportion-of-placid-turtles</metric>
    <metric>proportion-of-mixed-turtles</metric>
    <steppedValueSet variable="theft-threshold" first="20" step="10" last="80"/>
    <enumeratedValueSet variable="stop-at-5000-turtles-or-1000-ticks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-social-currency">
      <value value="21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mate-threshold">
      <value value="80"/>
    </enumeratedValueSet>
    <steppedValueSet variable="protect-threshold" first="20" step="10" last="80"/>
    <steppedValueSet variable="probability-of-getting-away" first="0" step="10" last="100"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
